<?php
// token_store.php
ob_start();
if (session_status() !== PHP_SESSION_ACTIVE) {
    session_start();
}

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Clear OPCache
if (function_exists('opcache_reset')) {
    opcache_reset();
}

require_once 'includes/functions.php';
require_once 'includes/db_config.php';

// Debug session status
if (session_status() !== PHP_SESSION_ACTIVE) {
    error_log("Session not active in token_store.php");
}

// Session timeout
$timeout = 1800; // 30 minutes
if (isset($_SESSION['last_activity']) && (time() - $_SESSION['last_activity']) > $timeout) {
    session_unset();
    session_destroy();
    redirect("login.php?timeout=1");
}
$_SESSION['last_activity'] = time();

// Restrict to logged-in users
if (!is_logged_in()) {
    redirect("login.php");
}

// Generate CSRF token
if (!isset($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

$message = '';
$message_type = '';
$currency_symbol = get_setting('currency_symbol', $link) ?? '₦';

// Check database connection
if (mysqli_connect_error()) {
    $message = "Database connection failed: " . mysqli_connect_error();
    $message_type = 'danger';
    error_log("Database connection failed in token_store.php: " . mysqli_connect_error());
    $account_name = 'Not set';
    $bank_name = 'Not set';
    $account_number = 'Not set';
    $token_packages = [];
} else {
    // Check if settings table exists
    $check_table = mysqli_query($link, "SHOW TABLES LIKE 'settings'");
    if (mysqli_num_rows($check_table) == 0) {
        $message = "Error: Settings table does not exist. Please contact the system administrator.";
        $message_type = 'danger';
        error_log("Settings table missing in token_store.php");
        $account_name = 'Not set';
        $bank_name = 'Not set';
        $account_number = 'Not set';
    } else {
        // Fetch super admin bank details
        $account_name = get_setting('super_admin_account_name', $link) ?? 'Not set';
        $bank_name = get_setting('super_admin_bank_name', $link) ?? 'Not set';
        $account_number = get_setting('super_admin_account_number', $link) ?? 'Not set';
    }

    // Check if token_packages table exists
    $check_packages = mysqli_query($link, "SHOW TABLES LIKE 'token_packages'");
    if (mysqli_num_rows($check_packages) == 0) {
        $message = "Error: Token packages table does not exist. Please contact the system administrator.";
        $message_type = 'danger';
        error_log("Token packages table missing in token_store.php");
        $token_packages = [];
    } else {
        // Fetch available token packages
        $token_packages = [];
        $sql = "SELECT id, package_name, quantity, price FROM token_packages WHERE status = 'active' ORDER BY price ASC";
        if ($result = mysqli_query($link, $sql)) {
            while ($row = mysqli_fetch_assoc($result)) {
                $token_packages[] = $row;
            }
            mysqli_free_result($result);
        } else {
            $message = "Error fetching token packages: " . mysqli_error($link);
            $message_type = 'danger';
            error_log("Error fetching token packages in token_store.php: " . mysqli_error($link));
        }
    }

    // Check if token_purchases table exists
    $check_purchases = mysqli_query($link, "SHOW TABLES LIKE 'token_purchases'");
    if (mysqli_num_rows($check_purchases) == 0) {
        $message = "Error: Token purchases table does not exist. Please contact the system administrator.";
        $message_type = 'danger';
        error_log("Token purchases table missing in token_store.php");
    }
}

// Handle purchase submission
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['purchase_package']) && isset($_POST['csrf_token']) && hash_equals($_SESSION['csrf_token'], $_POST['csrf_token']) && ($_SESSION['user_role'] === 'admin' || $_SESSION['user_role'] === 'super_admin')) {
    $package_id = intval($_POST['package_id'] ?? 0);

    if (mysqli_num_rows($check_purchases) == 0) {
        $message = "Error: Token purchases table does not exist.";
        $message_type = 'danger';
    } else {
        $sql = "SELECT package_name, quantity, price FROM token_packages WHERE id = ? AND status = 'active'";
        if ($stmt = mysqli_prepare($link, $sql)) {
            mysqli_stmt_bind_param($stmt, "i", $package_id);
            mysqli_stmt_execute($stmt);
            $result = mysqli_stmt_get_result($stmt);
            if ($row = mysqli_fetch_assoc($result)) {
                $package_name = $row['package_name'];
                $quantity = $row['quantity'];
                $price = $row['price'];

                $sql = "INSERT INTO token_purchases (user_id, package_id, package_name, quantity, amount, status, created_at) VALUES (?, ?, ?, ?, ?, 'pending', NOW())";
                if ($stmt2 = mysqli_prepare($link, $sql)) {
                    mysqli_stmt_bind_param($stmt2, "iisid", $_SESSION['user_id'], $package_id, $package_name, $quantity, $price);
                    $success = mysqli_stmt_execute($stmt2);
                    mysqli_stmt_close($stmt2);
                    if ($success) {
                        $message = "Purchase request submitted. Please make payment to the provided bank details and await approval.";
                        $message_type = 'success';
                        $super_admin_email = get_setting('admin_email', $link) ?? 'admin@clubportal.com';
                        $subject = "New Token Purchase Request";
                        $body = "A new token purchase request has been submitted by User ID: {$_SESSION['user_id']} for Package: $package_name.";
                        if (!send_mail($super_admin_email, $subject, $body)) {
                            error_log("Failed to send email notification in token_store.php");
                        }
                    } else {
                        $message = "Error submitting purchase: " . mysqli_error($link);
                        $message_type = 'danger';
                        error_log("Error submitting purchase in token_store.php: " . mysqli_error($link));
                    }
                } else {
                    $message = "Error preparing purchase statement: " . mysqli_error($link);
                    $message_type = 'danger';
                    error_log("Error preparing purchase statement in token_store.php: " . mysqli_error($link));
                }
            } else {
                $message = "Invalid or inactive package selected.";
                $message_type = 'danger';
                error_log("Invalid or inactive package selected in token_store.php: Package ID $package_id");
            }
            mysqli_free_result($result);
            mysqli_stmt_close($stmt);
        } else {
            $message = "Error verifying package: " . mysqli_error($link);
            $message_type = 'danger';
            error_log("Error verifying package in token_store.php: " . mysqli_error($link));
        }
    }
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
} elseif ($_SERVER["REQUEST_METHOD"] == "POST") {
    $message = "Invalid CSRF token or unauthorized action.";
    $message_type = 'danger';
    error_log("CSRF token validation failed or unauthorized in token_store.php");
}

ob_end_clean();
?>

<?php require_once 'includes/header.php'; ?>

<script>
document.title = 'Token Store - <?php echo htmlspecialchars(get_setting('system_name', $link) ?? 'Club Portal'); ?>';
</script>

<style>
.main-content {
    padding: 20px;
}
.bank-details {
    background-color: #f8f9fa;
    padding: 15px;
    border-radius: 5px;
}
</style>

<div class="container-fluid">
    <div class="row">
        <?php require_once 'includes/sidebar.php'; ?>
        <main class="col-md-9 col-lg-10 main-content">
            <h2>Token Store</h2>

            <?php if ($message): ?>
                <div class="alert alert-<?php echo htmlspecialchars($message_type); ?>" role="alert">
                    <?php echo htmlspecialchars($message); ?>
                </div>
            <?php endif; ?>

            <div class="card mb-4">
                <div class="card-header">Payment Instructions</div>
                <div class="card-body bank-details">
                    <p>Please make payment to the following bank account:</p>
                    <p><strong>Account Name:</strong> <?php echo htmlspecialchars($account_name); ?></p>
                    <p><strong>Bank Name:</strong> <?php echo htmlspecialchars($bank_name); ?></p>
                    <p><strong>Account Number:</strong> <?php echo htmlspecialchars($account_number); ?></p>
                    <p class="text-muted">After making the payment, your purchase request will be reviewed by the super admin.</p>
                </div>
            </div>

            <div class="card">
                <div class="card-header">Available Token Packages</div>
                <div class="card-body">
                    <?php if (empty($token_packages)): ?>
                        <p class="text-muted">No token packages available.</p>
                    <?php else: ?>
                        <div class="row">
                            <?php foreach ($token_packages as $package): ?>
                                <div class="col-md-4 mb-3">
                                    <div class="card">
                                        <div class="card-body">
                                            <h5 class="card-title"><?php echo htmlspecialchars($package['package_name']); ?></h5>
                                            <p class="card-text">Tokens: <?php echo htmlspecialchars($package['quantity']); ?></p>
                                            <p class="card-text">Price: <?php echo htmlspecialchars($currency_symbol . number_format($package['price'], 2)); ?></p>
                                            <?php if ($_SESSION['user_role'] === 'admin' || $_SESSION['user_role'] === 'super_admin'): ?>
                                                <form action="token_store.php" method="post">
                                                    <input type="hidden" name="purchase_package" value="1">
                                                    <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($_SESSION['csrf_token']); ?>">
                                                    <input type="hidden" name="package_id" value="<?php echo htmlspecialchars($package['id']); ?>">
                                                    <button type="submit" class="btn btn-primary">Purchase</button>
                                                </form>
                                            <?php else: ?>
                                                <p class="text-muted">Purchase available for admins only.</p>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </main>
    </div>
</div>

<?php require_once 'includes/footer.php'; ?>