<?php
// record_payments.php
ob_start();
session_start();

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Enable mysqli exceptions for better error handling
mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);

// Clear OPCache for fresh data
if (function_exists('opcache_reset')) {
    opcache_reset();
}

require_once 'includes/functions.php';
require_once 'includes/db_config.php';

// Session timeout
$timeout = 1800; // 30 minutes
if (isset($_SESSION['last_activity']) && (time() - $_SESSION['last_activity']) > $timeout) {
    session_unset();
    session_destroy();
    redirect("login.php?timeout=1");
}
$_SESSION['last_activity'] = time();

// Generate CSRF token
if (!isset($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

// Check if user is logged in and has view_payments permission
if (!is_logged_in() || !user_has_permission($link, 'view_payments')) {
    redirect("login.php");
    exit;
}

$message = '';
$message_type = '';
$currency_symbol = get_setting('currency_symbol', $link) ?? '₦';
$token_cost = (int)get_setting('tokens_per_payment', $link) ?: 1;

// NEW: Fetch system available tokens (matches dashboard)
$system_available_tokens = 0;
try {
    $sql_available_tokens = "SELECT COALESCE(SUM(balance), 0) AS total FROM token_balances";
    $result = mysqli_query($link, $sql_available_tokens);
    if ($result) {
        $system_available_tokens = mysqli_fetch_assoc($result)['total'] ?? 0;
        mysqli_free_result($result);
    } else {
        throw new Exception("Query failed");
    }
} catch (Exception $e) {
    error_log("Failed to fetch system available tokens: " . $e->getMessage());
}

// Handle form submission
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['log_payment']) && isset($_POST['csrf_token']) && hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {
    $user_ids = $_POST['user_ids'] ?? [];
    $amount = floatval($_POST['amount'] ?? 0);
    $obligation_id = intval($_POST['obligation_id'] ?? 0);
    $payment_type = sanitize_input($_POST['payment_type'] ?? '');
    $description = sanitize_input($_POST['description'] ?? '');
    $status = 'Paid';
    $current_user_id = $_SESSION['user_id'];
    $deduct_per_member = (int)get_setting('deduct_per_member', $link) ?? 0;

    error_log("Processing payment: user_ids=" . json_encode($user_ids) . ", amount=$amount, obligation_id=$obligation_id, payment_type=$payment_type, token_cost=$token_cost");

    // Dynamic token calculation
    $num_logs = $deduct_per_member ? count($user_ids) : 1;
    $required_tokens = $token_cost * $num_logs;
    $current_balance = get_token_balance($link, $current_user_id);
    
    if ($current_balance < $required_tokens) {
        $message = "Insufficient tokens. You need $required_tokens tokens, but you have $current_balance. <a href='token_store.php'>Please purchase more tokens</a>.";
        $message_type = 'danger';
    } else {
        // Validate obligation_id and fetch payment category
        $valid_obligation = false;
        $debt_amount = 0;
        $is_freewill = false;
        
        if ($obligation_id > 0) {
            $sql_check_obligation = "SELECT po.id, (COALESCE(po.required_amount, 0) - COALESCE(SUM(p.amount), 0)) AS debt, pt.payment_category
                                     FROM payment_obligations po
                                     LEFT JOIN payments p ON p.obligation_id = po.id AND p.payment_status = 'Paid'
                                     JOIN payment_types pt ON po.payment_type = pt.type_name AND pt.deleted_at IS NULL
                                     WHERE po.id = ? AND po.user_id IN (" . implode(',', array_fill(0, count($user_ids), '?')) . ")
                                     AND po.deleted_at IS NULL
                                     GROUP BY po.id, pt.payment_category
                                     HAVING debt > 0";
            try {
                $stmt_check = mysqli_prepare($link, $sql_check_obligation);
                $bind_params = array_merge([$obligation_id], array_map('intval', $user_ids));
                $bind_types = 'i' . str_repeat('i', count($user_ids));
                mysqli_stmt_bind_param($stmt_check, $bind_types, ...$bind_params);
                mysqli_stmt_execute($stmt_check);
                $result = mysqli_stmt_get_result($stmt_check);
                $row = mysqli_fetch_assoc($result);
                $valid_obligation = $row && $row['debt'] > 0;
                $debt_amount = $row ? $row['debt'] : 0;
                $is_freewill = $row && $row['payment_category'] === 'freewill';
                mysqli_stmt_close($stmt_check);
                mysqli_free_result($result);
            } catch (Exception $e) {
                error_log("Obligation check failed: " . $e->getMessage());
                $message = "Validation error.";
                $message_type = 'danger';
            }
        }

        // Final validation
        if (!empty($user_ids) && $amount > 0 && $valid_obligation && ($is_freewill || $amount <= $debt_amount)) {
            
            // *** BEGIN TRANSACTION BLOCK ***
            mysqli_begin_transaction($link);
            $transaction_success = true;

            // 1. Deduct tokens (dynamic amount)
            $tokens_used = deduct_tokens($link, $current_user_id, 'payment', $required_tokens);
            
            if ($tokens_used !== false) {
                // 2. Insert payment records
                $sql = "INSERT INTO payments (user_id, payment_type, amount, description, payment_status, obligation_id, payment_date, created_at)
                        VALUES (?, ?, ?, ?, ?, ?, NOW(), NOW())";
                try {
                    $stmt = mysqli_prepare($link, $sql);
                    foreach ($user_ids as $user_id) {
                        $user_id = intval($user_id);
                        mysqli_stmt_bind_param($stmt, "isdssi", $user_id, $payment_type, $amount, $description, $status, $obligation_id);
                        if (!mysqli_stmt_execute($stmt)) {
                            $transaction_success = false;
                            throw new Exception("Payment insert failed for user $user_id: " . mysqli_stmt_error($stmt));
                        }
                    }
                    mysqli_stmt_close($stmt);
                } catch (Exception $e) {
                    $transaction_success = false;
                    $message = "Error logging payment: " . $e->getMessage();
                    $message_type = 'danger';
                    error_log("Payment insert failed: " . $e->getMessage());
                }
            } else {
                // Token deduction failed
                $transaction_success = false;
                $message = "Failed to deduct tokens. <a href='token_store.php'>Please purchase more</a>.";
                $message_type = 'danger';
            }

            // *** END TRANSACTION BLOCK ***
            if ($transaction_success) {
                mysqli_commit($link);
                $message = "Payment logged successfully! Used $tokens_used tokens.";
                $message_type = 'success';
            } else {
                mysqli_rollback($link);
                if (empty($message)) {
                    $message = "Transaction failed. No payment was logged and no tokens were deducted.";
                    $message_type = 'danger';
                }
            }

        } else {
            $message = "Invalid input. Please check amount, select at least one member, verify obligation, or ensure payment does not exceed debt for non-freewill payments.";
            $message_type = 'danger';
        }
    }
    // Regenerate CSRF token
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
} elseif ($_SERVER["REQUEST_METHOD"] == "POST") {
    $message = "Invalid CSRF token.";
    $message_type = 'danger';
    error_log("CSRF token validation failed in record_payments.php");
}

// ... rest of the file (fetching members, HTML, scripts) remains the same ...

// Fetch members for the form
$members = [];
$sql_members = "SELECT id, full_name FROM users WHERE role = 'member' AND (deleted_at IS NULL OR deleted_at > NOW()) ORDER BY full_name ASC";
$result_members = mysqli_query($link, $sql_members);
if ($result_members) {
    while ($row = mysqli_fetch_assoc($result_members)) {
        $members[] = $row;
    }
    mysqli_free_result($result_members);
}
?>

<?php require_once 'includes/header.php'; ?>

<script>
document.title = 'Record Payments - <?php echo htmlspecialchars(get_setting('system_name', $link) ?? 'Club Portal'); ?>';
</script>

<style>
.main-content {
    padding: 20px;
}
</style>

<div class="container-fluid">
    <div class="row">
        <?php require_once 'includes/sidebar.php'; ?>
        <main class="col-md-9 col-lg-10 main-content">
            <h2>Record Payments</h2>

            <?php if ($message): ?>
                <div class="alert alert-<?php echo htmlspecialchars($message_type); ?>" role="alert">
                    <?php echo $message; ?>
                </div>
            <?php endif; ?>

            <div class="card mb-4">
                <div class="card-header">Log New Payment</div>
                <div class="card-body">
                    <p>System Available Tokens: <?php echo htmlspecialchars($system_available_tokens); ?> tokens</p>
                    <form action="record_payments.php" method="post">
                        <input type="hidden" name="log_payment" value="1">
                        <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($_SESSION['csrf_token']); ?>">
                        <div class="mb-3">
                            <label for="user_ids" class="form-label">Select Member(s)</label>
                            <select name="user_ids[]" id="user_ids" class="form-select" multiple size="10" required>
                                <?php foreach ($members as $member): ?>
                                    <option value="<?php echo htmlspecialchars($member['id']); ?>"><?php echo htmlspecialchars($member['full_name']); ?></option>
                                <?php endforeach; ?>
                            </select>
                            <small class="form-text text-muted">Hold Ctrl (Windows) or Cmd (Mac) to select multiple members. ($token_cost token per log)</small>
                        </div>
                        <div class="mb-3">
                            <label for="obligation_id" class="form-label">Select Obligation</label>
                            <select name="obligation_id" id="obligation_id" class="form-select" required onchange="updatePaymentType()">
                                <option value="">Select an Obligation</option>
                                <?php
                                $sql_obligations = "SELECT po.id, po.payment_type, (COALESCE(po.required_amount, 0) - COALESCE(SUM(p.amount), 0)) AS debt, pt.payment_category
                                                    FROM payment_obligations po
                                                    LEFT JOIN payments p ON p.obligation_id = po.id AND p.payment_status = 'Paid'
                                                    JOIN payment_types pt ON po.payment_type = pt.type_name AND pt.deleted_at IS NULL
                                                    WHERE po.deleted_at IS NULL
                                                    GROUP BY po.id, po.payment_type, pt.payment_category
                                                    HAVING debt > 0";
                                $result_obligations = mysqli_query($link, $sql_obligations);
                                if ($result_obligations) {
                                    while ($obligation = mysqli_fetch_assoc($result_obligations)) {
                                        $suffix = $obligation['payment_category'] === 'freewill' ? ' (Freewill)' : '';
                                        echo "<option value='{$obligation['id']}' data-payment-type='{$obligation['payment_type']}'>{$obligation['payment_type']}{$suffix} ({$currency_symbol}" . number_format($obligation['debt'], 2) . ")</option>";
                                    }
                                    mysqli_free_result($result_obligations);
                                }
                                ?>
                            </select>
                        </div>
                        <div class="mb-3">
                            <label for="payment_type" class="form-label">Payment Type</label>
                            <input type="text" name="payment_type" id="payment_type" class="form-control" readonly>
                        </div>
                        <div class="mb-3">
                            <label for="amount_payment" class="form-label">Amount (<?php echo htmlspecialchars($currency_symbol); ?>)</label>
                            <input type="number" name="amount" id="amount_payment" class="form-control" step="0.01" min="0.01" required>
                            <small class="form-text text-muted">For freewill donations, any positive amount is allowed.</small>
                        </div>
                        <div class="mb-3">
                            <label for="description_payment" class="form-label">Description (Optional)</label>
                            <textarea name="description" id="description_payment" class="form-control" rows="3"></textarea>
                        </div>
                        <button type="submit" class="btn btn-primary">Log Payment</button>
                    </form>
                </div>
            </div>
        </main>
    </div>
</div>

<script>
function updatePaymentType() {
    const obligationSelect = document.getElementById('obligation_id');
    const paymentTypeInput = document.getElementById('payment_type');
    const selectedOption = obligationSelect.options[obligationSelect.selectedIndex];
    if (selectedOption && selectedOption.dataset && selectedOption.dataset.paymentType) {
        paymentTypeInput.value = selectedOption.dataset.paymentType;
    } else {
        paymentTypeInput.value = '';
    }
}

function updateObligations() {
    const userSelect = document.getElementById('user_ids');
    const obligationSelect = document.getElementById('obligation_id');
    const selectedUsers = Array.from(userSelect.selectedOptions).map(option => option.value);

    if (selectedUsers.length === 0) {
        obligationSelect.innerHTML = '<option value="">Select an Obligation</option>';
        document.getElementById('payment_type').value = '';
        return;
    }

    fetch('api/get_obligations.php', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ user_ids: selectedUsers })
    })
    .then(response => response.json())
    .then(data => {
        obligationSelect.innerHTML = '<option value="">Select an Obligation</option>';
        data.obligations.forEach(obligation => {
            const option = document.createElement('option');
            option.value = obligation.id;
            const suffix = obligation.payment_category === 'freewill' ? ' (Freewill)' : '';
            option.text = `${obligation.payment_type}${suffix} (${obligation.currency_symbol}${parseFloat(obligation.debt).toFixed(2)})`;
            option.dataset.paymentType = obligation.payment_type;
            obligationSelect.appendChild(option);
        });
        updatePaymentType();
    })
    .catch(error => {
        console.error('Error fetching obligations:', error);
        obligationSelect.innerHTML = '<option value="">Error loading obligations</option>';
    });
}

document.addEventListener('DOMContentLoaded', () => {
    document.getElementById('user_ids').addEventListener('change', updateObligations);
    updateObligations();
});
</script>

<?php require_once 'includes/footer.php'; ?>