<?php
// Handle CSV export at the very top, before any HTML is sent.
if (isset($_GET['export']) && $_GET['export'] == 'csv') {
    require_once 'includes/functions.php';
    require_once 'includes/db_config.php';

    if (!is_logged_in() || !is_admin()) {
        redirect("login.php");
        exit();
    }

    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename=members.csv');

    $output = fopen('php://output', 'w');
    fputcsv($output, array('Member ID', 'Full Name', 'Email', 'Membership Status'));

    $sql = "SELECT
                u.member_id,
                u.full_name,
                u.email,
                u.membership_status,
                fd.data AS form_data,
                f.form_data AS form_structure
            FROM users u
            LEFT JOIN form_data fd ON u.id = fd.user_id
            LEFT JOIN forms f ON fd.form_id = f.id
            WHERE u.role = 'member' OR u.role = 'pending'";
            
    $result = mysqli_query($link, $sql);

    if ($result) {
        while ($row = mysqli_fetch_assoc($result)) {
            $surname = '';
            $other_names = '';
            
            $form_data_array = $row['form_data'] ? json_decode($row['form_data'], true) : null;
            $form_structure_array = $row['form_structure'] ? json_decode($row['form_structure'], true) : null; // Fixed: Use form_structure

            if ($form_data_array && $form_structure_array && isset($form_structure_array['fields'])) {
                foreach ($form_structure_array['fields'] as $field) {
                    if (isset($field['label']) && strpos(strtolower($field['label']), 'surname') !== false) {
                        if (isset($field['name']) && isset($form_data_array[$field['name']])) {
                            $surname = $form_data_array[$field['name']];
                        }
                    } elseif (isset($field['label']) && strpos(strtolower($field['label']), 'other names') !== false) {
                        if (isset($field['name']) && isset($form_data_array[$field['name']])) {
                            $other_names = $form_data_array[$field['name']];
                        }
                    }
                }
            }
            
            $member_full_name = trim($surname . ' ' . $other_names);
            if (empty($member_full_name)) {
                $member_full_name = $row['full_name'] ?? '';
            }
            
            fputcsv($output, [$row['member_id'], $member_full_name, $row['email'], $row['membership_status']]);
        }
    }
    fclose($output);
    exit();
}

// All other includes and code
require_once 'includes/functions.php';
require_once 'includes/db_config.php';

if (!is_logged_in() || !is_admin()) {
    redirect("login.php");
    exit();
}

$system_name = get_setting('system_name', $link) ?? 'Club Portal'; // For dynamic title
$message = '';
$debug_info = [];

// Handle add member form submission
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['form_submission'])) {
    $full_name = '';
    $email = '';
    $password = '';
    $custom_form_data = [];
    $is_valid_submission = true;

    // Fetch form structure
    $sql_form = "SELECT id, form_data FROM forms ORDER BY id DESC LIMIT 1";
    $result_form = mysqli_query($link, $sql_form);
    $registration_form_id = null;
    $form_data = null;
    if ($result_form && mysqli_num_rows($result_form) > 0) {
        $row = mysqli_fetch_assoc($result_form);
        $registration_form_id = $row['id'];
        $form_data = ($row['form_data'] !== null) ? json_decode($row['form_data'], true) : null;
    }

    // Process form data
    if ($form_data && isset($form_data['fields'])) {
        foreach ($form_data['fields'] as $field) {
            $fieldName = $field['name'] ?? '';
            if (isset($_POST[$fieldName])) {
                $value = $_POST[$fieldName];
                $sanitizedValue = is_array($value) ? $value : sanitize_input($value);
                $custom_form_data[$fieldName] = $sanitizedValue;
                
                switch ($field['type']) {
                    case 'text':
                    case 'username':
                        if (empty($full_name) && strpos(strtolower($field['label']), 'name') !== false) {
                            $full_name = is_array($value) ? reset($value) : $sanitizedValue;
                        }
                        if (empty($full_name) && strpos(strtolower($field['label']), 'surname') !== false) {
                            $full_name = is_array($value) ? reset($value) : $sanitizedValue;
                        }
                        if (empty($full_name) && strpos(strtolower($field['label']), 'other names') !== false) {
                            $full_name .= ' ' . (is_array($value) ? reset($value) : $sanitizedValue);
                        }
                        break;
                    case 'email':
                        $email = is_array($value) ? reset($value) : $sanitizedValue;
                        break;
                    case 'password':
                        $password = is_array($value) ? reset($value) : $value;
                        break;
                }
            }
        }
    }
    
    // Validate
    if (empty($full_name) || empty($email) || empty($password)) {
        $message = "Full Name, Email, and Password are required.";
        $is_valid_submission = false;
    } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $message = "Invalid email format.";
        $is_valid_submission = false;
    } elseif (strlen($password) < 6) {
        $message = "Password must be at least 6 characters long.";
        $is_valid_submission = false;
    }

    if ($is_valid_submission) {
        $hashed_password = password_hash($password, PASSWORD_DEFAULT);
        $check_sql = "SELECT id FROM users WHERE email = ?";
        $check_stmt = mysqli_prepare($link, $check_sql);
        mysqli_stmt_bind_param($check_stmt, "s", $email);
        mysqli_stmt_execute($check_stmt);
        mysqli_stmt_store_result($check_stmt);

        if (mysqli_stmt_num_rows($check_stmt) > 0) {
            $message = "An account with this email already exists.";
        } else {
            // Generate unique member ID
            $initials = '';
            $words = explode(' ', trim($full_name));
            foreach ($words as $word) {
                if (!empty($word)) {
                    $initials .= strtoupper($word[0]);
                }
            }
            if (empty($initials)) {
                $initials = 'M'; // Default initial if name is empty
            }
            
            do {
                $serialized_digits = str_pad(rand(0, 999), 3, '0', STR_PAD_LEFT);
                $unique_member_id = $initials . $serialized_digits;
                
                $check_unique_sql = "SELECT id FROM users WHERE member_id = ?";
                $check_unique_stmt = mysqli_prepare($link, $check_unique_sql);
                mysqli_stmt_bind_param($check_unique_stmt, "s", $unique_member_id);
                mysqli_stmt_execute($check_unique_stmt);
                mysqli_stmt_store_result($check_unique_stmt);
            } while (mysqli_stmt_num_rows($check_unique_stmt) > 0);
            mysqli_stmt_close($check_unique_stmt);

            // Insert new user with Pending status
            $insert_sql = "INSERT INTO users (member_id, full_name, email, password, role, created_at, membership_status) VALUES (?, ?, ?, ?, 'pending', NOW(), 'Pending')";
            if ($insert_stmt = mysqli_prepare($link, $insert_sql)) {
                mysqli_stmt_bind_param($insert_stmt, "ssss", $unique_member_id, $full_name, $email, $hashed_password);
                if (mysqli_stmt_execute($insert_stmt)) {
                    $user_id = mysqli_insert_id($link);
                    $message = "Member added successfully! Member ID: " . $unique_member_id;
                    error_log("New member added: ID=$user_id, Member ID=$unique_member_id, Status=Pending, Role=pending");
                    $debug_info['new_member_id'] = $user_id;

                    $custom_data = json_encode($custom_form_data);
                    $data_sql = "INSERT INTO form_data (form_id, user_id, data, created_at) VALUES (?, ?, ?, NOW())";
                    if ($data_stmt = mysqli_prepare($link, $data_sql)) {
                        mysqli_stmt_bind_param($data_stmt, "iis", $registration_form_id, $user_id, $custom_data);
                        mysqli_stmt_execute($data_stmt);
                        mysqli_stmt_close($data_stmt);
                    }

                    // --- CORRECTED CODE: Automatically create payment obligations for the new member ---
                    $payment_types_sql = "SELECT type_name FROM payment_types WHERE is_required = 1";
                    $payment_types_result = mysqli_query($link, $payment_types_sql);
                    if ($payment_types_result) {
                        $insert_obligation_sql = "INSERT INTO payment_obligations (user_id, payment_type, payment_cycle) VALUES (?, ?, YEAR(CURDATE()))";
                        $insert_obligation_stmt = mysqli_prepare($link, $insert_obligation_sql);
                        while ($type_row = mysqli_fetch_assoc($payment_types_result)) {
                            mysqli_stmt_bind_param($insert_obligation_stmt, "is", $user_id, $type_row['type_name']);
                            mysqli_stmt_execute($insert_obligation_stmt);
                        }
                        mysqli_stmt_close($insert_obligation_stmt);
                    }
                    // --- END OF CORRECTED CODE ---
                } else {
                    $message = "Error: " . mysqli_error($link);
                    error_log("Failed to add member: " . mysqli_error($link));
                }
                mysqli_stmt_close($insert_stmt);
            }
        }
        mysqli_stmt_close($check_stmt);
    }
}

// Handle actions (delete, change status)
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['action']) && isset($_POST['member_id'])) {
    $member_id = intval($_POST['member_id']);
    $action = $_POST['action'];

    // Retrieve the user's role to prevent changing admin status
    $user_sql = "SELECT role FROM users WHERE id = ?";
    $user_stmt = mysqli_prepare($link, $user_sql);
    mysqli_stmt_bind_param($user_stmt, "i", $member_id);
    mysqli_stmt_execute($user_stmt);
    $user_result = mysqli_stmt_get_result($user_stmt);
    $user_row = mysqli_fetch_assoc($user_result);
    $user_role = $user_row['role'] ?? '';
    mysqli_stmt_close($user_stmt);

    if ($user_role !== 'member' && $user_role !== 'pending') {
        $message = "You cannot change the status or delete an administrative user.";
        $message_type = 'danger';
    } else {
        if ($action === 'delete') {
            $sql_form_data = "DELETE FROM form_data WHERE user_id = ?";
            $stmt_form_data = mysqli_prepare($link, $sql_form_data);
            mysqli_stmt_bind_param($stmt_form_data, "i", $member_id);
            mysqli_stmt_execute($stmt_form_data);
            mysqli_stmt_close($stmt_form_data);

            // --- CORRECTED CODE: Delete associated payment obligations
            $sql_obligations = "DELETE FROM payment_obligations WHERE user_id = ?";
            $stmt_obligations = mysqli_prepare($link, $sql_obligations);
            mysqli_stmt_bind_param($stmt_obligations, "i", $member_id);
            mysqli_stmt_execute($stmt_obligations);
            mysqli_stmt_close($stmt_obligations);
            // --- END OF CORRECTED CODE ---
            
            $sql_user = "DELETE FROM users WHERE id = ?";
            if ($stmt_user = mysqli_prepare($link, $sql_user)) {
                mysqli_stmt_bind_param($stmt_user, "i", $member_id);
                if (mysqli_stmt_execute($stmt_user)) {
                    $message = "Member deleted successfully!";
                    error_log("Member ID=$member_id deleted");
                } else {
                    $message = "Error deleting member: " . mysqli_error($link);
                    error_log("Failed to delete member ID=$member_id: " . mysqli_error($link));
                }
                mysqli_stmt_close($stmt_user);
            }
            
        } elseif ($action === 'change_status') {
            $new_status = $_POST['membership_status'] ?? '';
            $valid_statuses = ['Approved', 'Rejected', 'Pending', 'Under Investigation', 'Banned'];
            
            if (in_array($new_status, $valid_statuses)) {
                $old_status_sql = "SELECT membership_status FROM users WHERE id = ?";
                $old_status_stmt = mysqli_prepare($link, $old_status_sql);
                mysqli_stmt_bind_param($old_status_stmt, "i", $member_id);
                mysqli_stmt_execute($old_status_stmt);
                $old_status_result = mysqli_stmt_get_result($old_status_stmt);
                $old_status = mysqli_fetch_assoc($old_status_result)['membership_status'];
                mysqli_stmt_close($old_status_stmt);

                $role = ($new_status === 'Approved') ? 'member' : 'pending';
                $sql = "UPDATE users SET membership_status = ?, role = ? WHERE id = ?";
                if ($stmt = mysqli_prepare($link, $sql)) {
                    mysqli_stmt_bind_param($stmt, "ssi", $new_status, $role, $member_id);
                    if (mysqli_stmt_execute($stmt)) {
                        $message = "Membership status updated to $new_status successfully!";
                        error_log("Member ID=$member_id updated to membership_status=$new_status, role=$role");

                        // --- CORRECTED CODE: Auto-assign obligations on 'Approved' status change
                        if ($new_status === 'Approved' && $old_status !== 'Approved') {
                            $payment_types_sql = "SELECT type_name FROM payment_types WHERE is_required = 1";
                            $payment_types_result = mysqli_query($link, $payment_types_sql);
                            if ($payment_types_result) {
                                $insert_obligation_sql = "INSERT INTO payment_obligations (user_id, payment_type, payment_cycle) VALUES (?, ?, YEAR(CURDATE()))";
                                $insert_obligation_stmt = mysqli_prepare($link, $insert_obligation_sql);
                                while ($type_row = mysqli_fetch_assoc($payment_types_result)) {
                                    mysqli_stmt_bind_param($insert_obligation_stmt, "is", $member_id, $type_row['type_name']);
                                    mysqli_stmt_execute($insert_obligation_stmt);
                                }
                                mysqli_stmt_close($insert_obligation_stmt);
                            }
                        }
                        // --- END OF CORRECTED CODE ---
                    } else {
                        $message = "Error updating membership status: " . mysqli_error($link);
                        error_log("Failed to update status for member ID=$member_id: " . mysqli_error($link));
                    }
                    mysqli_stmt_close($stmt);
                } else {
                    $message = "Error preparing update statement: " . mysqli_error($link);
                    error_log("Failed to prepare update statement for member ID=$member_id: " . mysqli_error($link));
                }
            } else {
                $message = "Invalid membership status selected: $new_status";
                error_log("Invalid status '$new_status' attempted for member ID=$member_id");
            }
        }
    }
}

// Fetch members data for display, only showing regular members
$sql = "SELECT
            u.id,
            u.member_id,
            u.full_name,
            u.email,
            u.membership_status,
            u.role,
            fd.data AS form_data,
            f.form_data AS form_structure
        FROM users u
        LEFT JOIN form_data fd ON u.id = fd.user_id
        LEFT JOIN forms f ON fd.form_id = f.id
        WHERE u.role = 'member' OR u.role = 'pending' OR (u.membership_status IS NULL OR u.membership_status = '')";
$result = mysqli_query($link, $sql);
$members = [];

if ($result) {
    while ($row = mysqli_fetch_assoc($result)) {
        $surname = '';
        $other_names = '';
        
        $form_data_array = $row['form_data'] ? json_decode($row['form_data'], true) : null;
        $form_structure_array = $row['form_structure'] ? json_decode($row['form_structure'], true) : null; // Fixed: Use form_structure
        
        if ($form_data_array && $form_structure_array && isset($form_structure_array['fields'])) {
            foreach ($form_structure_array['fields'] as $field) {
                if (isset($field['label']) && strpos(strtolower($field['label']), 'surname') !== false) {
                    if (isset($field['name']) && isset($form_data_array[$field['name']])) {
                        $surname = $form_data_array[$field['name']];
                    }
                } elseif (isset($field['label']) && strpos(strtolower($field['label']), 'other names') !== false) {
                    if (isset($field['name']) && isset($form_data_array[$field['name']])) {
                        $other_names = $form_data_array[$field['name']];
                    }
                }
            }
        }
        
        $member_full_name = trim($surname . ' ' . $other_names);
        if (empty($member_full_name)) {
            $member_full_name = $row['full_name'] ?? '';
        }
        
        $members[] = [
            'id' => $row['id'],
            'member_id' => $row['member_id'],
            'full_name' => $member_full_name,
            'email' => $row['email'],
            'membership_status' => $row['membership_status'] ?? 'Pending',
            'role' => $row['role'] ?? 'pending',
            'raw_form_data' => $row['form_data']
        ];
    }
    $debug_info['members_count'] = count($members);
} else {
    $message = "Error fetching members: " . mysqli_error($link);
    error_log("Failed to fetch members: " . mysqli_error($link));
}
?>

<?php require_once 'includes/header.php'; ?>

<script>
// Set dynamic page title
document.title = 'Member Management - <?php echo htmlspecialchars($system_name); ?>';
</script>

<div class="container-fluid">
    <div class="row">
        <?php require_once 'includes/sidebar.php'; ?>
        <main class="col-md-9 col-lg-10 main-content">
            <h2>Manage Members</h2>
            
            <?php if ($message): ?>
                <div class="alert alert-<?php echo strpos($message, 'Error') !== false ? 'danger' : 'success'; ?>">
                    <?php echo htmlspecialchars($message); ?>
                </div>
            <?php endif; ?>

            <div class="member-management-tools mb-4">
                <a href="?export=csv" class="btn btn-primary"><i class="fas fa-file-download me-2"></i>Export to CSV</a>
                <button type="button" class="btn btn-success" onclick="document.getElementById('addMemberModal').style.display='block';"><i class="fas fa-user-plus me-2"></i>Add Member</button>
            </div>

            <div class="card">
                <div class="card-header">Member List</div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table table-striped table-hover">
                            <thead>
                                <tr>
                                    <th>Member ID</th>
                                    <th>Full Name</th>
                                    <th>Email</th>
                                    <th>Status</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (empty($members)): ?>
                                    <tr><td colspan="5" class="text-center">No members found.</td></tr>
                                <?php else: ?>
                                    <?php foreach ($members as $member): ?>
                                        <tr>
                                            <td><?php echo htmlspecialchars($member['member_id'] ?? ''); ?></td>
                                            <td><?php echo htmlspecialchars($member['full_name']); ?></td>
                                            <td><?php echo htmlspecialchars($member['email'] ?? ''); ?></td>
                                            <td id="status-<?php echo $member['id']; ?>">
                                                <?php echo htmlspecialchars(ucfirst($member['membership_status'])); ?>
                                            </td>
                                            <td>
                                                <a href="view_member.php?id=<?php echo $member['id']; ?>" class="btn btn-info btn-sm" title="View Details"><i class="fas fa-eye"></i> View</a>
                                                
                                                <form method="post" action="member_management.php" style="display:inline;">
                                                    <input type="hidden" name="member_id" value="<?php echo $member['id']; ?>">
                                                    <input type="hidden" name="action" value="change_status">
                                                    <select name="membership_status" onchange="this.form.submit();" class="form-select form-select-sm d-inline-block" style="width: auto;">
                                                        <option value="Approved" <?php echo $member['membership_status'] === 'Approved' ? 'selected' : ''; ?>>Approved</option>
                                                        <option value="Rejected" <?php echo $member['membership_status'] === 'Rejected' ? 'selected' : ''; ?>>Rejected</option>
                                                        <option value="Pending" <?php echo $member['membership_status'] === 'Pending' ? 'selected' : ''; ?>>Pending</option>
                                                        <option value="Under Investigation" <?php echo $member['membership_status'] === 'Under Investigation' ? 'selected' : ''; ?>>Under Investigation</option>
                                                        <option value="Banned" <?php echo $member['membership_status'] === 'Banned' ? 'selected' : ''; ?>>Banned</option>
                                                    </select>
                                                </form>

                                                <form id="delete-form-<?php echo $member['id']; ?>" method="post" action="member_management.php" style="display:inline;" onsubmit="return confirm('Are you sure you want to delete this member? This action cannot be undone.');">
                                                    <input type="hidden" name="member_id" value="<?php echo $member['id']; ?>">
                                                    <input type="hidden" name="action" value="delete">
                                                    <button type="submit" class="btn btn-danger btn-sm"><i class="fas fa-trash"></i> Delete</button>
                                                </form>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>

            <div id="addMemberModal" class="modal">
                <div class="modal-content">
                    <span class="close" onclick="document.getElementById('addMemberModal').style.display='none';">&times;</span>
                    <h3>Add New Member</h3>
                    <?php 
                    // Fetch the latest form structure for the modal
                    $sql_form_modal = "SELECT id, form_data FROM forms ORDER BY id DESC LIMIT 1";
                    $result_form_modal = mysqli_query($link, $sql_form_modal);
                    $registration_form_id = null;
                    $form_data = null;
                    if ($result_form_modal && mysqli_num_rows($result_form_modal) > 0) {
                        $row = mysqli_fetch_assoc($result_form_modal);
                        $registration_form_id = $row['id'];
                        $form_data = ($row['form_data'] !== null) ? json_decode($row['form_data'], true) : null;
                    }
                    ?>
                    <?php if ($registration_form_id !== null && $form_data !== null): ?>
                        <form id="add-member-form" action="member_management.php" method="post">
                            <div id="dynamic-form-container">
                                <?php
                                    // Generate the form fields dynamically for the modal
                                    if (isset($form_data['fields'])) {
                                        foreach ($form_data['fields'] as $field) {
                                            $field_name = htmlspecialchars($field['name']);
                                            $field_label = htmlspecialchars($field['label']);
                                            $field_type = htmlspecialchars($field['type']);
                                            $required = isset($field['required']) && $field['required'] ? 'required' : '';
                                            echo '<div class="mb-3">';
                                            echo '<label for="' . $field_name . '" class="form-label">' . $field_label . '</label>';
                                            
                                            switch ($field_type) {
                                                case 'textarea':
                                                    echo '<textarea class="form-control" id="' . $field_name . '" name="' . $field_name . '" ' . $required . '></textarea>';
                                                    break;
                                                case 'select':
                                                    echo '<select class="form-control" id="' . $field_name . '" name="' . $field_name . '" ' . $required . '>';
                                                    if (isset($field['options']) && is_array($field['options'])) {
                                                        foreach ($field['options'] as $option) {
                                                            echo '<option value="' . htmlspecialchars($option['value']) . '">' . htmlspecialchars($option['label']) . '</option>';
                                                        }
                                                    }
                                                    echo '</select>';
                                                    break;
                                                case 'radio':
                                                case 'checkbox':
                                                    echo '<div class="form-check">';
                                                    if (isset($field['options']) && is_array($field['options'])) {
                                                        foreach ($field['options'] as $option) {
                                                            $option_value = htmlspecialchars($option['value']);
                                                            $option_label = htmlspecialchars($option['label']);
                                                            echo '<input class="form-check-input" type="' . $field_type . '" name="' . $field_name . '[]" value="' . $option_value . '" id="' . $field_name . '_' . $option_value . '" ' . $required . '>';
                                                            echo '<label class="form-check-label" for="' . $field_name . '_' . $option_value . '">' . $option_label . '</label><br>';
                                                        }
                                                    }
                                                    echo '</div>';
                                                    break;
                                                default:
                                                    echo '<input type="' . $field_type . '" class="form-control" id="' . $field_name . '" name="' . $field_name . '" ' . $required . '>';
                                                    break;
                                            }
                                            echo '</div>';
                                        }
                                    }
                                ?>
                            </div>
                            <input type="hidden" name="form_submission" value="1">
                            <div class="form-actions mt-3">
                                <button type="submit" class="btn btn-success">Add Member</button>
                            </div>
                        </form>
                    <?php else: ?>
                        <div class="alert alert-warning">
                            No registration form is currently available. Please set up a form in the Form Builder.
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </main>
    </div>
</div>

<script>
// Handle modal close on outside click
window.onclick = function(event) {
    const addModal = document.getElementById('addMemberModal');
    if (event.target == addModal) {
        addModal.style.display = 'none';
    }
}
</script>

<?php require_once 'includes/footer.php'; ?>