<?php
session_start();
require_once 'includes/functions.php';
require_once 'includes/db_config.php';

// Check if user is logged in and has permission to view the directory
if (!is_logged_in()) {
    redirect("login.php");
}

// --- PAGINATION AND SEARCH SETUP ---
$members_per_page = 10;
$current_page = isset($_GET['page']) && is_numeric($_GET['page']) ? intval($_GET['page']) : 1;
$offset = ($current_page - 1) * $members_per_page;

$search_query = "";
$search_param = "";
$search_term = "";
if (isset($_GET['search'])) {
    $search_query = sanitize_input($_GET['search']);
    $search_param = " AND full_name LIKE ?";
    $search_term = "%" . $search_query . "%";
}

// --- COUNT TOTAL REGISTERED MEMBERS FOR PAGINATION ---
$count_sql = "SELECT COUNT(*) AS total_members FROM users WHERE role IN ('member', 'pending')" . $search_param;
$count_stmt = mysqli_prepare($link, $count_sql);
if (!empty($search_param)) {
    mysqli_stmt_bind_param($count_stmt, "s", $search_term);
}
mysqli_stmt_execute($count_stmt);
$count_result = mysqli_stmt_get_result($count_stmt);
$total_members = mysqli_fetch_assoc($count_result)['total_members'];
$total_pages = ceil($total_members / $members_per_page);
mysqli_stmt_close($count_stmt);

// --- FETCH REGISTERED MEMBERS FOR THE CURRENT PAGE ---
$sql = "SELECT id, member_id, full_name, email, membership_status, profile_picture, role 
        FROM users WHERE role IN ('member', 'pending')"
        . $search_param .
        " ORDER BY full_name ASC 
        LIMIT ? OFFSET ?";

$stmt = mysqli_prepare($link, $sql);
if (!empty($search_param)) {
    mysqli_stmt_bind_param($stmt, "sii", $search_term, $members_per_page, $offset);
} else {
    mysqli_stmt_bind_param($stmt, "ii", $members_per_page, $offset);
}
mysqli_stmt_execute($stmt);
$result = mysqli_stmt_get_result($stmt);

$members = [];
if ($result) {
    while ($row = mysqli_fetch_assoc($result)) {
        $members[] = $row;
    }
}
mysqli_stmt_close($stmt);
?>

<?php require_once 'includes/header.php'; ?>
<div class="container-fluid">
    <div class="row">
        <?php require_once 'includes/sidebar.php'; ?>
        <main class="col-md-9 col-lg-10 main-content">
            <h2>Member Directory</h2>
            
            <div class="card mb-4">
                <div class="card-header">Search Directory</div>
                <div class="card-body">
                    <form action="member_directory.php" method="GET">
                        <div class="input-group">
                            <input type="text" name="search" placeholder="Search members by name..." value="<?php echo htmlspecialchars($search_query); ?>" class="form-control">
                            <button type="submit" class="btn btn-primary">Search</button>
                        </div>
                    </form>
                </div>
            </div>

            <div class="card">
                <div class="card-header d-flex justify-content-between align-items-center">
                    Registered Members List
                    <button class="btn btn-secondary btn-sm" onclick="printTable('membersTable')"><i class="fas fa-print me-1"></i> Print</button>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table table-striped table-hover" id="membersTable">
                            <thead>
                                <tr>
                                    <th>Full Name</th>
                                    <th>Email</th>
                                    <th>Status</th>
                                    <th>Membership Rank</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (empty($members)): ?>
                                    <tr>
                                        <td colspan="4" class="text-center">No registered members found.</td>
                                    </tr>
                                <?php else: ?>
                                    <?php foreach ($members as $member): ?>
                                    <tr>
                                        <td>
                                            <div class="d-flex align-items-center">
                                                <img src="<?php echo htmlspecialchars($member['profile_picture'] ?? 'assets/images/default_avatar.png'); ?>" alt="<?php echo htmlspecialchars($member['full_name']); ?>" class="rounded-circle me-2" style="width: 40px; height: 40px; object-fit: cover;">
                                                <span><?php echo htmlspecialchars($member['full_name']); ?></span>
                                            </div>
                                        </td>
                                        <td><?php echo htmlspecialchars($member['email'] ?? 'N/A'); ?></td>
                                        <td><?php echo htmlspecialchars($member['membership_status'] ?? 'N/A'); ?></td>
                                        <td><?php echo htmlspecialchars($member['role'] ?? 'N/A'); ?></td>
                                    </tr>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                    
                    <nav>
                        <ul class="pagination justify-content-center">
                            <?php if ($total_pages > 1): ?>
                                <li class="page-item <?php echo ($current_page <= 1) ? 'disabled' : ''; ?>">
                                    <a class="page-link" href="?page=<?php echo $current_page - 1; ?>&search=<?php echo urlencode($search_query); ?>">Previous</a>
                                </li>
                                <?php for ($i = 1; $i <= $total_pages; $i++): ?>
                                    <li class="page-item <?php echo ($current_page == $i) ? 'active' : ''; ?>">
                                        <a class="page-link" href="?page=<?php echo $i; ?>&search=<?php echo urlencode($search_query); ?>"><?php echo $i; ?></a>
                                    </li>
                                <?php endfor; ?>
                                <li class="page-item <?php echo ($current_page >= $total_pages) ? 'disabled' : ''; ?>">
                                    <a class="page-link" href="?page=<?php echo $current_page + 1; ?>&search=<?php echo urlencode($search_query); ?>">Next</a>
                                </li>
                            <?php endif; ?>
                        </ul>
                    </nav>
                </div>
            </div>
        </main>
    </div>
</div>
<script>
    function printTable(tableId) {
        const table = document.getElementById(tableId);
        if (!table) return;

        const tableClone = table.cloneNode(true);

        // Remove the d-flex align-items-center for a cleaner print layout
        const tableRows = tableClone.querySelectorAll('tr');
        tableRows.forEach(row => {
            const firstCell = row.querySelector('td:first-child');
            if (firstCell) {
                const flexContainer = firstCell.querySelector('.d-flex');
                if (flexContainer) {
                    const textNode = flexContainer.querySelector('span');
                    if (textNode) {
                        firstCell.innerHTML = textNode.innerHTML;
                    }
                }
            }
        });

        // Create a new window for printing
        const win = window.open('', '', 'width=800,height=600');
        win.document.write('<html><head><title>Print Member Directory</title>');
        win.document.write('<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">');
        win.document.write('<style>');
        win.document.write('@media print {');
        win.document.write('  body { font-size: 10pt; }');
        win.document.write('  .table-responsive { overflow: visible !important; }');
        win.document.write('  .table thead th { background-color: #f2f2f2 !important; -webkit-print-color-adjust: exact; }');
        win.document.write('  table { width: 100%; border-collapse: collapse; }');
        win.document.write('  th, td { border: 1px solid #dee2e6; padding: 8px; }');
        win.document.write('  img { display: none; }'); // Hide avatars for printing
        win.document.write('}');
        win.document.write('</style>');
        win.document.write('</head><body>');
        win.document.write('<div class="container mt-4">');
        win.document.write('<h2 class="text-center">Member Directory</h2>');
        win.document.write('<table class="table table-striped">');
        win.document.write(tableClone.innerHTML);
        win.document.write('</table>');
        win.document.write('</div>');
        win.document.write('</body></html>');
        win.document.close();
        win.focus();
        win.print();
    }
</script>
<?php require_once 'includes/footer.php'; ?>