<?php
// includes/functions.php

// Initialize session with secure settings
function initialize_session() {
    if (session_status() === PHP_SESSION_NONE) {
        ini_set('session.use_strict_mode', 1);
        ini_set('session.cookie_httponly', 1);
        ini_set('session.cookie_secure', strpos($_SERVER['SERVER_PROTOCOL'], 'https') !== false ? 1 : 0);
        ini_set('session.gc_maxlifetime', 1800); // 30 minutes
        session_start();
    }
    error_log("Session initialized: ID=" . session_id() . ", Session Data=" . json_encode([
        'user_id' => $_SESSION['user_id'] ?? 'none',
        'user_role' => $_SESSION['user_role'] ?? 'none',
        'last_activity' => $_SESSION['last_activity'] ?? 'none'
    ]));
}

// Function to check if a user is logged in
function is_logged_in() {
    initialize_session();
    if (!isset($_SESSION['user_id']) || !isset($_SESSION['user_role']) || empty($_SESSION['user_role'])) {
        error_log("Invalid session: user_id=" . ($_SESSION['user_id'] ?? 'none') . ", user_role=" . ($_SESSION['user_role'] ?? 'none'));
        return false;
    }
    if (isset($_SESSION['last_activity']) && (time() - $_SESSION['last_activity'] > 1800)) {
        error_log("Session timed out for user_id=" . ($_SESSION['user_id'] ?? 'none'));
        session_unset();
        session_destroy();
        return false;
    }
    $_SESSION['last_activity'] = time();
    return true;
}

// Function to check if the logged-in user is an admin
function is_admin() {
    initialize_session();
    return isset($_SESSION['user_role']) && in_array($_SESSION['user_role'], ['admin', 'super_admin', 'accountant']);
}

// Function to check if the user has a specific permission
function user_has_permission($link, $permission_key) {
    if (!$link) {
        error_log("Database connection error in user_has_permission");
        return false;
    }
    initialize_session();
    if (!isset($_SESSION['user_role'])) {
        error_log("No role set in user_has_permission. Session: " . json_encode($_SESSION));
        return false;
    }
    $role = $_SESSION['user_role'];
    if ($role === 'super_admin') {
        return true;
    }
    $sql = "SELECT id FROM role_permissions WHERE role_name = ? AND permission_key = ?";
    $stmt = mysqli_prepare($link, $sql);
    if (!$stmt) {
        error_log("Failed to prepare statement in user_has_permission: " . mysqli_error($link));
        return false;
    }
    mysqli_stmt_bind_param($stmt, "ss", $role, $permission_key);
    mysqli_stmt_execute($stmt);
    mysqli_stmt_store_result($stmt);
    $has_permission = mysqli_stmt_num_rows($stmt) > 0;
    mysqli_stmt_close($stmt);
    return $has_permission;
}

// Function to redirect to a different page
function redirect($page) {
    header("Location: " . $page);
    exit();
}

// Function to get permissions for a specific role
function get_role_permissions($link, $role_name) {
    if (!$link) {
        error_log("Database connection error in get_role_permissions");
        return [];
    }
    $permissions = [];
    $sql = "SELECT permission_key FROM role_permissions WHERE role_name = ?";
    $stmt = mysqli_prepare($link, $sql);
    if (!$stmt) {
        error_log("Failed to prepare statement in get_role_permissions: " . mysqli_error($link));
        return [];
    }
    mysqli_stmt_bind_param($stmt, "s", $role_name);
    mysqli_stmt_execute($stmt);
    $result = mysqli_stmt_get_result($stmt);
    while ($row = mysqli_fetch_assoc($result)) {
        $permissions[] = $row['permission_key'];
    }
    mysqli_stmt_close($stmt);
    return $permissions;
}

// Function to sanitize user input
function sanitize_input($data) {
    $data = trim($data);
    $data = stripslashes($data);
    $data = htmlspecialchars($data, ENT_QUOTES, 'UTF-8');
    return $data;
}

// Function to send emails using PHPMailer
use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;

function send_mail($recipient_email, $subject, $body) {
    require 'vendor/PHPMailer/src/Exception.php';
    require 'vendor/PHPMailer/src/PHPMailer.php';
    require 'vendor/PHPMailer/src/SMTP.php';
    require_once 'includes/db_config.php';
    global $link;
    if (!$link) {
        error_log("Database connection error in send_mail");
        return "Database connection error.";
    }
    $sql = "SELECT * FROM smtp_settings LIMIT 1";
    $result = mysqli_query($link, $sql);
    if (!$result) {
        error_log("Failed to fetch SMTP settings: " . mysqli_error($link));
        return "Failed to fetch SMTP settings.";
    }
    if (mysqli_num_rows($result) === 0) {
        return "No SMTP settings configured.";
    }
    $settings = mysqli_fetch_assoc($result);
    $mail = new PHPMailer(true);
    try {
        $mail->isSMTP();
        $mail->Host = $settings['host'];
        $mail->SMTPAuth = true;
        $mail->Username = $settings['username'];
        $mail->Password = $settings['password'];
        $mail->SMTPSecure = $settings['encryption'];
        $mail->Port = $settings['port'];
        $mail->setFrom($settings['from_email'], 'Club Portal');
        $mail->addAddress($recipient_email);
        $mail->isHTML(true);
        $mail->Subject = $subject;
        $mail->Body = $body;
        $mail->send();
        return true;
    } catch (Exception $e) {
        error_log("Mailer Error: " . $mail->ErrorInfo);
        return "Mailer Error: " . $mail->ErrorInfo;
    }
}

// Function to get a single system setting
function get_setting($setting_name, $link) {
    if (!$link) {
        error_log("Database connection error in get_setting");
        return null;
    }
    $sql = "SELECT setting_value FROM system_settings WHERE setting_name = ?";
    $stmt = mysqli_prepare($link, $sql);
    if (!$stmt) {
        error_log("Failed to prepare statement in get_setting: " . mysqli_error($link));
        return null;
    }
    mysqli_stmt_bind_param($stmt, "s", $setting_name);
    mysqli_stmt_execute($stmt);
    $result = mysqli_stmt_get_result($stmt);
    if ($row = mysqli_fetch_assoc($result)) {
        mysqli_stmt_close($stmt);
        return $row['setting_value'];
    }
    mysqli_stmt_close($stmt);
    return null;
}

// Function to update a system setting
function update_setting($link, $setting_name, $setting_value) {
    if (!$link) {
        error_log("Database connection error in update_setting");
        return false;
    }
    $sql = "INSERT INTO system_settings (setting_name, setting_value) VALUES (?, ?) ON DUPLICATE KEY UPDATE setting_value = ?";
    $stmt = mysqli_prepare($link, $sql);
    if (!$stmt) {
        error_log("Failed to prepare statement in update_setting: " . mysqli_error($link));
        return false;
    }
    mysqli_stmt_bind_param($stmt, "sss", $setting_name, $setting_value, $setting_value);
    $success = mysqli_stmt_execute($stmt);
    mysqli_stmt_close($stmt);
    return $success;
}

// Function to get all payment types
function get_all_payment_types($link) {
    if (!$link) {
        error_log("Database connection error in get_all_payment_types");
        return [];
    }
    $sql = "SELECT type_name FROM payment_types WHERE deleted_at IS NULL ORDER BY type_name ASC";
    $result = mysqli_query($link, $sql);
    if (!$result) {
        error_log("Failed to fetch payment types: " . mysqli_error($link));
        return [];
    }
    $types = [];
    while ($row = mysqli_fetch_assoc($result)) {
        $types[] = $row['type_name'];
    }
    return $types;
}

// Function to get all expense types
function get_all_expense_types($link) {
    if (!$link) {
        error_log("Database connection error in get_all_expense_types");
        return [];
    }
    $sql = "SELECT type_name FROM expense_types WHERE deleted_at IS NULL ORDER BY type_name ASC";
    $result = mysqli_query($link, $sql);
    if (!$result) {
        error_log("Failed to fetch expense types: " . mysqli_error($link));
        return [];
    }
    $types = [];
    while ($row = mysqli_fetch_assoc($result)) {
        $types[] = $row['type_name'];
    }
    return $types;
}

// Function to get all benefit types
function get_all_benefit_types($link) {
    if (!$link) {
        error_log("Database connection error in get_all_benefit_types");
        return [];
    }
    $sql = "SELECT type_name FROM benefit_types WHERE deleted_at IS NULL ORDER BY type_name ASC";
    $result = mysqli_query($link, $sql);
    if (!$result) {
        error_log("Failed to fetch benefit types: " . mysqli_error($link));
        return [];
    }
    $types = [];
    while ($row = mysqli_fetch_assoc($result)) {
        $types[] = $row['type_name'];
    }
    return $types;
}

// Function to get financial totals
function get_financial_totals($link, $record_type = 'all', $start_date = null, $end_date = null) {
    if (!$link) {
        error_log("Database connection error in get_financial_totals");
        return ['total_payments' => 0, 'total_expenses' => 0, 'total_benefits' => 0, 'total_debts' => 0, 'balance_previous_month' => 0, 'balance_previous_year' => 0];
    }

    $time_filter = '';
    $bind_params = [];
    $bind_types = '';
    if ($start_date && $end_date) {
        $time_filter = " AND created_at BETWEEN ? AND ?";
        $bind_params[] = $start_date;
        $bind_params[] = $end_date;
        $bind_types .= 'ss';
    }

    $totals = [
        'total_payments' => 0,
        'total_expenses' => 0,
        'total_benefits' => 0,
        'total_debts' => 0,
        'balance_previous_month' => 0,
        'balance_previous_year' => 0
    ];

    // Total Payments
    $sql_payments = "SELECT COALESCE(SUM(amount), 0) AS total_payments
                     FROM payments
                     WHERE payment_status = 'Paid'" . ($record_type == 'all' || $record_type == 'income' ? $time_filter : ' AND FALSE');
    $stmt = mysqli_prepare($link, $sql_payments);
    if ($bind_types && !empty($bind_params)) {
        mysqli_stmt_bind_param($stmt, $bind_types, ...$bind_params);
    }
    if (mysqli_stmt_execute($stmt)) {
        $result = mysqli_stmt_get_result($stmt);
        $totals['total_payments'] = mysqli_fetch_assoc($result)['total_payments'] ?? 0;
        error_log("Total Payments: " . $totals['total_payments']);
        mysqli_free_result($result);
    } else {
        error_log("Failed to execute total_payments query: " . mysqli_stmt_error($stmt));
    }
    mysqli_stmt_close($stmt);

    // Total Expenses
    $sql_expenses = "SELECT COALESCE(SUM(amount), 0) AS total_expenses
                     FROM expenses
                     WHERE 1=1" . ($record_type == 'all' || $record_type == 'expenses' ? $time_filter : ' AND FALSE');
    $stmt = mysqli_prepare($link, $sql_expenses);
    if ($bind_types && !empty($bind_params)) {
        mysqli_stmt_bind_param($stmt, $bind_types, ...$bind_params);
    }
    if (mysqli_stmt_execute($stmt)) {
        $result = mysqli_stmt_get_result($stmt);
        $totals['total_expenses'] = mysqli_fetch_assoc($result)['total_expenses'] ?? 0;
        error_log("Total Expenses: " . $totals['total_expenses']);
        mysqli_free_result($result);
    } else {
        error_log("Failed to execute total_expenses query: " . mysqli_stmt_error($stmt));
    }
    mysqli_stmt_close($stmt);

    // Total Benefits
    $sql_benefits = "SELECT COALESCE(SUM(amount), 0) AS total_benefits
                     FROM member_benefits
                     WHERE 1=1" . ($record_type == 'all' || $record_type == 'benefits' ? $time_filter : ' AND FALSE');
    $stmt = mysqli_prepare($link, $sql_benefits);
    if ($bind_types && !empty($bind_params)) {
        mysqli_stmt_bind_param($stmt, $bind_types, ...$bind_params);
    }
    if (mysqli_stmt_execute($stmt)) {
        $result = mysqli_stmt_get_result($stmt);
        $totals['total_benefits'] = mysqli_fetch_assoc($result)['total_benefits'] ?? 0;
        error_log("Total Benefits: " . $totals['total_benefits']);
        mysqli_free_result($result);
    } else {
        error_log("Failed to execute total_benefits query: " . mysqli_stmt_error($stmt));
    }
    mysqli_stmt_close($stmt);

    // Total Debts
    $sql_debts = "SELECT COALESCE(SUM(debt), 0) AS total_debt
                  FROM (
                      SELECT 
                          po.id,
                          COALESCE(po.required_amount, 0) AS required_amount,
                          COALESCE(SUM(p.amount), 0) AS total_paid,
                          (COALESCE(po.required_amount, 0) - COALESCE(SUM(p.amount), 0)) AS debt
                      FROM payment_obligations po
                      JOIN users u ON po.user_id = u.id
                      JOIN payment_types pt ON po.payment_type = pt.type_name
                      LEFT JOIN payments p ON p.obligation_id = po.id 
                          AND p.payment_status = 'Paid'
                      WHERE po.deleted_at IS NULL
                          AND u.role = 'member'
                          AND (u.deleted_at IS NULL OR u.deleted_at > NOW())" . ($record_type == 'all' || $record_type == 'debts' ? ($start_date && $end_date ? " AND po.due_date BETWEEN ? AND ?" : '') : ' AND FALSE') . "
                      GROUP BY po.id, po.required_amount
                      HAVING debt > 0
                  ) AS debtor_summary";
    $stmt = mysqli_prepare($link, $sql_debts);
    if ($bind_types && !empty($bind_params) && ($record_type == 'all' || $record_type == 'debts')) {
        mysqli_stmt_bind_param($stmt, $bind_types, ...$bind_params);
    }
    if (mysqli_stmt_execute($stmt)) {
        $result = mysqli_stmt_get_result($stmt);
        $totals['total_debts'] = mysqli_fetch_assoc($result)['total_debt'] ?? 0;
        error_log("Total Debts: " . $totals['total_debts']);
        mysqli_free_result($result);
    } else {
        error_log("Failed to execute total_debts query: " . mysqli_stmt_error($stmt));
    }
    mysqli_stmt_close($stmt);

    // Balance Brought Forward for Previous Month
    $sql_prev_month = "SELECT (
        (SELECT COALESCE(SUM(amount), 0) FROM payments WHERE payment_status = 'Paid' AND created_at < DATE_SUB(DATE_FORMAT(NOW(), '%Y-%m-01'), INTERVAL 1 MONTH)) -
        (SELECT COALESCE(SUM(amount), 0) FROM expenses WHERE created_at < DATE_SUB(DATE_FORMAT(NOW(), '%Y-%m-01'), INTERVAL 1 MONTH)) -
        (SELECT COALESCE(SUM(amount), 0) FROM member_benefits WHERE created_at < DATE_SUB(DATE_FORMAT(NOW(), '%Y-%m-01'), INTERVAL 1 MONTH))
    ) AS balance_previous_month";
    $result = mysqli_query($link, $sql_prev_month);
    $totals['balance_previous_month'] = $result ? mysqli_fetch_assoc($result)['balance_previous_month'] ?? 0 : 0;
    error_log("Balance Previous Month: " . $totals['balance_previous_month']);

    // Balance Brought Forward for Previous Year
    $sql_prev_year = "SELECT (
        (SELECT COALESCE(SUM(amount), 0) FROM payments WHERE payment_status = 'Paid' AND created_at < DATE_SUB(DATE_FORMAT(NOW(), '%Y-01-01'), INTERVAL 1 YEAR)) -
        (SELECT COALESCE(SUM(amount), 0) FROM expenses WHERE created_at < DATE_SUB(DATE_FORMAT(NOW(), '%Y-01-01'), INTERVAL 1 YEAR)) -
        (SELECT COALESCE(SUM(amount), 0) FROM member_benefits WHERE created_at < DATE_SUB(DATE_FORMAT(NOW(), '%Y-01-01'), INTERVAL 1 YEAR))
    ) AS balance_previous_year";
    $result = mysqli_query($link, $sql_prev_year);
    $totals['balance_previous_year'] = $result ? mysqli_fetch_assoc($result)['balance_previous_year'] ?? 0 : 0;
    error_log("Balance Previous Year: " . $totals['balance_previous_year']);

    return $totals;
}

// Function to get financial type summaries
function get_financial_type_summaries($link, $record_type, $time_period, $start_date, $end_date, $type_filter) {
    $summaries = [
        'payment_types' => [],
        'expense_types' => [],
        'benefit_types' => []
    ];

    // Payment Types Summary
    if ($record_type === 'all' || $record_type === 'income') {
        $sql = "SELECT 
                    pt.type_name,
                    COALESCE(SUM(po.required_amount), 0) AS required_amount,
                    COALESCE(SUM(p.amount), 0) AS total_paid,
                    SUM(CASE 
                        WHEN pt.type_name = 'Freewill Donation' THEN 0 
                        ELSE GREATEST(COALESCE(po.required_amount, 0) - COALESCE(p.amount, 0), 0)
                    END) AS total_debt,
                    SUM(CASE 
                        WHEN pt.type_name = 'Freewill Donation' AND COALESCE(p.amount, 0) > COALESCE(po.required_amount, 0) 
                        THEN COALESCE(p.amount, 0) - COALESCE(po.required_amount, 0) 
                        ELSE 0 
                    END) AS excess_amount
                FROM payment_types pt
                LEFT JOIN payment_obligations po ON po.payment_type = pt.type_name 
                    AND po.deleted_at IS NULL
                LEFT JOIN users u ON po.user_id = u.id 
                    AND u.role = 'member' 
                    AND (u.deleted_at IS NULL OR u.deleted_at > NOW())
                LEFT JOIN payments p ON p.obligation_id = po.id 
                    AND p.payment_status = 'Paid'
                WHERE pt.deleted_at IS NULL";
        
        $bind_types = '';
        $bind_params = [];
        
        if ($time_period === 'date_range' && $start_date && $end_date) {
            $sql .= " AND DATE(po.due_date) BETWEEN ? AND ?";
            $bind_types .= 'ss';
            $bind_params[] = $start_date;
            $bind_params[] = $end_date;
        }
        
        if ($type_filter && strpos($type_filter, 'income_') === 0) {
            $type_name = explode('_', $type_filter, 2)[1];
            $sql .= " AND pt.type_name = ?";
            $bind_types .= 's';
            $bind_params[] = $type_name;
        }
        
        $sql .= " GROUP BY pt.type_name";
        
        if ($stmt = mysqli_prepare($link, $sql)) {
            if ($bind_types) {
                mysqli_stmt_bind_param($stmt, $bind_types, ...$bind_params);
            }
            if (mysqli_stmt_execute($stmt)) {
                $result = mysqli_stmt_get_result($stmt);
                while ($row = mysqli_fetch_assoc($result)) {
                    $summaries['payment_types'][] = $row;
                    error_log("Payment Type Summary - " . $row['type_name'] . ": " . json_encode($row));
                }
                mysqli_free_result($result);
            } else {
                error_log("Failed to execute payment_types summary query: " . mysqli_stmt_error($stmt));
            }
            mysqli_stmt_close($stmt);
        } else {
            error_log("Failed to prepare payment_types summary query: " . mysqli_error($link));
        }
    }

    // Expense Types Summary
    if ($record_type === 'all' || $record_type === 'expenses') {
        $sql = "SELECT et.type_name, COALESCE(SUM(e.amount), 0) AS total
                FROM expense_types et
                LEFT JOIN expenses e ON e.expense_type_id = et.id
                WHERE et.deleted_at IS NULL";
        $bind_types = '';
        $bind_params = [];
        
        if ($time_period === 'date_range' && $start_date && $end_date) {
            $sql .= " AND DATE(e.created_at) BETWEEN ? AND ?";
            $bind_types .= 'ss';
            $bind_params[] = $start_date;
            $bind_params[] = $end_date;
        }
        
        if ($type_filter && strpos($type_filter, 'expense_') === 0) {
            $type_name = explode('_', $type_filter, 2)[1];
            $sql .= " AND et.type_name = ?";
            $bind_types .= 's';
            $bind_params[] = $type_name;
        }
        
        $sql .= " GROUP BY et.type_name";
        
        if ($stmt = mysqli_prepare($link, $sql)) {
            if ($bind_types) {
                mysqli_stmt_bind_param($stmt, $bind_types, ...$bind_params);
            }
            if (mysqli_stmt_execute($stmt)) {
                $result = mysqli_stmt_get_result($stmt);
                while ($row = mysqli_fetch_assoc($result)) {
                    $summaries['expense_types'][] = $row;
                    error_log("Expense Type Summary - " . $row['type_name'] . ": " . json_encode($row));
                }
                mysqli_free_result($result);
            } else {
                error_log("Failed to execute expense_types summary query: " . mysqli_stmt_error($stmt));
            }
            mysqli_stmt_close($stmt);
        } else {
            error_log("Failed to prepare expense_types summary query: " . mysqli_error($link));
        }
    }

    // Benefit Types Summary
    if ($record_type === 'all' || $record_type === 'benefits') {
        $sql = "SELECT bt.type_name, COALESCE(SUM(b.amount), 0) AS total
                FROM benefit_types bt
                LEFT JOIN member_benefits b ON b.benefit_type_id = bt.id
                WHERE bt.deleted_at IS NULL";
        $bind_types = '';
        $bind_params = [];
        
        if ($time_period === 'date_range' && $start_date && $end_date) {
            $sql .= " AND DATE(b.created_at) BETWEEN ? AND ?";
            $bind_types .= 'ss';
            $bind_params[] = $start_date;
            $bind_params[] = $end_date;
        }
        
        if ($type_filter && strpos($type_filter, 'benefit_') === 0) {
            $type_name = explode('_', $type_filter, 2)[1];
            $sql .= " AND bt.type_name = ?";
            $bind_types .= 's';
            $bind_params[] = $type_name;
        }
        
        $sql .= " GROUP BY bt.type_name";
        
        if ($stmt = mysqli_prepare($link, $sql)) {
            if ($bind_types) {
                mysqli_stmt_bind_param($stmt, $bind_types, ...$bind_params);
            }
            if (mysqli_stmt_execute($stmt)) {
                $result = mysqli_stmt_get_result($stmt);
                while ($row = mysqli_fetch_assoc($result)) {
                    $summaries['benefit_types'][] = $row;
                    error_log("Benefit Type Summary - " . $row['type_name'] . ": " . json_encode($row));
                }
                mysqli_free_result($result);
            } else {
                error_log("Failed to execute benefit_types summary query: " . mysqli_stmt_error($stmt));
            }
            mysqli_stmt_close($stmt);
        } else {
            error_log("Failed to prepare benefit_types summary query: " . mysqli_error($link));
        }
    }

    return $summaries;
}

// Function to get financial metrics
function get_financial_metrics($link, $start_date = null, $end_date = null, $type_filter = null) {
    if (!$link) {
        error_log("Database connection error in get_financial_metrics");
        return ['total_required' => 0, 'total_income' => 0, 'excess_donations' => 0, 'available_balance' => 0];
    }

    $metrics = [
        'total_required' => 0,
        'total_income' => 0,
        'excess_donations' => 0,
        'available_balance' => 0
    ];

    // Total Required
    $sql_required = "SELECT COALESCE(SUM(po.required_amount), 0) AS total_required
                     FROM payment_obligations po
                     JOIN users u ON po.user_id = u.id
                     JOIN payment_types pt ON po.payment_type = pt.type_name
                     WHERE po.deleted_at IS NULL
                         AND u.role = 'member'
                         AND (u.deleted_at IS NULL OR u.deleted_at > NOW())";
    
    $bind_types = '';
    $bind_params = [];
    
    if ($start_date && $end_date) {
        $sql_required .= " AND DATE(po.due_date) BETWEEN ? AND ?";
        $bind_types .= 'ss';
        $bind_params[] = $start_date;
        $bind_params[] = $end_date;
    }
    
    if ($type_filter && strpos($type_filter, 'income_') === 0) {
        $type_name = explode('_', $type_filter, 2)[1];
        $sql_required .= " AND pt.type_name = ?";
        $bind_types .= 's';
        $bind_params[] = $type_name;
    }
    
    if ($stmt = mysqli_prepare($link, $sql_required)) {
        if ($bind_types) {
            mysqli_stmt_bind_param($stmt, $bind_types, ...$bind_params);
        }
        if (mysqli_stmt_execute($stmt)) {
            $result = mysqli_stmt_get_result($stmt);
            $metrics['total_required'] = mysqli_fetch_assoc($result)['total_required'] ?? 0;
            error_log("Total Required: " . $metrics['total_required']);
            mysqli_free_result($result);
        } else {
            error_log("Failed to execute total_required query: " . mysqli_stmt_error($stmt));
        }
        mysqli_stmt_close($stmt);
    } else {
        error_log("Failed to prepare total_required query: " . mysqli_error($link));
    }

    // Total Income
    $sql_income = "SELECT COALESCE(SUM(p.amount), 0) AS total_income
                   FROM payments p
                   JOIN payment_types pt ON p.payment_type = pt.type_name
                   WHERE p.payment_status = 'Paid'";
    
    $bind_types = '';
    $bind_params = [];
    
    if ($start_date && $end_date) {
        $sql_income .= " AND DATE(p.created_at) BETWEEN ? AND ?";
        $bind_types .= 'ss';
        $bind_params[] = $start_date;
        $bind_params[] = $end_date;
    }
    
    if ($type_filter && strpos($type_filter, 'income_') === 0) {
        $type_name = explode('_', $type_filter, 2)[1];
        $sql_income .= " AND pt.type_name = ?";
        $bind_types .= 's';
        $bind_params[] = $type_name;
    }
    
    if ($stmt = mysqli_prepare($link, $sql_income)) {
        if ($bind_types) {
            mysqli_stmt_bind_param($stmt, $bind_types, ...$bind_params);
        }
        if (mysqli_stmt_execute($stmt)) {
            $result = mysqli_stmt_get_result($stmt);
            $metrics['total_income'] = mysqli_fetch_assoc($result)['total_income'] ?? 0;
            error_log("Total Income: " . $metrics['total_income']);
            mysqli_free_result($result);
        } else {
            error_log("Failed to execute total_income query: " . mysqli_stmt_error($stmt));
        }
        mysqli_stmt_close($stmt);
    } else {
        error_log("Failed to prepare total_income query: " . mysqli_error($link));
    }

    // Excess Donations
    $sql_excess = "SELECT COALESCE(SUM(CASE 
                        WHEN pt.type_name = 'Freewill Donation' AND COALESCE(p.amount, 0) > COALESCE(po.required_amount, 0) 
                        THEN COALESCE(p.amount, 0) - COALESCE(po.required_amount, 0) 
                        ELSE 0 
                    END), 0) AS excess_donations
                   FROM payment_types pt
                   LEFT JOIN payment_obligations po ON po.payment_type = pt.type_name 
                       AND po.deleted_at IS NULL
                   LEFT JOIN users u ON po.user_id = u.id 
                       AND u.role = 'member' 
                       AND (u.deleted_at IS NULL OR u.deleted_at > NOW())
                   LEFT JOIN payments p ON p.obligation_id = po.id 
                       AND p.payment_status = 'Paid'
                   WHERE pt.deleted_at IS NULL
                       AND pt.type_name = 'Freewill Donation'";
    
    $bind_types = '';
    $bind_params = [];
    
    if ($start_date && $end_date) {
        $sql_excess .= " AND DATE(po.due_date) BETWEEN ? AND ?";
        $bind_types .= 'ss';
        $bind_params[] = $start_date;
        $bind_params[] = $end_date;
    }
    
    if ($stmt = mysqli_prepare($link, $sql_excess)) {
        if ($bind_types) {
            mysqli_stmt_bind_param($stmt, $bind_types, ...$bind_params);
        }
        if (mysqli_stmt_execute($stmt)) {
            $result = mysqli_stmt_get_result($stmt);
            $metrics['excess_donations'] = mysqli_fetch_assoc($result)['excess_donations'] ?? 0;
            error_log("Excess Donations: " . $metrics['excess_donations']);
            mysqli_free_result($result);
        } else {
            error_log("Failed to execute excess_donations query: " . mysqli_stmt_error($stmt));
        }
        mysqli_stmt_close($stmt);
    } else {
        error_log("Failed to prepare excess_donations query: " . mysqli_error($link));
    }

    // Calculate Available Balance
    $totals = get_financial_totals($link, 'all', $start_date, $end_date);
    $metrics['available_balance'] = $metrics['total_income'] - $totals['total_expenses'] - $totals['total_benefits'];
    error_log("Available Balance: " . $metrics['available_balance']);

    error_log("Financial Metrics: " . json_encode($metrics));
    return $metrics;
}

// Function to truncate text to a specified length
function truncate_text($text, $length) {
    if (strlen($text) > $length) {
        return substr($text, 0, $length) . '...';
    }
    return $text;
}

// Function to format currency
function format_currency($amount, $currency_symbol) {
    return htmlspecialchars($currency_symbol . ' ' . number_format((float)$amount, 2), ENT_QUOTES, 'UTF-8');
}

// Function to include global scripts
function include_global_scripts() {
    echo '<script src="includes/global.js"></script>';
}

// Token System Functions
function get_token_balance($link, $user_id) {
    if (!$link) {
        error_log("Database connection error in get_token_balance");
        return 0;
    }
    $sql = "SELECT SUM(balance) AS total_balance FROM token_balances WHERE user_id = ? AND balance > 0";
    $stmt = mysqli_prepare($link, $sql);
    if (!$stmt) {
        error_log("Failed to prepare statement in get_token_balance: " . mysqli_error($link));
        return 0;
    }
    mysqli_stmt_bind_param($stmt, "i", $user_id);
    mysqli_stmt_execute($stmt);
    $result = mysqli_stmt_get_result($stmt);
    $row = mysqli_fetch_assoc($result);
    mysqli_stmt_close($stmt);
    return (int)($row['total_balance'] ?? 0);
}

function deduct_tokens($link, $user_id, $action_type, $tokens_to_deduct) {
    if (!$link) {
        error_log("Database connection error in deduct_tokens");
        return false;
    }
    $total_balance = get_token_balance($link, $user_id);
    if ($total_balance < $tokens_to_deduct) {
        error_log("Insufficient tokens for user_id $user_id: need $tokens_to_deduct, have $total_balance");
        return false;
    }

    // Map action_type string to int
    $action_type_map = ['payment' => 1, 'expense' => 2, 'benefit' => 3];
    $action_type_id = $action_type_map[$action_type] ?? 0;

    mysqli_begin_transaction($link);
    try {
        // FIFO: Get oldest positive balance rows
        $sql_rows = "SELECT id, balance FROM token_balances WHERE user_id = ? AND balance > 0 ORDER BY id ASC"; // Oldest first
        $stmt_rows = mysqli_prepare($link, $sql_rows);
        mysqli_stmt_bind_param($stmt_rows, "i", $user_id);
        mysqli_stmt_execute($stmt_rows);
        $result_rows = mysqli_stmt_get_result($stmt_rows);
        $remaining = $tokens_to_deduct;

        while ($row = mysqli_fetch_assoc($result_rows)) {
            if ($remaining <= 0) break;
            $deduct_from_this = min($remaining, $row['balance']);
            $new_balance = $row['balance'] - $deduct_from_this;

            // Update balance row
            $sql_update = "UPDATE token_balances SET balance = ?, last_updated = NOW() WHERE id = ?";
            $stmt_update = mysqli_prepare($link, $sql_update);
            mysqli_stmt_bind_param($stmt_update, "ii", $new_balance, $row['id']);
            mysqli_stmt_execute($stmt_update);
            mysqli_stmt_close($stmt_update);

            $remaining -= $deduct_from_this;
        }
        mysqli_stmt_close($stmt_rows);

        if ($remaining > 0) {
            throw new Exception("Insufficient after deduction—race condition?");
        }

        // Log the usage (now logs actual deducted amount)
        $sql_log = "INSERT INTO token_usage_logs (user_id, action_type, tokens_used, timestamp) VALUES (?, ?, ?, NOW())";
        $stmt_log = mysqli_prepare($link, $sql_log);
        mysqli_stmt_bind_param($stmt_log, "iii", $user_id, $action_type_id, $tokens_to_deduct);
        mysqli_stmt_execute($stmt_log);
        mysqli_stmt_close($stmt_log);

        mysqli_commit($link);
        error_log("Deducted $tokens_to_deduct tokens for user_id $user_id, action_type: $action_type (id: $action_type_id)");
        return $tokens_to_deduct;
    } catch (Exception $e) {
        mysqli_rollback($link);
        error_log("Token deduction failed: " . $e->getMessage());
        return false;
    }
}

// Function to get total users by role
function get_total_users_by_role($link, $role) {
    if (!$link) {
        error_log("Database connection error in get_total_users_by_role");
        return 0;
    }
    $sql = "SELECT COUNT(*) AS total FROM users WHERE role = ? AND deleted_at IS NULL";
    if ($stmt = mysqli_prepare($link, $sql)) {
        mysqli_stmt_bind_param($stmt, "s", $role);
        mysqli_stmt_execute($stmt);
        $result = mysqli_stmt_get_result($stmt);
        $row = mysqli_fetch_assoc($result);
        mysqli_stmt_close($stmt);
        return (int)$row['total'];
    }
    error_log("Error fetching total users for role $role: " . mysqli_error($link));
    return 0;
}
?>