<?php
require_once '../includes/db_config.php';
require_once '../includes/functions.php';

header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
    exit;
}

$input = json_decode(file_get_contents('php://input'), true);
$user_ids = isset($input['user_ids']) ? array_map('intval', $input['user_ids']) : [];

if (empty($user_ids)) {
    echo json_encode(['obligations' => []]);
    exit;
}

$currency_symbol = get_setting('currency_symbol', $link) ?? '₦';
$placeholders = implode(',', array_fill(0, count($user_ids), '?'));
$sql = "SELECT po.id, po.user_id, po.payment_type, po.required_amount, 
               COALESCE(SUM(p.amount), 0) AS total_paid,
               (COALESCE(po.required_amount, 0) - COALESCE(SUM(p.amount), 0)) AS debt
        FROM payment_obligations po
        LEFT JOIN payments p ON p.obligation_id = po.id AND p.payment_status = 'Paid'
        WHERE po.user_id IN ($placeholders) AND po.deleted_at IS NULL
        GROUP BY po.id, po.user_id, po.payment_type, po.required_amount
        HAVING debt > 0
        ORDER BY po.payment_type, po.created_at";

if ($stmt = mysqli_prepare($link, $sql)) {
    $bind_types = str_repeat('i', count($user_ids));
    mysqli_stmt_bind_param($stmt, $bind_types, ...$user_ids);
    mysqli_stmt_execute($stmt);
    $result = mysqli_stmt_get_result($stmt);
    $obligations = [];
    while ($row = mysqli_fetch_assoc($result)) {
        $row['currency_symbol'] = $currency_symbol;
        $obligations[] = $row;
    }
    mysqli_stmt_close($stmt);
    echo json_encode(['obligations' => $obligations]);
} else {
    error_log("Failed to prepare obligations query: " . mysqli_error($link));
    http_response_code(500);
    echo json_encode(['error' => 'Database error']);
}

mysqli_close($link);
?>