<?php
// This API fetches all financial data (summary, transactions, and payment types)
// for the member financials dashboard.
require_once '../includes/db_config.php';

// Set the content type to JSON
header('Content-Type: application/json');

// Get search and period filters from the request.
$search = isset($_GET['search']) ? $_GET['search'] : '';
$period = isset($_GET['period']) ? $_GET['period'] : 'all';

// Initialize the response array
$response = [
    'success' => false,
    'summary' => [],
    'transactions' => [],
    'payment_types_summary' => [],
    'message' => 'An unknown error occurred.'
];

// Ensure the database connection is valid
if (!$link) {
    $response['message'] = 'Database connection failed.';
    echo json_encode($response);
    exit;
}

try {
    // --- 1. Fetch Summary Totals (Total Paid, This Month, Debtors) ---
    $summary_sql = "
        SELECT 
            (SELECT COALESCE(SUM(amount), 0) FROM payments WHERE status = 'Paid') AS total_paid,
            (SELECT COALESCE(SUM(amount), 0) FROM payments WHERE status = 'Paid' AND MONTH(created_at) = MONTH(CURRENT_DATE()) AND YEAR(created_at) = YEAR(CURRENT_DATE())) AS paid_this_month,
            (SELECT COUNT(DISTINCT user_id) FROM payments WHERE status = 'Pending') AS outstanding_debtors
    ";
    $summary_stmt = $link->prepare($summary_sql);
    $summary_stmt->execute();
    $summary_result = $summary_stmt->get_result();
    $response['summary'] = $summary_result->fetch_assoc();

    // --- 2. Fetch Financial Transactions Table Data ---
    $transactions_sql = "
        SELECT 
            u.id AS member_id, 
            u.full_name, 
            u.membership_status, 
            p.payment_type, 
            p.amount AS amount_paid, 
            p.status AS payment_status,
            p.created_at AS transaction_date
        FROM users u 
        LEFT JOIN payments p ON u.id = p.user_id 
        WHERE 1=1
    ";

    $params = [];
    $types = '';

    // Add search filter if provided
    if (!empty($search)) {
        $transactions_sql .= " AND (u.full_name LIKE ? OR u.surname LIKE ? OR u.other_names LIKE ?)";
        $search_term = "%" . $search . "%";
        $params[] = $search_term;
        $params[] = $search_term;
        $params[] = $search_term;
        $types .= 'sss';
    }

    // Add period filter if provided
    switch ($period) {
        case 'today':
            $transactions_sql .= " AND DATE(p.created_at) = CURRENT_DATE()";
            break;
        case 'this_week':
            $transactions_sql .= " AND YEARWEEK(p.created_at, 1) = YEARWEEK(CURRENT_DATE(), 1)";
            break;
        case 'this_month':
            $transactions_sql .= " AND MONTH(p.created_at) = MONTH(CURRENT_DATE()) AND YEAR(p.created_at) = YEAR(CURRENT_DATE())";
            break;
        case 'this_year':
            $transactions_sql .= " AND YEAR(p.created_at) = YEAR(CURRENT_DATE())";
            break;
    }
    
    // Order by the most recent payments
    $transactions_sql .= " ORDER BY p.created_at DESC";

    $transactions_stmt = $link->prepare($transactions_sql);
    if ($transactions_stmt === false) {
        throw new Exception("Prepare failed on transactions query: " . $link->error);
    }

    if (!empty($params)) {
        $transactions_stmt->bind_param($types, ...$params);
    }
    
    $transactions_stmt->execute();
    $transactions_result = $transactions_stmt->get_result();
    $response['transactions'] = $transactions_result->fetch_all(MYSQLI_ASSOC);

    // --- 3. Fetch Payment Types Summary ---
    $payment_types_sql = "
        SELECT 
            p.payment_type, 
            u.id AS member_id, 
            u.full_name, 
            COALESCE(SUM(p.amount), 0) AS amount
        FROM payments p
        INNER JOIN users u ON u.id = p.user_id
        WHERE p.status = 'Paid'
        GROUP BY p.payment_type, u.id, u.full_name
        ORDER BY p.payment_type, u.full_name
    ";

    $payment_types_stmt = $link->prepare($payment_types_sql);
    $payment_types_stmt->execute();
    $payment_types_result = $payment_types_stmt->get_result();

    $summary_by_type = [];
    while ($row = $payment_types_result->fetch_assoc()) {
        $type = $row['payment_type'];
        if (!isset($summary_by_type[$type])) {
            $summary_by_type[$type] = [
                'total_paid' => 0,
                'members' => []
            ];
        }
        $summary_by_type[$type]['total_paid'] += $row['amount'];
        $summary_by_type[$type]['members'][] = [
            'member_id' => $row['member_id'],
            'full_name' => $row['full_name'],
            'amount' => $row['amount']
        ];
    }
    $response['payment_types_summary'] = $summary_by_type;

    // Set success to true if all queries were successful
    $response['success'] = true;
    $response['message'] = 'Financial data retrieved successfully.';

} catch (Exception $e) {
    // Catch any exception and set the response message
    $response['message'] = 'Error fetching data: ' . $e->getMessage();
}

// Close database connection
if ($link) {
    $link->close();
}

// Return the final JSON response
echo json_encode($response);
?>
