<?php
require_once 'includes/functions.php';
require_once 'includes/db_config.php';

// FIX 1: Start session to access CSRF token
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// FIX 2: Add CSRF token validation
if (!isset($_POST['csrf_token']) || !hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {
    echo json_encode(['error' => 'Invalid CSRF token']);
    exit;
}

if (!is_logged_in() || !is_admin()) {
    echo json_encode(['error' => 'Unauthorized']);
    exit;
}

$id = intval($_POST['id'] ?? 0);
if ($id <= 0) {
    echo json_encode(['error' => 'Invalid member ID.']);
    exit;
}

$full_name = sanitize_input($_POST['full_name'] ?? '');
$email = sanitize_input($_POST['email'] ?? '');
$form_data_raw = $_POST['form_data'] ?? '{}'; // JSON string from JS
$form_data_changes = json_decode($form_data_raw, true);

if (json_last_error() !== JSON_ERROR_NONE) {
    echo json_encode(['error' => 'Invalid form data.']);
    exit;
}

$profile_picture = null;
$profile_picture_sql_part = ""; // We'll build this dynamically

// Handle profile picture upload
if (isset($_FILES['profile_picture']) && $_FILES['profile_picture']['error'] == UPLOAD_ERR_OK) {
    // Basic validation for image type
    $allowed_types = ['image/jpeg', 'image/png', 'image/gif'];
    $finfo = finfo_open(FILEINFO_MIME_TYPE);
    $mime_type = finfo_file($finfo, $_FILES['profile_picture']['tmp_name']);
    finfo_close($finfo);

    if (in_array($mime_type, $allowed_types)) {
        $ext = strtolower(pathinfo($_FILES['profile_picture']['name'], PATHINFO_EXTENSION));
        $filename = 'profile_' . $id . '_' . time() . '.' . $ext;
        $upload_dir = __DIR__ . '/uploads/profile_pictures/';
        
        if (!file_exists($upload_dir)) {
            if (!mkdir($upload_dir, 0777, true)) {
                // Failed to create directory, stop execution
                echo json_encode(['error' => 'Failed to create upload directory. Check permissions.']);
                exit;
            }
        }
        
        $upload_path = $upload_dir . $filename;
        if (move_uploaded_file($_FILES['profile_picture']['tmp_name'], $upload_path)) {
            $profile_picture = 'uploads/profile_pictures/' . $filename;
            // Only if upload is successful, we add this to the SQL
            $profile_picture_sql_part = ", profile_picture = ?";
        } else {
            echo json_encode(['error' => 'Failed to move uploaded file. Check permissions.']);
            exit;
        }
    } else {
        echo json_encode(['error' => 'Invalid file type. Only JPEG, PNG, and GIF are allowed.']);
        exit;
    }
}

// Update basic user info
// The SQL now dynamically includes the profile picture part
$sql = "UPDATE users
        SET full_name = ?, email = ? $profile_picture_sql_part
        WHERE id = ?";
$stmt = mysqli_prepare($link, $sql);

if (!$stmt) {
    echo json_encode(['error' => 'Database error (prepare): ' . mysqli_error($link)]);
    exit;
}

// Bind parameters based on whether a new picture was uploaded
if ($profile_picture) {
    mysqli_stmt_bind_param($stmt, "sssi", $full_name, $email, $profile_picture, $id);
} else {
    mysqli_stmt_bind_param($stmt, "ssi", $full_name, $email, $id);
}

if (!mysqli_stmt_execute($stmt)) {
    echo json_encode(['error' => 'Database error (execute): ' . mysqli_stmt_error($stmt)]);
    exit;
}
mysqli_stmt_close($stmt);


// Now update or insert the registration (form_data) section
// FIX 3: Non-destructive update (merge)
if (!empty($form_data_changes)) {
    // Sanitize all incoming editable values
    foreach ($form_data_changes as $key => $val) {
        $form_data_changes[$key] = sanitize_input($val);
    }

    // Check if a record exists in form_data for this user
    $check_sql = "SELECT id, data FROM form_data WHERE user_id = ?";
    $check_stmt = mysqli_prepare($link, $check_sql);
    mysqli_stmt_bind_param($check_stmt, "i", $id);
    mysqli_stmt_execute($check_stmt);
    $check_result = mysqli_stmt_get_result($check_stmt);
    $existing = mysqli_fetch_assoc($check_result);
    mysqli_stmt_close($check_stmt);

    if ($existing) {
        // Update existing record
        $existing_data = json_decode($existing['data'], true) ?: [];
        // Merge the changes into the existing data
        $merged_data = array_merge($existing_data, $form_data_changes);
        $updated_json = json_encode($merged_data, JSON_UNESCAPED_UNICODE);

        $update_sql = "UPDATE form_data SET data = ? WHERE id = ?";
        $update_stmt = mysqli_prepare($link, $update_sql);
        mysqli_stmt_bind_param($update_stmt, "si", $updated_json, $existing['id']);
        mysqli_stmt_execute($update_stmt);
        mysqli_stmt_close($update_stmt);
    } else {
        // Insert a new record (as no data existed before)
        $new_json = json_encode($form_data_changes, JSON_UNESCAPED_UNICODE);
        $insert_sql = "INSERT INTO form_data (user_id, data) VALUES (?, ?)";
        $insert_stmt = mysqli_prepare($link, $insert_sql);
        mysqli_stmt_bind_param($insert_stmt, "is", $id, $new_json);
        mysqli_stmt_execute($insert_stmt);
        mysqli_stmt_close($insert_stmt);
    }
}

echo json_encode(['message' => 'Member updated successfully']);
?>
