<?php
// transfer_funds.php - Form to move money between accounts (e.g., Cash at Hand to Bank)

ob_start();
session_start();
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);
mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);

// IMPORTANT: Include the ledger functions for balance updates and double-entry recording
require_once 'includes/functions.php';
require_once 'includes/db_config.php';
// --- NOTE: We are replacing the old ledger logic, so ledger_functions.php is no longer strictly needed but good to keep. ---
require_once 'includes/ledger_functions.php'; 

/* ---------- SESSION / PERMISSIONS ---------- */
if (!is_logged_in() || !is_admin()) {
    redirect("login.php");
    exit;
}
if (!isset($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}
$current_user_id = $_SESSION['user_id']; 

$message = '';
$currency_symbol = get_setting('currency_symbol', $link) ?? '₦';

// 1. Fetch available accounts for dropdowns
$accounts = [];
$sql_accounts = "SELECT id, account_name, current_balance FROM bank_accounts WHERE deleted_at IS NULL ORDER BY id";
if ($result = mysqli_query($link, $sql_accounts)) {
    while ($row = mysqli_fetch_assoc($result)) {
        $accounts[] = $row;
    }
} else {
    $message = "Error fetching accounts: " . mysqli_error($link);
}

// 2. Handle Fund Transfer POST request
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['transfer'])) {
    // CSRF check
    if (!isset($_POST['csrf_token']) || !hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {
        $message = "Error: Invalid request token.";
    } else {
        $amount = (float)sanitize_input($_POST['amount']); // Ensure float
        $from_account_id = intval($_POST['from_account_id']);
        $to_account_id = intval($_POST['to_account_id']);
        $reason = sanitize_input($_POST['reason']);
        
        // Basic validation
        if ($from_account_id === $to_account_id) {
            $message = "Error: Cannot transfer funds to the same account.";
        } elseif ($amount <= 0) {
            $message = "Error: Amount must be greater than zero.";
        } else {
            // Find the current balance of the 'From' account
            $from_account = array_filter($accounts, function ($acc) use ($from_account_id) {
                return $acc['id'] == $from_account_id;
            });
            $from_account = reset($from_account);

            if (!$from_account) {
                $message = "Error: Source account not found.";
            } elseif ($from_account['current_balance'] < $amount) {
                // --- LOGIC CHANGE: Allow Cheque Payable to go negative ---
                // We check if the account name *contains* 'Cheques Payable'
                if (strpos($from_account['account_name'], 'Cheques Payable') === false) {
                    $message = "Error: Insufficient funds in the source account (" . $from_account['account_name'] . ").";
                }
            } 
            
            // Proceed if no error message was set
            if (empty($message)) {

                // Start transaction for atomicity of ALL operations
                mysqli_begin_transaction($link);
                $success = true;

                try {
                    $transfer_description = "Internal Fund Transfer: " . $reason;
                    
                    // --- *** LEDGER FIX START *** ---
                    // Replace old debit/credit logic with new transaction_type/amount logic
                    
                    // 1. Log 'transfer_out' (Outflow) from the 'From' account
                    // This is like an 'expense'
                    $sql_outflow = "INSERT INTO financial_ledger 
                                      (bank_account_id, transaction_type, amount, description, reference_id, transaction_date, recorded_by_id) 
                                      VALUES (?, 'transfer_out', ?, ?, NULL, NOW(), ?)";
                    if (!($stmt_outflow = mysqli_prepare($link, $sql_outflow))) { $success = false; throw new Exception("Prepare failed (Outflow): " . mysqli_error($link)); }
                    mysqli_stmt_bind_param($stmt_outflow, "idsi", $from_account_id, $amount, $transfer_description, $current_user_id);
                    if (!mysqli_stmt_execute($stmt_outflow)) { $success = false; throw new Exception("Execution failed (Outflow): " . mysqli_stmt_error($stmt_outflow)); }
                    mysqli_stmt_close($stmt_outflow);


                    // 2. Log 'transfer_in' (Inflow) to the 'To' account
                    // This is like an 'income'
                    $sql_inflow = "INSERT INTO financial_ledger 
                                     (bank_account_id, transaction_type, amount, description, reference_id, transaction_date, recorded_by_id) 
                                     VALUES (?, 'transfer_in', ?, ?, NULL, NOW(), ?)";
                    if (!($stmt_inflow = mysqli_prepare($link, $sql_inflow))) { $success = false; throw new Exception("Prepare failed (Inflow): " . mysqli_error($link)); }
                    mysqli_stmt_bind_param($stmt_inflow, "idsi", $to_account_id, $amount, $transfer_description, $current_user_id);
                    if (!mysqli_stmt_execute($stmt_inflow)) { $success = false; throw new Exception("Execution failed (Inflow): " . mysqli_stmt_error($stmt_inflow)); }
                    mysqli_stmt_close($stmt_inflow);

                    // --- *** LEDGER FIX END *** ---


                    // --- 3. Update Denormalized Account Balances (Unchanged) ---

                    // 3a. Subtract from 'From' account
                    $sql_out_bal = "UPDATE bank_accounts SET current_balance = current_balance - ? WHERE id = ?";
                    if (!($stmt_out_bal = mysqli_prepare($link, $sql_out_bal))) { $success = false; throw new Exception("Prepare failed (Balance Out): " . mysqli_error($link)); }
                    mysqli_stmt_bind_param($stmt_out_bal, "di", $amount, $from_account_id);
                    if (!mysqli_stmt_execute($stmt_out_bal)) { $success = false; throw new Exception("Execution failed (Balance Out): " . mysqli_stmt_error($stmt_out_bal)); }
                    mysqli_stmt_close($stmt_out_bal);

                    // 3b. Add to 'To' account
                    $sql_in_bal = "UPDATE bank_accounts SET current_balance = current_balance + ? WHERE id = ?";
                    if (!($stmt_in_bal = mysqli_prepare($link, $sql_in_bal))) { $success = false; throw new Exception("Prepare failed (Balance In): " . mysqli_error($link)); }
                    mysqli_stmt_bind_param($stmt_in_bal, "di", $amount, $to_account_id);
                    if (!mysqli_stmt_execute($stmt_in_bal)) { $success = false; throw new Exception("Execution failed (Balance In): " . mysqli_stmt_error($stmt_in_bal)); }
                    mysqli_stmt_close($stmt_in_bal);


                    // If all steps succeeded
                    mysqli_commit($link);
                    $message = "Success: Funds of {$currency_symbol}" . number_format($amount, 2) . " successfully transferred.";
                    // Redirect to clear POST data
                    redirect("transfer_funds.php?msg=" . urlencode($message));
                    exit;

                } catch (Exception $e) {
                    // Rollback on any error
                    mysqli_rollback($link);
                    error_log("Transfer Error: " . $e->getMessage());
                    $message = "Transaction failed: " . $e->getMessage();
                }
            }
        }
    }
}

// Check for success message from redirect
if (isset($_GET['msg'])) {
    $message = sanitize_input($_GET['msg']);
}

// --- Refresh accounts data after potential transfer ---
$accounts = []; // Clear old data
$sql_accounts = "SELECT id, account_name, current_balance FROM bank_accounts WHERE deleted_at IS NULL ORDER BY id";
if ($result = mysqli_query($link, $sql_accounts)) {
    while ($row = mysqli_fetch_assoc($result)) {
        $accounts[] = $row;
    }
}

require_once 'includes/header.php';
?>
<!-- Main application layout: Sidebar + Main Content Area -->
<div class.container-fluid">
    <div class="row">
        <?php require_once 'includes/sidebar.php'; ?>
        
        <!-- MAIN CONTENT AREA -->
        <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4 pt-5">
            
            <!-- Header Section -->
            <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pb-2 mb-3 border-bottom">
                <h1 class="h2">Internal Fund Transfer</h1>
            </div>
            
            <div class="row justify-content-center">
                <div class="col-lg-8">
                    <div class="card shadow-sm">
                        <div class="card-header bg-primary text-white">
                            <i class="fas fa-exchange-alt me-2"></i>Transfer Funds Between Accounts
                        </div>
                        <div class="card-body">
                            <?php if ($message): ?>
                                <div class="alert alert-<?= strpos($message, 'Error') !== false || strpos($message, 'failed') !== false ? 'danger' : 'success' ?>" role="alert">
                                    <?= htmlspecialchars($message) ?>
                                </div>
                            <?php endif; ?>

                            <form action="transfer_funds.php" method="POST">
                                <input type="hidden" name="csrf_token" value="<?= htmlspecialchars($_SESSION['csrf_token']) ?>">
                                <input type="hidden" name="transfer" value="1">

                                <div class="mb-3">
                                    <label for="from_account_id" class="form-label fw-bold">Transfer FROM Account</label>
                                    <select name="from_account_id" id="from_account_id" class="form-select" required>
                                        <option value="">Select Source Account</option>
                                        <?php foreach ($accounts as $account): ?>
                                            <option value="<?= htmlspecialchars($account['id']) ?>">
                                                <?= htmlspecialchars($account['account_name']) ?> (Balance: <?= $currency_symbol . number_format($account['current_balance'], 2) ?>)
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>

                                <div class="mb-3">
                                    <label for="to_account_id" class="form-label fw-bold">Transfer TO Account</label>
                                    <select name="to_account_id" id="to_account_id" class="form-select" required>
                                        <option value="">Select Destination Account</option>
                                        <?php foreach ($accounts as $account): ?>
                                            <option value="<?= htmlspecialchars($account['id']) ?>">
                                                <?= htmlspecialchars($account['account_name']) ?> (Balance: <?= $currency_symbol . number_format($account['current_balance'], 2) ?>)
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>

                                <div class="mb-3">
                                    <label for="amount" class="form-label fw-bold">Amount (<?= $currency_symbol ?>)</label>
                                    <input type="number" name="amount" id="amount" class="form-control" step="0.01" min="0.01" required>
                                </div>

                                <div class="mb-3">
                                    <label for="reason" class="form-label">Reason/Reference</label>
                                    <input type="text" name="reason" id="reason" class="form-control" maxlength="255" required placeholder="e.g., Clear cheque #12345">
                                </div>

                                <button type="submit" class="btn btn-success w-100">
                                    <i class="fas fa-check-circle me-2"></i>Execute Transfer
                                </button>
                            </form>
                        </div>
                    </div>
                </div>
            </div>
        </main>
    </div>
</div>

<?php require_once 'includes/footer.php'; ?>
<?php ob_end_flush(); ?>