<?php
ob_start(); // Start output buffering to prevent header issues
require_once 'includes/functions.php';
require_once 'includes/db_config.php';

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

$error_message = '';
$success_message = '';
$form_data = null;
$posted_data = []; // Store POST data to repopulate form on error

// Generate math CAPTCHA only if not already set
if (!isset($_SESSION['captcha_answer']) || !isset($_SESSION['captcha_question'])) {
    $num1 = rand(1, 10);
    $num2 = rand(1, 10);
    $_SESSION['captcha_answer'] = $num1 + $num2;
    $_SESSION['captcha_question'] = "What is $num1 + $num2?";
}
$captcha_question = $_SESSION['captcha_question'];

// Fetch the most recent form from the database
$sql = "SELECT id, form_data FROM forms ORDER BY id DESC LIMIT 1";
$result = mysqli_query($link, $sql);
if ($result && mysqli_num_rows($result) > 0) {
    $row = mysqli_fetch_assoc($result);
    $registration_form_id = $row['id'];
    $form_data = json_decode($row['form_data'], true);
} else {
    $error_message = "No registration form available. Please contact the administrator.";
}

// --- Conditional Logic Extraction ---
$conditional_logic_map = [];
// --- NEW: Identify core fields needed for validation later ---
$has_email_field = false;
$has_username_field = false;
$has_password_field = false;

if ($form_data && isset($form_data['fields']) && is_array($form_data['fields'])) {
    foreach ($form_data['fields'] as $field) {
        // Identify core fields
        if ($field['type'] === 'email') $has_email_field = true;
        // Fix for 'text' field acting as 'Full Name' or 'Username'
        if ($field['type'] === 'username' || ($field['type'] === 'text' && (strpos(strtolower($field['label'] ?? ''), 'name') !== false || strpos(strtolower($field['label'] ?? ''), 'username') !== false))) $has_username_field = true;
        if ($field['type'] === 'password') $has_password_field = true;

        // Extract conditional logic
        if (!empty($field['name']) && !empty($field['conditional_logic']) && !empty($field['conditional_logic']['rules'])) {
            $logic = $field['conditional_logic'];
            $conditional_logic_map[$field['name']] = [
                'action' => $logic['action'] ?? 'show',
                'match' => $logic['match'] ?? 'all',
                'rules' => $logic['rules']
            ];
        }
    }
}
// --- END NEW BLOCK ---

/**
 * --- Secure File Upload Handler ---
 */
function handle_file_upload($field_name, $field_label, $is_required, &$error_message) {
    if (!isset($_FILES[$field_name]) || $_FILES[$field_name]['error'] == UPLOAD_ERR_NO_FILE) {
        if ($is_required) {
            $error_message = "The file '$field_label' is required.";
        }
        return null; // No file uploaded, but not required
    }

    $file = $_FILES[$field_name];

    if ($file['error'] !== UPLOAD_ERR_OK) {
        $error_message = "Error uploading '$field_label': Error code " . $file['error'];
        return null;
    }

    $allowed_types = ['image/jpeg', 'image/png', 'application/pdf'];
    $max_size = 5 * 1024 * 1024; // 5 MB

    if (!in_array($file['type'], $allowed_types)) {
        $error_message = "Invalid file type for '$field_label'. Only JPEG, PNG, or PDF are allowed.";
        return null;
    }

    if ($file['size'] > $max_size) {
        $error_message = "File '$field_label' is too large. Maximum size is 5 MB.";
        return null;
    }

    $upload_dir = 'uploads/'; // Make sure this directory exists and is writable
    if (!is_dir($upload_dir)) {
        mkdir($upload_dir, 0755, true);
    }
    
    $file_ext = pathinfo($file['name'], PATHINFO_EXTENSION);
    $safe_filename = uniqid('userfile_', true) . '.' . strtolower($file_ext);
    $new_file_path = $upload_dir . $safe_filename;

    if (move_uploaded_file($file['tmp_name'], $new_file_path)) {
        return $new_file_path; // Return the path to store in DB
    } else {
        $error_message = "Could not move uploaded file for '$field_label'.";
        return null;
    }
}


// Handle form submission
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['form_submission'])) {
    
    $posted_data = $_POST;

    // Validate CAPTCHA
    $captcha_input = (int) sanitize_input($_POST['captcha_input']);
    $captcha_answer = (int) $_SESSION['captcha_answer'];
    
    if (!isset($_SESSION['captcha_answer']) || $captcha_input !== $captcha_answer) {
        $error_message = "Incorrect CAPTCHA answer.";
    } else {
        unset($_SESSION['captcha_answer']);
        unset($_SESSION['captcha_question']);

        $user_data = [];
        $email = null;
        $full_name = ''; // This variable MUST be populated correctly
        $password = '';

        // Process form fields
        if ($form_data && isset($form_data['fields']) && is_array($form_data['fields'])) {
            foreach ($form_data['fields'] as $field) {
                if (empty($field['name']) || empty($field['type'])) continue;

                $field_type = $field['type'];
                $field_label = $field['label'] ?? $field['name'];
                $field_name = $field['name'];
                $is_required = $field['required'] ?? false;
                
                $input_value = null;

                // --- Handle File Uploads Separately ---
                if ($field_type === 'file') {
                    $input_value = handle_file_upload($field_name, $field_label, $is_required, $error_message);
                    if ($error_message) break;
                
                // --- Handle standard POST data ---
                } else if (isset($_POST[$field_name])) {
                    $input_value = $_POST[$field_name];
                    
                    if (is_array($input_value)) {
                        $input_value = array_map('sanitize_input', $input_value);
                    } else {
                        $input_value = sanitize_input($input_value);
                    }
                }

                // Check for required fields
                // NOTE: Conditional logic handles hiding/showing. Server still checks for required fields.
                if ($is_required && empty($input_value)) {
                    $error_message = "The field '$field_label' is required.";
                    break;
                }

                // Specific validations and core field capture
                if ($field_type === 'email') {
                    if ($input_value && !filter_var($input_value, FILTER_VALIDATE_EMAIL)) {
                        $error_message = "Invalid email format for '$field_label'.";
                        break;
                    }
                    $email = $input_value;
                } elseif ($field_type === 'password') {
                    if ($input_value && strlen($input_value) < 6) {
                        $error_message = "Password must be at least 6 characters long.";
                        break;
                    }
                    $password = $input_value;
                } 
                // --- FIX: Correctly identify and capture the 'Full Name' text field ---
                elseif ($field_type === 'username' || 
                        ($field_type === 'text' && (strpos(strtolower($field_label), 'name') !== false || strpos(strtolower($field_label), 'username') !== false))) {
                    // Only assign if we haven't already captured a full_name, or if the current field is explicitly a username type.
                    if (empty($full_name) || $field_type === 'username') {
                        $full_name = $input_value;
                    }
                }
                // --- END FIX ---

                // Store data for JSON
                if (is_array($input_value)) {
                    $user_data[$field_name] = implode(',', $input_value);
                } else if ($field_type === 'password') {
                    continue; // Do not store raw password in custom data
                } else {
                    $user_data[$field_name] = $input_value;
                }
            }
        }

        // --- Original Block: Dynamic check for core required fields ---
        if (empty($error_message)) {
            // This ensures $full_name is set when $has_username_field is true.
            if ($has_username_field && empty($full_name)) {
                $error_message = "A 'Full Name' or 'Username' field is required for registration.";
            } elseif ($has_email_field && empty($email)) {
                $error_message = "An 'Email' field is required for registration.";
            } elseif ($has_password_field && empty($password)) {
                $error_message = "A 'Password' field is required for registration.";
            }
        }
        // --- END Original Block ---

        // Proceed with registration if no errors
        if (empty($error_message)) {
            // Check if email already exists
            if ($has_email_field && !empty($email)) {
                $sql = "SELECT id FROM users WHERE email = ?";
                $stmt = mysqli_prepare($link, $sql);
                // ... (database connection and execution code for email check) ...
                if ($stmt === false) { $error_message = "A database error occurred. Please try again later."; error_log("Email check prepare failed: " . mysqli_error($link)); } else { mysqli_stmt_bind_param($stmt, "s", $email); if (!mysqli_stmt_execute($stmt)) { $error_message = "A database error occurred. Please try again later."; error_log("Email check execute failed: " . mysqli_error($link)); } else { mysqli_stmt_store_result($stmt); if (mysqli_stmt_num_rows($stmt) > 0) { $error_message = "This email is already registered."; } } mysqli_stmt_close($stmt); }
            }

            // Proceed only if no errors and email is unique
            if (empty($error_message)) {
                // Only register if we have the necessary credentials defined by the form builder
                if ($has_username_field && $has_email_field && $has_password_field) {
                    $hashed_password = password_hash($password, PASSWORD_DEFAULT);

                    // Generate unique member ID
                    $initials = '';
                    $words = explode(' ', $full_name);
                    foreach ($words as $word) { if (!empty($word)) { $initials .= strtoupper($word[0]); } }
                    $unique_member_id = '';
                    $attempts = 0;
                    $max_attempts = 10;
                    do {
                        $serialized_digits = str_pad(rand(0, 999), 3, '0', STR_PAD_LEFT);
                        $unique_member_id = $initials . $serialized_digits;
                        $check_sql = "SELECT id FROM users WHERE member_id = ?";
                        $check_stmt = mysqli_prepare($link, $check_sql);
                        if ($check_stmt === false) { $error_message = "A database error occurred. Please try again later."; break; }
                        mysqli_stmt_bind_param($check_stmt, "s", $unique_member_id);
                        if (!mysqli_stmt_execute($check_stmt)) { $error_message = "A database error occurred. Please try again later."; mysqli_stmt_close($check_stmt); break; }
                        mysqli_stmt_store_result($check_stmt);
                        $exists = mysqli_stmt_num_rows($check_stmt) > 0;
                        mysqli_stmt_close($check_stmt);
                        $attempts++;
                    } while ($exists && $attempts < $max_attempts);

                    if ($attempts >= $max_attempts) {
                        $error_message = "Unable to generate a unique member ID. Please try again.";
                    } elseif (empty($error_message)) {
                        // Insert new user
                        $sql = "INSERT INTO users (member_id, full_name, email, password, role, membership_status, created_at) VALUES (?, ?, ?, ?, 'member', 'pending', NOW())";
                        $stmt = mysqli_prepare($link, $sql);
                        if ($stmt === false) { $error_message = "A database error occurred. Please try again later."; } else { mysqli_stmt_bind_param($stmt, "ssss", $unique_member_id, $full_name, $email, $hashed_password); if (mysqli_stmt_execute($stmt)) { $user_id = mysqli_insert_id($link); $success_message = "Registration successful! Your account is pending approval. Your Member ID is: $unique_member_id"; $posted_data = []; // Clear data on success
                                // Save custom form data
                                $custom_data = json_encode($user_data);
                                $data_sql = "INSERT INTO form_data (form_id, user_id, data, created_at) VALUES (?, ?, ?, NOW())";
                                $data_stmt = mysqli_prepare($link, $data_sql);
                                if ($data_stmt === false) { $error_message = "A database error occurred. Please try again later."; } else { mysqli_stmt_bind_param($data_stmt, "iis", $registration_form_id, $user_id, $custom_data); if (!mysqli_stmt_execute($data_stmt)) { $error_message = "A database error occurred. Please try again later."; } mysqli_stmt_close($data_stmt); }
                            } else { $error_message = "A database error occurred. Please try again later."; }
                            mysqli_stmt_close($stmt);
                        }
                    }
                } else {
                    // Only save form data if we don't need to create a new user record
                    if (!empty($user_data)) {
                        // This case is typically for forms that DON'T create a user account, but we'll leave the logic simple for now.
                        // If you intend for this page to ONLY register users, this entire 'else' block can be omitted,
                        // and you can enforce the three core fields as required in the form builder.
                    }
                }
            }
        }
    }
    
    // Generate new CAPTCHA for next attempt if there was an error
    if (!empty($error_message)) {
        $num1 = rand(1, 10);
        $num2 = rand(1, 10);
        $_SESSION['captcha_answer'] = $num1 + $num2;
        $_SESSION['captcha_question'] = "What is $num1 + $num2?";
        $captcha_question = $_SESSION['captcha_question'];
    }
}
?>

<?php require_once 'includes/header.php'; ?>
<script src="assets/js/jquery-3.7.1.min.js"></script>

<div class="container-fluid">
    <div class="row justify-content-center">
        <div class="col-md-8 col-lg-6">
            <div class="card">
                <div class="card-header text-center">
                    <h2>Join Our Club</h2>
                </div>
                <div class="card-body">
                    <?php if (!empty($error_message)): ?>
                        <div class="message error-message"><?php echo htmlspecialchars($error_message); ?></div>
                    <?php endif; ?>
                    <?php if (!empty($success_message)): ?>
                        <div class="message success-message"><?php echo htmlspecialchars($success_message); ?></div>
                    <?php else: ?>
                        <div class="registration-form-container">
                            <form id="registration-form" action="<?php echo htmlspecialchars($_SERVER["PHP_SELF"]); ?>" method="post" enctype="multipart/form-data">
                                <?php if ($form_data && isset($form_data['fields']) && is_array($form_data['fields'])): ?>
                                    <?php foreach ($form_data['fields'] as $field): ?>
                                        <?php
                                        // Field properties extraction
                                        $field_type = $field['type'] ?? 'text';
                                        $field_label = $field['label'] ?? $field['name'];
                                        $field_name = $field['name'];
                                        $is_required = $field['required'] ?? false;
                                        $consent_text = $field['consent_text'] ?? '';
                                        
                                        // UX: Get posted value for repopulation
                                        $posted_value = $posted_data[$field_name] ?? null;
                                        ?>
                                        
                                        <div class="form-group mb-3 form-field-<?php echo htmlspecialchars($field_name); ?>">
                                            <label for="<?php echo htmlspecialchars($field_name); ?>" class="form-label">
                                                <?php echo htmlspecialchars($field_label); ?>
                                                <?php echo $is_required ? '<span class="text-danger">*</span>' : ''; ?>
                                            </label>
                                            
                                            <?php switch ($field_type):
                                                case 'text':
                                                case 'email':
                                                case 'username':
                                                case 'date':
                                                     ?>
                                                     <input type="<?php echo htmlspecialchars($field_type); ?>" 
                                                            class="form-control" 
                                                            id="<?php echo htmlspecialchars($field_name); ?>" 
                                                            name="<?php echo htmlspecialchars($field_name); ?>" 
                                                            value="<?php echo htmlspecialchars($posted_value ?? ''); ?>"
                                                            <?php echo $is_required ? 'required' : ''; ?>>
                                                     <?php break; case 'password': ?>
                                                     <input type="password" 
                                                            class="form-control" 
                                                            id="<?php echo htmlspecialchars($field_name); ?>" 
                                                            name="<?php echo htmlspecialchars($field_name); ?>" 
                                                            minlength="6" 
                                                            <?php echo $is_required ? 'required' : ''; ?>>
                                                     <?php break; case 'textarea': ?>
                                                     <textarea class="form-control" 
                                                                id="<?php echo htmlspecialchars($field_name); ?>" 
                                                                name="<?php echo htmlspecialchars($field_name); ?>" 
                                                                rows="4" 
                                                                <?php echo $is_required ? 'required' : ''; ?>><?php echo htmlspecialchars($posted_value ?? ''); ?></textarea>
                                                     <?php break; case 'select': ?>
                                                     <select class="form-select" 
                                                             id="<?php echo htmlspecialchars($field_name); ?>" 
                                                             name="<?php echo htmlspecialchars($field_name); ?>" 
                                                             <?php echo $is_required ? 'required' : ''; ?>>
                                                          <option value="">Select an option...</option>
                                                          <?php if (isset($field['options']) && is_array($field['options'])): ?>
                                                              <?php foreach ($field['options'] as $option): 
                                                                  $opt_val = $option['value'] ?? '';
                                                                  $opt_label = $option['label'] ?? '';
                                                              ?>
                                                                  <option value="<?php echo htmlspecialchars($opt_val); ?>" <?php echo ($posted_value == $opt_val) ? 'selected' : ''; ?>>
                                                                      <?php echo htmlspecialchars($opt_label); ?>
                                                                  </option>
                                                              <?php endforeach; ?>
                                                          <?php endif; ?>
                                                     </select>
                                                     <?php break; ?>

                                                 <?php case 'checkbox': // Checkbox GROUP ?>
                                                     <?php if (isset($field['options']) && is_array($field['options'])): ?>
                                                         <?php foreach ($field['options'] as $index => $option): 
                                                             $opt_val = $option['value'] ?? '';
                                                             $opt_label = $option['label'] ?? '';
                                                             $is_checked = is_array($posted_value) && in_array($opt_val, $posted_value);
                                                         ?>
                                                             <div class="form-check">
                                                                 <input type="checkbox" 
                                                                        class="form-check-input" 
                                                                        name="<?php echo htmlspecialchars($field_name); ?>[]" 
                                                                        value="<?php echo htmlspecialchars($opt_val); ?>" 
                                                                        id="<?php echo htmlspecialchars($field_name . '_' . $index); ?>"
                                                                        <?php echo $is_checked ? 'checked' : ''; ?>>
                                                                 <label class="form-check-label" for="<?php echo htmlspecialchars($field_name . '_' . $index); ?>">
                                                                      <?php echo htmlspecialchars($opt_label); ?>
                                                                 </label>
                                                             </div>
                                                         <?php endforeach; ?>
                                                     <?php endif; ?>
                                                     <?php break; ?>

                                                 <?php case 'consent': ?>
                                                     <div class="form-check">
                                                         <input type="checkbox" 
                                                                class="form-check-input" 
                                                                id="<?php echo htmlspecialchars($field_name); ?>" 
                                                                name="<?php echo htmlspecialchars($field_name); ?>" 
                                                                value="1" 
                                                                <?php echo ($posted_value == '1') ? 'checked' : ''; ?>
                                                                <?php echo $is_required ? 'required' : ''; ?>>
                                                         <label class="form-check-label" for="<?php echo htmlspecialchars($field_name); ?>">
                                                              <?php echo htmlspecialchars($field_label); ?>
                                                         </label>
                                                     </div>
                                                     <?php if (!empty($consent_text)): ?>
                                                         <div class="consent-text-container mt-2 p-2 border rounded" style="font-size: 0.9em; max-height: 150px; overflow-y: auto;">
                                                              <?php echo $consent_text; ?>
                                                         </div>
                                                     <?php endif; ?>
                                                     <?php break; ?>

                                                 <?php case 'radio': ?>
                                                     <?php if (isset($field['options']) && is_array($field['options'])): ?>
                                                         <?php foreach ($field['options'] as $index => $option): 
                                                             $opt_val = $option['value'] ?? '';
                                                             $opt_label = $option['label'] ?? '';
                                                         ?>
                                                             <div class="form-check">
                                                                 <input type="radio" 
                                                                        class="form-check-input" 
                                                                        name="<?php echo htmlspecialchars($field_name); ?>" 
                                                                        value="<?php echo htmlspecialchars($opt_val); ?>"
                                                                        id="<?php echo htmlspecialchars($field_name . '_' . $index); ?>"
                                                                        <?php echo ($posted_value == $opt_val) ? 'checked' : ''; ?>
                                                                        <?php echo $is_required ? 'required' : ''; ?>>
                                                                 <label class="form-check-label" for="<?php echo htmlspecialchars($field_name . '_' . $index); ?>">
                                                                      <?php echo htmlspecialchars($opt_label); ?>
                                                                 </label>
                                                             </div>
                                                         <?php endforeach; ?>
                                                     <?php endif; ?>
                                                     <?php break; case 'file': ?>
                                                     <input type="file" 
                                                            class="form-control" 
                                                            id="<?php echo htmlspecialchars($field_name); ?>" 
                                                            name="<?php echo htmlspecialchars($field_name); ?>" 
                                                            <?php echo $is_required ? 'required' : ''; ?>>
                                                     <?php break; default: ?>
                                                     <input type="text" 
                                                            class="form-control" 
                                                            id="<?php echo htmlspecialchars($field_name); ?>" 
                                                            name="<?php echo htmlspecialchars($field_name); ?>" 
                                                            value="<?php echo htmlspecialchars($posted_value ?? ''); ?>"
                                                            <?php echo $is_required ? 'required' : ''; ?>>
                                            <?php endswitch; ?>
                                        </div>
                                    <?php endforeach; ?>
                                <?php else: ?>
                                    <div class="alert alert-warning">
                                         No form fields available. Please contact the administrator.
                                    </div>
                                <?php endif; ?>
                                
                                <hr>
                                
                                <div class="form-group mb-3">
                                    <label for="captcha_input" class="form-label">
                                        <strong>Security Question:</strong> <?php echo htmlspecialchars($captcha_question); ?>
                                    </label>
                                    <input type="text" class="form-control" id="captcha_input" name="captcha_input" required>
                                </div>
                                
                                <input type="hidden" name="form_submission" value="1">
                                <div class="form-actions mt-3 text-center">
                                    <button type="submit" class="btn btn-success btn-lg">Complete Registration</button>
                                </div>
                            </form>
                        </div>
                    <?php endif; ?>
                </div>
                <div class="card-footer text-center">
                    <p>Already have an account? <a href="login.php">Login here</a></p>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
    // Pass the extracted logic data to the form_renderer.js script
    var FORM_CONDITIONAL_LOGIC = <?php echo json_encode($conditional_logic_map); ?>;
</script>
<script src="assets/js/form_renderer.js"></script>

<?php ob_end_flush(); // End output buffering ?>
<?php require_once 'includes/footer.php'; ?>