<?php
// record_payments.php
ob_start();
session_start();
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);
mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);
if (function_exists('opcache_reset')) { opcache_reset(); }

require_once 'includes/functions.php';
require_once 'includes/db_config.php';

/* ---------- SESSION / PERMISSIONS ---------- */
$timeout = 1800;
if (isset($_SESSION['last_activity']) && (time() - $_SESSION['last_activity']) > $timeout) {
    session_unset(); session_destroy();
    redirect("login.php?timeout=1");
}
$_SESSION['last_activity'] = time();

if (!isset($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}
if (!is_logged_in() || !user_has_permission($link, 'view_payments') ||
    !in_array($_SESSION['user_role'], ['admin', 'accountant', 'super_admin'])) { // Added super_admin
    redirect("login.php"); exit;
}

/* ---------- SETTINGS ---------- */
$currency_symbol = get_setting('currency_symbol', $link) ?? '₦';
$token_cost      = (int)get_setting('tokens_per_payment', $link) ?: 1;
$deduct_per_member = (int)get_setting('deduct_per_member', $link) ?? 0;

// --- NEW: Get User Role and ID ---
$current_user_id = $_SESSION['user_id'];
$user_role = $_SESSION['user_role']; 

/* ---------- ORGANIZATION BALANCE FUNCTION (ALL ADMINS + ACCOUNTANTS) ---------- */
/**
 * Retrieves the total active token balance across all users with 'admin' or 'accountant' roles.
 * @param mysqli $link The database connection link.
 * @return int The total organization token balance.
 */
function get_organization_token_balance($link) {
    $sql = "SELECT COALESCE(SUM(tb.balance), 0) AS total
            FROM token_balances tb
            JOIN users u ON tb.user_id = u.id
            WHERE u.role IN ('admin', 'accountant', 'super_admin')
              AND u.deleted_at IS NULL
              AND tb.balance > 0";
    $result = mysqli_query($link, $sql);
    if ($result && $row = mysqli_fetch_assoc($result)) {
        mysqli_free_result($result);
        return (int)$row['total'];
    }
    return 0;
}

/* ---------- INITIAL BALANCES ---------- */
$current_user_balance      = get_user_token_balance($link, $current_user_id);
$organization_total_tokens = get_organization_token_balance($link);

/* ---------- FETCH BANK ACCOUNTS (FUNDING SOURCES) - SECURED ---------- */
$bank_accounts = [];
// --- NEW: Secure query based on user role ---
$sql_banks = "SELECT id, account_name, bank_name, account_number, is_cash_on_hand, current_balance, owner_user_id 
              FROM bank_accounts 
              WHERE is_active = 1 AND deleted_at IS NULL"; // Also check deleted_at

$params = [];
$types = "";

// --- *** SECURITY FIX V2 (Corrected) *** ---
// If user is NOT an admin, restrict to ONLY public accounts (bank or cash).
// Hide private imprest accounts and the Cheque Holding account.
if (!in_array($user_role, ['admin', 'super_admin'])) {
    $sql_banks .= " AND owner_user_id IS NULL 
                  AND account_name != 'Cheques Payable (Holding)'";
}
// --- *** END FIX *** ---

$sql_banks .= " ORDER BY is_cash_on_hand DESC, bank_name ASC";

$stmt_banks = mysqli_prepare($link, $sql_banks);
if ($stmt_banks) {
    if (!empty($params)) {
        mysqli_stmt_bind_param($stmt_banks, $types, ...$params);
    }
    mysqli_stmt_execute($stmt_banks);
    $result_banks = mysqli_stmt_get_result($stmt_banks);
    
    while ($row = mysqli_fetch_assoc($result_banks)) {
        // Use 'bank_name' if 'account_name' is missing, fallback to 'Cash at Hand'
        $account_display = $row['account_name'] ?? $row['bank_name'];
        // Create a user-friendly display name for the dropdown
        $display_balance = $currency_symbol . number_format($row['current_balance'], 2);
        if ($row['is_cash_on_hand']) {
            $row['display_name'] = htmlspecialchars($account_display) . " (Cash/Imprest) - {$display_balance}";
        } else {
            $row['display_name'] = htmlspecialchars($account_display . ' (' . $row['bank_name'] . ') - ') . $display_balance;
        }
        $bank_accounts[] = $row;
    }
    mysqli_stmt_close($stmt_banks);
}
// --- END: Secure query ---


/* ---------- FORM HANDLING ---------- */
$message = $message_type = '';
if ($_SERVER["REQUEST_METHOD"] == "POST"
    && isset($_POST['log_payment'])
    && isset($_POST['csrf_token'])
    && hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {

    $user_ids        = $_POST['user_ids'] ?? [];
    $amount          = floatval($_POST['amount'] ?? 0);
    $obligation_id   = intval($_POST['obligation_id'] ?? 0);
    // Note: payment_type is read from the obligation and is not sent via POST, but is needed for logging
    $payment_type    = ''; 
    $description     = sanitize_input($_POST['description'] ?? '');
    $status          = 'Paid';
    
    // Get the bank account ID where the income will be deposited
    $bank_account_id = intval($_POST['bank_account_id'] ?? 0);

    // Calculate required tokens for logging the transaction(s)
    $num_logs        = $deduct_per_member ? count($user_ids) : 1;
    $required_tokens = $token_cost * $num_logs;

    // Check 1: User has enough tokens to perform the transaction
    if ($current_user_balance < $required_tokens) {
        $message = "Insufficient tokens. You need $required_tokens token".($required_tokens>1?'s':'')
                 .", but you have $current_user_balance. <a href='token_store.php'>Buy / request more</a>.";
        $message_type = 'danger';
    } else {
        $valid_obligation = false;
        $debt_amount = 0;
        $is_freewill = false;
        
        // --- NEW: Validate that the user is allowed to deposit into this account ---
        $is_allowed_account = false;
        if ($bank_account_id > 0) {
            // We check against the SECURE list of $bank_accounts fetched earlier
            foreach ($bank_accounts as $allowed_acc) {
                if ($allowed_acc['id'] == $bank_account_id) {
                    $is_allowed_account = true;
                    break;
                }
            }
        }
        if (!$is_allowed_account) {
            $message = "Invalid 'Deposit To' account selection. You may not have permission to deposit into this account.";
            $message_type = 'danger';
        }
        // --- END: Account validation ---
        
        // Check 2: Validate Obligation ID and Debt (must exist and have remaining debt/be freewill)
        if ($message_type !== 'danger' && $obligation_id > 0 && !empty($user_ids)) {
            // Check obligation validity and get remaining debt amount
            $sql_check = "SELECT po.id,
                                (COALESCE(po.required_amount,0)-COALESCE(SUM(p.amount),0)) AS debt,
                                pt.payment_category, po.payment_type
                          FROM payment_obligations po
                          LEFT JOIN payments p ON p.obligation_id=po.id AND p.payment_status='Paid'
                          JOIN payment_types pt ON po.payment_type=pt.type_name AND pt.deleted_at IS NULL
                          WHERE po.id=? AND po.user_id IN (".implode(',',array_fill(0,count($user_ids),'?')).")
                            AND po.deleted_at IS NULL
                          GROUP BY po.id,pt.payment_category,po.payment_type
                          HAVING debt>0 OR pt.payment_category = 'freewill'"; // Allow freewill even if debt is 0
            
            $stmt_check = mysqli_prepare($link, $sql_check);
            $bind = array_merge([$obligation_id], array_map('intval',$user_ids));
            // Create type string: 'i' for obligation_id, then 'i' for each user_id
            $types = 'i'.str_repeat('i',count($user_ids));
            
            // Need to manually pass arguments since array_merge of reference and values is complex
            $params = array_merge([$types], $bind);
            mysqli_stmt_bind_param($stmt_check, ...$params);
            
            mysqli_stmt_execute($stmt_check);
            $res = mysqli_stmt_get_result($stmt_check);
            if ($row = mysqli_fetch_assoc($res)) {
                $valid_obligation = true;
                $debt_amount = $row['debt'];
                $is_freewill = $row['payment_category'] === 'freewill';
                $payment_type = $row['payment_type']; // Get the actual type name
            }
            mysqli_stmt_close($stmt_check);
        }

        // Check 3: Final validation before transaction begins
        if ($message_type !== 'danger' && !empty($user_ids) && $amount > 0 && $valid_obligation && $bank_account_id > 0 && ($is_freewill || $amount <= $debt_amount)) {
            
            mysqli_begin_transaction($link);
            $ok = true;
            $tokens_used = 0;
            // Calculate total income based on the amount per member * number of members
            $total_income = $amount * count($user_ids); 

            // 1. Deduct tokens from the current user
            $tokens_used = deduct_tokens($link, $current_user_id, 'payment', $required_tokens);
            if ($tokens_used === false) {
                $ok = false;
                $message = "Token deduction failed.";
                $message_type = 'danger';
            }

            // 2. Update Bank Account Balance (One time for the total income)
            if ($ok) {
                $sql_bank = "UPDATE bank_accounts SET current_balance = current_balance + ? WHERE id = ?";
                $stmt_bank = mysqli_prepare($link, $sql_bank);
                if ($stmt_bank) {
                    mysqli_stmt_bind_param($stmt_bank, "di", $total_income, $bank_account_id);
                    if (!mysqli_stmt_execute($stmt_bank)) {
                        $ok = false;
                        $message = "Failed to update bank balance: " . mysqli_stmt_error($stmt_bank);
                        $message_type = 'danger';
                    }
                    mysqli_stmt_close($stmt_bank);
                } else {
                    $ok = false;
                    $message = "Failed to prepare bank update statement.";
                    $message_type = 'danger';
                }
            }

            // 3. Prepare SQL for Payments and Ledger (for loop use)
            $stmt_pay = $stmt_ledger = null; // Init vars
            if ($ok) {
                // Payment insertion statement
                $sql_pay = "INSERT INTO payments
                            (user_id, payment_type, amount, description, payment_status, obligation_id, bank_account_id, payment_date, created_at)
                            VALUES (?, ?, ?, ?, ?, ?, ?, NOW(), NOW())";
                $stmt_pay = mysqli_prepare($link, $sql_pay);
                
                // Ledger insertion statement
                $sql_ledger = "INSERT INTO financial_ledger
                               (bank_account_id, transaction_type, amount, description, reference_id, transaction_date, recorded_by_id)
                               VALUES (?, 'income', ?, ?, ?, NOW(), ?)";
                $stmt_ledger = mysqli_prepare($link, $sql_ledger);

                if (!$stmt_pay || !$stmt_ledger) {
                    $ok = false;
                    $message = "Failed to prepare SQL statements.";
                    $message_type = 'danger';
                    if ($stmt_pay) mysqli_stmt_close($stmt_pay);
                    if ($stmt_ledger) mysqli_stmt_close($stmt_ledger);
                }
            }

            // 4. Loop through each user to log payment AND ledger entry
            if ($ok) {
                foreach ($user_ids as $uid) {
                    $uid = intval($uid);
                    
                    // Log the payment
                    // Bind parameters for $stmt_pay: user_id(i), payment_type(s), amount(d), description(s), status(s), obligation_id(i), bank_account_id(i)
                    mysqli_stmt_bind_param($stmt_pay, "isdssii",
                        $uid, $payment_type, $amount, $description, $status, $obligation_id, $bank_account_id);
                    
                    if (!mysqli_stmt_execute($stmt_pay)) {
                        $ok = false;
                        $message = "Payment insert error for user $uid: " . mysqli_stmt_error($stmt_pay);
                        $message_type = 'danger';
                        break; // Exit loop on failure
                    }
                    
                    $payment_id = mysqli_insert_id($link); // Get the ID of the payment we just made
                    
                    // Log to financial ledger
                    $ledger_desc = "Payment: '$payment_type' by User ID: $uid (Ref Pay ID: $payment_id)";
                    // Bind parameters for $stmt_ledger: bank_account_id(i), amount(d), description(s), reference_id(i), recorded_by_id(i)
                    mysqli_stmt_bind_param($stmt_ledger, "idsii", 
                        $bank_account_id, $amount, $ledger_desc, $payment_id, $current_user_id);

                    if (!mysqli_stmt_execute($stmt_ledger)) {
                        $ok = false;
                        $message = "Ledger insert error for user $uid: " . mysqli_stmt_error($stmt_ledger);
                        $message_type = 'danger';
                        break; // Exit loop on failure
                    }
                }
                
                // Close statements after loop
                if ($stmt_pay) mysqli_stmt_close($stmt_pay);
                if ($stmt_ledger) mysqli_stmt_close($stmt_ledger);
            }

            // 5. Commit or Rollback
            if ($ok) {
                mysqli_commit($link);
                $message = "Payment logged! Used $tokens_used token".($tokens_used>1?'s':'').".";
                $message_type = 'success';

                /* ---- REFRESH BALANCES ---- */
                $current_user_balance      = get_user_token_balance($link, $current_user_id);
                $organization_total_tokens = get_organization_token_balance($link);
            } else {
                mysqli_rollback($link);
                // Ensure a fallback error message if a specific one wasn't set
                if (!$message) $message = "Transaction rolled back due to an unknown error.";
                $message_type = 'danger';
            }
        } else {
            // More descriptive error
            $error_reason = "";
            if ($bank_account_id <= 0) $error_reason = "a funding source was not selected.";
            elseif (!$is_allowed_account) $error_reason = "you do not have permission to deposit into that account."; // New check
            elseif (!$valid_obligation) $error_reason = "the obligation is invalid or already paid.";
            elseif ($amount <= 0) $error_reason = "the amount is zero.";
            elseif (empty($user_ids)) $error_reason = "no members were selected.";
            // This covers the case where amount > debt and it's not freewill
            elseif (!$is_freewill && $amount > $debt_amount) $error_reason = "the amount ($currency_symbol" . number_format($amount, 2) . ") is greater than the remaining debt ($currency_symbol" . number_format($debt_amount, 2) . ") for the selected obligation.";
            else $error_reason = "an unexpected validation error occurred.";
            
            $message = "Invalid data – check members, amount, or obligation. Reason: $error_reason";
            $message_type = 'danger';
        }
    }
    // Refresh CSRF token on POST
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
} elseif ($_SERVER["REQUEST_METHOD"] == "POST") {
    $message = "Invalid CSRF token.";
    $message_type = 'danger';
}

/* ---------- MEMBERS LIST FOR DROPDOWN ---------- */
$members = [];
$res = mysqli_query($link,
    "SELECT id,full_name,member_id FROM users
     WHERE role='member' AND (deleted_at IS NULL OR deleted_at>NOW())
     ORDER BY full_name");
if ($res) {
    while ($row = mysqli_fetch_assoc($res)) $members[] = $row;
    mysqli_free_result($res);
}

// Fetch current currency symbol for display purposes
$currency_symbol = get_setting('currency_symbol', $link) ?? '₦';
?>
<?php require_once 'includes/header.php'; ?>
<script>document.title='Record Payments - <?=htmlspecialchars(get_setting('system_name',$link)??'Club Portal')?>';</script>
<style>.main-content{padding:20px;}</style>

<div class="container-fluid">
    <div class="row">
        <?php require_once 'includes/sidebar.php'; ?>
        <main class="col-md-9 col-lg-10 main-content">
            <h2>Record Payments</h2>
            <?php if ($message): ?>
                <div class="alert alert-<?=$message_type?>" role="alert"><?=$message?></div>
            <?php endif; ?>

            <div class="card mb-4 shadow-sm">
                <div class="card-header bg-primary text-white">
                    <i class="fas fa-money-bill-wave"></i> Log New Payment
                </div>
                <div class="card-body">
                    <div class="d-flex justify-content-between mb-3 border-bottom pb-2">
                        <p class="mb-0"><strong>Your Token Balance:</strong> <span class="badge bg-success"><?=$current_user_balance?> tokens</span></p>
                        <p class="mb-0"><strong>Org Token Balance:</strong> <span class="badge bg-info text-dark"><?=$organization_total_tokens?> tokens</span></p>
                    </div>

                    <form action="record_payments.php" method="post">
                        <input type="hidden" name="log_payment" value="1">
                        <input type="hidden" name="csrf_token" value="<?=htmlspecialchars($_SESSION['csrf_token'])?>">

                        <div class="mb-3">
                            <label class="form-label fw-bold">Select Member(s)</label>
                            <select name="user_ids[]" id="user_ids" class="form-select" multiple size="6" required>
                                <?php foreach ($members as $m): ?>
                                    <option value="<?=htmlspecialchars($m['id'])?>">
                                        <?=htmlspecialchars($m['full_name'].' (ID: '.$m['member_id'].')')?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                            <small class="form-text text-muted">
                                Hold Ctrl/Cmd to select multiple members. Cost: <strong><?=$token_cost?> token<?=$token_cost>1?'s':''?></strong>
                                <?= $deduct_per_member ? ' (per member logged)' : ' (per batch log)' ?>.
                            </small>
                        </div>

                        <div class="mb-3">
                            <label class="form-label fw-bold">Select Obligation</label>
                            <select name="obligation_id" id="obligation_id" class="form-select" required onchange="updatePaymentType()">
                                <option value="">Select an Obligation</option>
                                <?php
                                // Fetch obligations relevant to the selected members (or all if none selected initially)
                                $sql_ob = "SELECT po.id, po.payment_type,
                                            (COALESCE(po.required_amount,0)-COALESCE(SUM(p.amount),0)) AS debt,
                                            pt.payment_category
                                       FROM payment_obligations po
                                       LEFT JOIN payments p ON p.obligation_id=po.id AND p.payment_status='Paid'
                                       JOIN payment_types pt ON po.payment_type=pt.type_name AND pt.deleted_at IS NULL
                                       WHERE po.deleted_at IS NULL
                                       GROUP BY po.id,po.payment_type,pt.payment_category
                                       HAVING debt>0 OR pt.payment_category = 'freewill'
                                       ORDER BY po.payment_type ASC";
                                $res_ob = mysqli_query($link, $sql_ob);
                                if ($res_ob) {
                                    while ($ob = mysqli_fetch_assoc($res_ob)) {
                                        $suffix = $ob['payment_category']==='freewill' ? ' (Freewill/Donation)' : '';
                                        echo "<option value='{$ob['id']}' data-payment-type='{$ob['payment_type']}' data-debt='{$ob['debt']}'>"
                                            .htmlspecialchars($ob['payment_type']).$suffix." (Remaining: ".$currency_symbol.number_format($ob['debt'],2).")</option>";
                                    }
                                    mysqli_free_result($res_ob);
                                }
                                ?>
                            </select>
                            <small class="form-text text-muted">Obligations are filtered based on selected members via AJAX.</small>
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label fw-bold text-success">Deposit To (Funding Source)</label>
                            <select name="bank_account_id" id="bank_account_id" class="form-select" required>
                                <option value="">Select Account (Cash or Bank)</option>
                                <?php foreach ($bank_accounts as $account): ?>
                                    <option value="<?= htmlspecialchars($account['id']) ?>">
                                        <?= htmlspecialchars($account['display_name']) ?>
                                    </option>
                                <?php endforeach; ?>
                                <?php if (empty($bank_accounts)): ?>
                                    <option value="" disabled>No public deposit accounts configured. Please contact Admin.</option>
                                <?php endif; ?>
                            </select>
                            <small class="form-text text-muted">Select the public bank or cash account where this income is being deposited.</small>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Payment Type</label>
                            <input type="text" name="payment_type_display" id="payment_type" class="form-control" readonly placeholder="Will auto-fill from selected obligation">
                            <input type="hidden" name="payment_type" id="payment_type_hidden">
                            <small class="form-text text-muted">This is the type associated with the selected obligation.</small>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Amount (<?=$currency_symbol?>)</label>
                            <input type="number" name="amount" id="amount_payment" class="form-control" step="0.01" min="0.01" required>
                            <small class="form-text text-muted">Enter the amount paid by *each* member. For non-freewill, this must not exceed the remaining debt.</small>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Description (optional)</label>
                            <textarea name="description" id="description_payment" class="form-control" rows="2"></textarea>
                        </div>

                        <button type="submit" class="btn btn-primary w-100 mt-3">
                            <i class="fas fa-plus-circle"></i> Log Payment
                        </button>
                    </form>
                </div>
            </div>
        </main>
    </div>
</div>

<script>
/**
 * Updates the Payment Type display field based on the selected obligation.
 */
function updatePaymentType() {
    const sel = document.getElementById('obligation_id');
    const ptDisplay  = document.getElementById('payment_type');
    const ptHidden  = document.getElementById('payment_type_hidden');
    const opt = sel.options[sel.selectedIndex];
    
    // Set both the display and the hidden input value
    if (opt && opt.dataset.paymentType) {
        ptDisplay.value = opt.dataset.paymentType;
        ptHidden.value = opt.dataset.paymentType;
    } else {
        ptDisplay.value = '';
        ptHidden.value = '';
    }
}

/**
 * Fetches and updates the list of available obligations based on the currently selected members.
 */
function updateObligations() {
    const userSel = document.getElementById('user_ids');
    const oblSel  = document.getElementById('obligation_id');
    const users   = Array.from(userSel.selectedOptions).map(o=>parseInt(o.value, 10)); // Ensure integers
    const currencySymbol = '<?= $currency_symbol ?>'; // Passed from PHP

    // Clear and reset the obligation dropdown
    oblSel.innerHTML='<option value="">Select an Obligation</option>';
    if (!users.length) {
        updatePaymentType();
        return;
    }
    
    // Add a loading indicator
    oblSel.innerHTML = '<option value="" disabled selected>Loading obligations...</option>';

    // Use fetch to call the API endpoint
    fetch('api/get_obligations.php',{
        method:'POST',
        headers:{'Content-Type':'application/json'},
        // Send selected user IDs to the API
        body:JSON.stringify({user_ids:users})
    })
    .then(r=>{
        if (!r.ok) throw new Error('Network response was not ok');
        return r.json();
    })
    .then(d=>{
        // Clear options and repopulate
        oblSel.innerHTML='<option value="">Select an Obligation</option>';
        if (d.obligations && d.obligations.length > 0) {
            d.obligations.forEach(o=>{
                const opt=document.createElement('option');
                opt.value=o.id;
                const suf=o.payment_category==='freewill'?' (Freewill/Donation)':'';
                // Handle potential null or undefined debt
                const debtAmount = (o.debt !== null && o.debt !== undefined) ? parseFloat(o.debt) : 0;
                const debtFormatted = debtAmount.toFixed(2);
                
                opt.text=`${o.payment_type}${suf} (Remaining: ${currencySymbol}${debtFormatted})`;
                opt.dataset.paymentType=o.payment_type;
                opt.dataset.debt=debtAmount;
                oblSel.appendChild(opt);
            });
        } else {
            oblSel.innerHTML='<option value="" disabled>No open obligations for these members.</option>';
        }
        
        // Update the payment type display after populating obligations
        updatePaymentType();
    })
    .catch(e=>{
        console.error("Error fetching obligations:", e);
        oblSel.innerHTML='<option value="" disabled>Error loading obligations. Try refreshing.</option>';
        updatePaymentType();
    });
}

// Attach event listener on DOMContentLoaded
document.addEventListener('DOMContentLoaded',()=>{ 
    const userIdsSelect = document.getElementById('user_ids');
    if(userIdsSelect) {
        userIdsSelect.addEventListener('change',updateObligations);
        // Initial call to populate obligations based on any default/previous selection (if any)
        updateObligations(); 
    }
});
</script>

<?php require_once 'includes/footer.php'; ?>