<?php
// record_expenses.php - Allows the Accountant or Admin to log club expenses, explicitly specifying the source (Imprest or Cheque/Bank).
ob_start();
session_start();
ini_set('display_errors',1);
ini_set('display_startup_errors',1);
error_reporting(E_ALL);
mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);
if (function_exists('opcache_reset')) { opcache_reset(); }

require_once 'includes/functions.php';
require_once 'includes/db_config.php';
require_once 'includes/ledger_functions.php'; // Required for double-entry system

/*
 * --- Management Functions (START) ---
 */

// --- BUG FIX: Removed local, broken versions of get_user_tokens() and deduct_tokens() ---
// The pages will now use the correct global functions from includes/functions.php

/**
 * Retrieves the total balance by querying the bank_accounts table.
 */
if (!function_exists('get_total_balance')) {
    function get_total_balance($link) {
        // --- UPDATED: Only sum balances from accounts the user can see ---
        $current_user_id = $_SESSION['user_id'] ?? 0;
        $current_user_role = $_SESSION['user_role'] ?? '';
        
        $sql = "SELECT SUM(current_balance) AS total_balance FROM bank_accounts WHERE is_active = 1 AND deleted_at IS NULL"; // Added deleted_at
        
        // --- *** SECURITY FIX *** ---
        // Apply owner logic if user is not a top-level admin
        if (!in_array($current_user_role, ['admin', 'super_admin'])) {
             // Accountants can ONLY see their own accounts OR the Cheque Holding account
             $sql .= " AND (owner_user_id = " . (int)$current_user_id . " OR account_name = 'Cheques Payable (Holding)')";
        }
        // --- *** END FIX *** ---
        
        $result = mysqli_query($link, $sql);
        if ($result && $row = mysqli_fetch_assoc($result)) {
            return (float)$row['total_balance'];
        }
        return 0.00;
    }
}

/**
 * Helper to get the name of an expense type.
 */
function get_expense_type_name_helper($link, $type_id) {
    $sql = "SELECT type_name FROM expense_types WHERE id = ? LIMIT 1";
    $stmt = mysqli_prepare($link, $sql);
    $name = 'Unknown Expense Type';
    if ($stmt) {
        mysqli_stmt_bind_param($stmt, "i", $type_id);
        mysqli_stmt_execute($stmt);
        $result = mysqli_stmt_get_result($stmt);
        if ($row = mysqli_fetch_assoc($result)) {
            $name = $row['type_name'];
        }
        mysqli_stmt_close($stmt);
    }
    return $name;
}

// --- Management Functions (END) ---


/* ---------- SESSION / PERMISSIONS ---------- */
$timeout = 1800;
if (isset($_SESSION['last_activity']) && (time() - $_SESSION['last_activity']) > $timeout) {
    session_unset(); session_destroy();
    redirect("login.php?timeout=1");
}
$_SESSION['last_activity'] = time();

if (!isset($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

// Only Accountant or Admin can log expenses
if (!is_logged_in() || !user_has_permission($link, 'view_expenses') ||
    !in_array($_SESSION['user_role'], ['admin','accountant', 'super_admin'])) { // Added super_admin
    redirect("login.php"); exit;
}

/* ---------- SETTINGS & DATA SETUP ---------- */
$currency_symbol = get_setting('currency_symbol',$link) ?? '₦';
$token_cost      = (int)get_setting('tokens_per_expense',$link) ?: 1;
$user_id         = $_SESSION['user_id'];
$user_role       = $_SESSION['user_role']; // <-- NEW: Get user role
$csrf_token      = $_SESSION['csrf_token'];

$message = '';
$error = '';
$expense_type_id = '';
$amount = '';
$description = '';
$fund_request_id = '';
$bank_account_id = '';

/* ---------- FETCH DATA: EXPENSE TYPES ---------- */
$expense_types = [];
$sql_types = "SELECT id, type_name FROM expense_types WHERE deleted_at IS NULL ORDER BY type_name ASC"; // Added deleted_at
$result_types = mysqli_query($link, $sql_types);
while ($row = mysqli_fetch_assoc($result_types)) {
    $expense_types[] = $row;
}

/* ---------- FETCH DATA: APPROVED IMPREST REQUESTS (For linking) ---------- */
// (This logic correctly fetches requests for the current user, so it's fine)
$approved_requests = [];
$imprest_type_id = 0;
// 1. Get the ID for 'Expense Imprest'
$sql_type = "SELECT id FROM request_types WHERE type_name = 'Expense Imprest' AND target_role = 'accountant' LIMIT 1";
$result_type = mysqli_query($link, $sql_type);
if ($result_type && mysqli_num_rows($result_type) > 0) {
    $imprest_type_id = mysqli_fetch_assoc($result_type)['id'];
}

if ($imprest_type_id > 0) {
    // 2. Fetch approved/partially_spent requests with a remaining balance
    $sql_approved = "SELECT 
                        id, amount, amount_spent, purpose 
                       FROM fund_requests 
                       WHERE requester_id = ? 
                         AND request_type_id = ? 
                         AND status IN ('approved', 'partially_spent') 
                         AND amount > amount_spent
                       ORDER BY id ASC";
    
    $stmt_approved = mysqli_prepare($link, $sql_approved);
    mysqli_stmt_bind_param($stmt_approved, "ii", $user_id, $imprest_type_id);
    mysqli_stmt_execute($stmt_approved);
    $result_approved = mysqli_stmt_get_result($stmt_approved);
    while ($row = mysqli_fetch_assoc($result_approved)) {
        $row['remaining_balance'] = $row['amount'] - $row['amount_spent'];
        $approved_requests[] = $row;
    }
    mysqli_stmt_close($stmt_approved);
}

/* ---------- FETCH BANK ACCOUNTS (PAYMENT SOURCES) - SECURED ---------- */
$bank_accounts = [];
// --- NEW: Secure query based on user role ---
$sql_banks = "SELECT id, account_name, bank_name, account_number, is_cash_on_hand, current_balance, owner_user_id 
              FROM bank_accounts 
              WHERE is_active = 1 AND deleted_at IS NULL"; // Added deleted_at

$params = [];
$types = "";

// --- *** SECURITY FIX *** ---
// If user is NOT an admin, restrict to ONLY their own imprest or the public Cheque Holding account
if (!in_array($user_role, ['admin', 'super_admin'])) {
    $sql_banks .= " AND (owner_user_id = ? OR account_name = 'Cheques Payable (Holding)')";
    $params[] = $user_id;
    $types .= "i";
}
// --- *** END FIX *** ---

$sql_banks .= " ORDER BY is_cash_on_hand DESC, bank_name ASC";

$stmt_banks = mysqli_prepare($link, $sql_banks);
if ($stmt_banks) {
    if (!empty($params)) {
        mysqli_stmt_bind_param($stmt_banks, $types, ...$params);
    }
    mysqli_stmt_execute($stmt_banks);
    $result_banks = mysqli_stmt_get_result($stmt_banks);
    
    while ($row = mysqli_fetch_assoc($result_banks)) {
        // Use 'bank_name' if 'account_name' is missing, fallback to 'Cash at Hand'
        $account_display = $row['account_name'] ?? $row['bank_name'];
        // Create a user-friendly display name for the dropdown
        // Updated display name for clarity
        $display_balance = $currency_symbol . number_format($row['current_balance'], 2);
        if ($row['is_cash_on_hand']) {
            $row['display_name'] = htmlspecialchars($account_display) . " (Cash/Imprest) - {$display_balance}";
        } else {
            $row['display_name'] = htmlspecialchars($account_display . ' (' . $row['bank_name'] . ') - ') . $display_balance;
        }
        $bank_accounts[] = $row;
    }
    mysqli_stmt_close($stmt_banks);
}
// --- END: Secure query ---


/* ---------- HANDLE FORM SUBMISSION ---------- */
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    
    // 1. CSRF Check
    if (!isset($_POST['csrf_token']) || !hash_equals($csrf_token, $_POST['csrf_token'])) {
        $error = "Security token mismatch. Please try again.";
    } else {
        // Collect and sanitize inputs
        $expense_type_id = isset($_POST['expense_type_id']) ? intval($_POST['expense_type_id']) : 0;
        $amount = isset($_POST['amount']) ? (float)sanitize_input($_POST['amount']) : 0.00; // Cast to float early
        $description = isset($_POST['description']) ? sanitize_input($_POST['description']) : null;
        $bank_account_id = isset($_POST['bank_account_id']) ? intval($_POST['bank_account_id']) : 0;
        $fund_request_id = isset($_POST['fund_request_id']) ? intval($_POST['fund_request_id']) : null; 

        // Basic validation
        if ($expense_type_id <= 0) {
            $error = "Please select a valid Expense Type.";
        } elseif ($bank_account_id <= 0) {
            $error = "Please select a valid account to pay from.";
        } elseif (!is_numeric($amount) || $amount <= 0) {
            $error = "Please enter a valid amount greater than zero.";
        }
        
        // --- NEW: Validate that the user is allowed to use this account ---
        if (empty($error)) {
            $is_allowed_account = false;
            $account_balance = 0.00;
            $is_imprest = false;
            
            // We check against the SECURE list of accounts we fetched earlier
            foreach ($bank_accounts as $allowed_acc) {
                if ($allowed_acc['id'] == $bank_account_id) {
                    $is_allowed_account = true;
                    $account_balance = (float)$allowed_acc['current_balance'];
                    $is_imprest = (bool)$allowed_acc['is_cash_on_hand'];
                    break;
                }
            }
            if (!$is_allowed_account) {
                 $error = "Invalid account selection. You do not have permission to use this account.";
            }
        }
        
        // Validate balance
        // *** LOGIC CORRECTION: Allow Cheque Holding account to go negative ***
        if (empty($error) && $amount > $account_balance && $is_imprest) {
            // Only enforce balance check for IMPREST accounts
            $error = "Insufficient balance in the selected Imprest account. Current balance is {$currency_symbol}" . number_format($account_balance, 2) . ".";
        }

        $link_to_fund_request = false;
        $new_spent_amount = 0.00;
        $new_status = '';
        $request_amount = 0.00;
        
        // 2. Validate Fund Request Link ONLY if account is imprest (cash)
        if (empty($error) && $is_imprest) {
            
            if ($fund_request_id <= 0) {
                $error = "You must link an Imprest expense to an approved Fund Request.";
            } else {
                // Re-fetch the selected request details for current balance validation
                $sql_check_fund = "SELECT amount, amount_spent FROM fund_requests 
                                   WHERE id = ? AND requester_id = ? AND status IN ('approved', 'partially_spent')";
                $stmt_check = mysqli_prepare($link, $sql_check_fund);
                mysqli_stmt_bind_param($stmt_check, "ii", $fund_request_id, $user_id);
                mysqli_stmt_execute($stmt_check);
                $result_check = mysqli_stmt_get_result($stmt_check);
                
                if ($row_fund = mysqli_fetch_assoc($result_check)) {
                    $request_amount = (float)$row_fund['amount'];
                    $amount_spent_current = (float)$row_fund['amount_spent'];
                    $remaining_balance = $request_amount - $amount_spent_current;
                    
                    if ($amount > $remaining_balance) {
                        $error = "The expense amount ({$currency_symbol}" . number_format($amount, 2) . ") exceeds the remaining balance of the selected fund request ({$currency_symbol}" . number_format($remaining_balance, 2) . ").";
                    } else {
                        $link_to_fund_request = true;
                        $new_spent_amount = $amount_spent_current + $amount;
                        $new_status = ($new_spent_amount >= $request_amount) ? 'spent' : 'partially_spent';
                    }
                } else {
                    $error = "The selected fund request is invalid, already fully spent, or not approved.";
                }
                mysqli_stmt_close($stmt_check);
            }
        } else {
            // If not imprest, ensure fund_request_id is NULL
            $fund_request_id = null;
        }

        /* 4. Process insertion with Transaction (The Core Logic Update) */
        if (empty($error)) {
            
            // Start the transaction
            mysqli_begin_transaction($link);
            $success = true;
            $expense_id = null;
            $final_message = '';
            
            // --- A. Insert the Expense Record ---
            $sql_insert_expense = "INSERT INTO expenses (expense_type_id, amount, description, fund_request_id) 
                                   VALUES (?, ?, ?, ?)";
            
            $stmt_insert_expense = mysqli_prepare($link, $sql_insert_expense);
            mysqli_stmt_bind_param($stmt_insert_expense, "idsi", $expense_type_id, $amount, $description, $fund_request_id);
            
            if (mysqli_stmt_execute($stmt_insert_expense)) {
                $expense_id = mysqli_insert_id($link);
                $final_message = "Expense of {$currency_symbol}" . number_format($amount, 2) . " successfully recorded and posted to the ledger.";
            } else {
                $success = false;
                $error = "Failed to log expense record: " . mysqli_error($link);
            }
            mysqli_stmt_close($stmt_insert_expense);
            
            if ($success) {
                // Fetch expense type name for ledger logging
                $expense_type_name = get_expense_type_name_helper($link, $expense_type_id);
                $ledger_description = "Expense: '{$expense_type_name}' by User ID: {$user_id} (Ref Expense ID: {$expense_id})" . ($description ? " - {$description}" : "");
                
                // --- B. Update Fund Request if linked (Imprest) ---
                if ($link_to_fund_request) {
                    $sql_update_fund = "UPDATE fund_requests SET amount_spent = ?, status = ? WHERE id = ?";
                    $stmt_update_fund = mysqli_prepare($link, $sql_update_fund);
                    mysqli_stmt_bind_param($stmt_update_fund, "dsi", $new_spent_amount, $new_status, $fund_request_id);
                    if (!mysqli_stmt_execute($stmt_update_fund)) {
                        $success = false;
                        $error = "Failed to update linked Fund Request.";
                    }
                    mysqli_stmt_close($stmt_update_fund);
                    $final_message .= " Linked to Fund Request #{$fund_request_id}. Status updated to " . ucfirst(str_replace('_', ' ', $new_status)) . ".";
                } 
                
                // --- C. Update Asset Account Balance (Credit the Asset) ---
                if ($success) {
                    $sql_update_balance = "UPDATE bank_accounts SET current_balance = current_balance - ? WHERE id = ?";
                    $stmt_update_balance = mysqli_prepare($link, $sql_update_balance);
                    mysqli_stmt_bind_param($stmt_update_balance, "di", $amount, $bank_account_id);
                    
                    if (!mysqli_stmt_execute($stmt_update_balance)) {
                        $success = false;
                        $error = "Failed to update asset balance for account ID {$bank_account_id}.";
                    }
                    mysqli_stmt_close($stmt_update_balance);
                }


                // --- D. Ledger Logging (Single Entry into financial_ledger) ---
                if ($success) {
                    $transaction_type = 'expense';
                    $reference_id = $expense_id;
                    $recorded_by_id = $user_id;

                    $sql_ledger = "INSERT INTO financial_ledger 
                        (bank_account_id, transaction_type, amount, description, reference_id, transaction_date, recorded_by_id) 
                        VALUES (?, ?, ?, ?, ?, NOW(), ?)";
                    $stmt_ledger = mysqli_prepare($link, $sql_ledger);
                    mysqli_stmt_bind_param($stmt_ledger, "isdsii", $bank_account_id, $transaction_type, $amount, $ledger_description, $reference_id, $recorded_by_id);
                    
                    if (!mysqli_stmt_execute($stmt_ledger)) {
                        $success = false;
                        $error = "Failed to log ledger entry: " . mysqli_error($link);
                    }
                    mysqli_stmt_close($stmt_ledger);
                }
            }


            // --- E. Commit/Rollback/Token Deduction ---
            if ($success) {
                // Attempt token deduction (using global function)
                if (deduct_tokens($link, $user_id, 'expense', $token_cost)) {
                    mysqli_commit($link);
                    $message = $final_message . " Your token balance was reduced by {$token_cost}.";
                    
                    // Clear form data on successful commit
                    $expense_type_id = '';
                    $amount = '';
                    $description = '';
                    $fund_request_id = '';
                    $bank_account_id = '';
                    
                } else {
                    // Token failure -> rollback everything
                    mysqli_rollback($link);
                    $error = "Transaction failed: Core expense logged, but token deduction failed. All financial changes rolled back. Please contact support.";
                }
            } else {
                // Financial operation failure -> rollback everything
                mysqli_rollback($link);
                if (empty($error)) {
                     $error = "An unknown error occurred during transaction processing. Changes rolled back.";
                }
            }
        }
    }
    // Regenerate CSRF token
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    $csrf_token = $_SESSION['csrf_token'];
}

/* ---------- DISPLAY USER TOKENS AND BALANCE ---------- */
// --- BUG FIX: Call the correct global function, just like the income pages do ---
$user_tokens = get_user_token_balance($link, $user_id); 
// get_total_balance() is now secure and context-aware
$total_balance = get_total_balance($link); 
?>

<?php require_once 'includes/header.php'; ?>
<script>document.title='Record Expense - <?=htmlspecialchars(get_setting('system_name',$link)??'Club Portal')?>';</script>
<style>.main-content{padding:20px;}</style>

<div class="container-fluid">
    <div class="row">
        <?php 
        require_once 'includes/sidebar.php'; 
        ?>
        
        <main class="col-md-9 col-lg-10 main-content"> 
            <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                <h1 class="h2">Record Club Expense</h1>
            </div>

            <?php if (!empty($error)): ?>
                <div class="alert alert-danger"><?=htmlspecialchars($error)?></div>
            <?php endif; ?>
            <?php if (!empty($message)): ?>
                <div class="alert alert-success"><?=htmlspecialchars($message)?></div>
            <?php endif; ?>
            
            <div class="alert alert-info shadow-sm">
                You currently have **<?=htmlspecialchars($user_tokens)?>** tokens. Logging an expense costs **<?=$token_cost?>** token(s).<br>
                <!-- This balance is now filtered to what you can see/use -->
                <strong>Your Accessible Balance Total:</strong> <?= $currency_symbol . number_format($total_balance, 2) ?>
                <?php if ($user_tokens < $token_cost): ?>
                    <p class="mb-0 mt-2 text-danger fw-bold">You need more tokens to log an expense.</p>
                <?php endif; ?>
            </div>

            <!-- Expense Logging Form -->
            <div class="card shadow-sm mb-5">
                <div class="card-header bg-primary text-white">
                    Log New Expense
                </div>
                <div class="card-body">
                    <form method="POST" action="record_expenses.php">
                        <input type="hidden" name="csrf_token" value="<?=htmlspecialchars($csrf_token)?>">
                        
                        <!-- Paid From Account Selection -->
                        <div class="mb-3">
                            <label class="form-label fw-bold">Paid From Account</label>
                            <select name="bank_account_id" id="bank_account_id" class="form-select" required>
                                <option value="">Select Account (Cash or Cheque)</option> <!-- Updated help text -->
                                <?php foreach ($bank_accounts as $account): ?>
                                    <option value="<?= htmlspecialchars($account['id']) ?>" data-is-cash-on-hand="<?= htmlspecialchars($account['is_cash_on_hand']) ?>">
                                        <?= htmlspecialchars($account['display_name']) ?>
                                    </option>
                                <?php endforeach; ?>
                                <?php if (empty($bank_accounts)): ?>
                                    <option value="" disabled>No accounts (or Imprest) assigned to you. Please contact Admin.</option>
                                <?php endif; ?>
                            </select>
                            <div class="form-text">Select your Imprest account (for cash) or the Cheques Payable account (for cheques).</div>
                        </div>


                        <!-- Fund Request Link Dropdown (Conditional for Cash/Imprest) -->
                        <div id="imprest_link_container" 
                             class="mb-3 p-3 border rounded bg-light"
                             style="display: none;">
                             
                            <label class="form-label fw-bold">Link to Approved Imprest Request</label>
                            
                            <?php if (!empty($approved_requests)): ?>
                                <select name="fund_request_id" id="fund_request_id" class="form-select">
                                    <option value="">-- Select Approved Fund Request --</option>
                                    <?php foreach ($approved_requests as $r): ?>
                                        <option value="<?=htmlspecialchars($r['id'])?>"
                                            <?=($fund_request_id == $r['id']) ? 'selected' : ''?>
                                        >
                                            #<?=htmlspecialchars($r['id'])?> - <?=htmlspecialchars($r['purpose'])?> 
                                            (Remaining: <?=$currency_symbol . number_format($r['remaining_balance'], 2)?>)
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                                <div class="form-text">Required for Imprest expenses.</div>
                            <?php else: ?>
                                <!-- Message when Imprest is selected but no funds are available -->
                                <div class="alert alert-warning mb-0">No approved, unspent Imprest funds are currently available to link expenses to.</div>
                                <input type="hidden" name="fund_request_id" value="">
                            <?php endif; ?>
                        </div>


                        <!-- Expense Details (Always Visible) -->
                        <div class="mb-3">
                            <label class="form-label">Expense Type</label>
                            <select name="expense_type_id" class="form-select" required>
                                <option value="">Select Type</option>
                                <?php foreach ($expense_types as $t): ?>
                                    <option value="<?=htmlspecialchars($t['id'])?>"
                                        <?=($expense_type_id == $t['id']) ? 'selected' : ''?>
                                    ><?=htmlspecialchars($t['type_name'])?></option>
                                <?php endforeach; ?>
                            </select>
                            <div class="form-text">If the expense is for a **Benefit**, select the appropriate 'Benefit' expense type here.</div>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Amount (<?=$currency_symbol?>)</label>
                            <input type="number" name="amount" class="form-control" step="0.01" min="0.01" required value="<?=htmlspecialchars($amount)?>">
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Description (optional)</label>
                            <textarea name="description" class="form-control" rows="3"><?=htmlspecialchars($description)?></textarea>
                        </div>

                        <button type="submit" class="btn btn-primary"
                            <?php if ($user_tokens < $token_cost): ?>disabled<?php endif; ?>>
                            Log Expense (<?=$token_cost?> token<?=$token_cost>1?'s':''?>)
                        </button>
                    </form>
                </div>
            </div>
        </main>
    </div>
</div>

<script>
    // JAVASCRIPT FOR CONDITIONAL DISPLAY
    document.addEventListener('DOMContentLoaded', function() {
        const accountSelect = document.getElementById('bank_account_id');
        const imprestContainer = document.getElementById('imprest_link_container');
        const fundRequestSelect = document.getElementById('fund_request_id');
        
        function toggleImprestFields() {
            const selectedOption = accountSelect.options[accountSelect.selectedIndex];
            // Check if the data-is-cash-on-hand attribute is '1'
            const isImprest = selectedOption && selectedOption.dataset.isCashOnHand === '1';
            
            if (isImprest) {
                imprestContainer.style.display = 'block';
                // Only try to set 'required' if the select element actually exists (it won't if the PHP rendered the warning message)
                if (fundRequestSelect && fundRequestSelect.tagName === 'SELECT') {
                    fundRequestSelect.setAttribute('required', 'required');
                }
            } else {
                imprestContainer.style.display = 'none';
                // Remove required attribute when hidden
                if (fundRequestSelect && fundRequestSelect.tagName === 'SELECT') {
                    fundRequestSelect.removeAttribute('required');
                }
            }
        }

        // Attach event listener to the source selection
        accountSelect.addEventListener('change', toggleImprestFields);

        // Initial check on load (for cases where the form submission failed and reloads with data)
        toggleImprestFields();
    });
</script>

<?php require_once 'includes/footer.php'; ?>
<?php ob_end_flush(); ?>