<?php
// modify_records.php
ob_start();
session_start();

require_once 'includes/functions.php';
require_once 'includes/db_config.php';

// Check if user is logged in and has necessary permissions
if (!is_logged_in() || !(
    user_has_permission($link, 'view_payments') ||
    user_has_permission($link, 'view_expenses') ||
    user_has_permission($link, 'view_benefits')
)) {
    redirect("login.php");
    exit;
}

$currency_symbol = get_setting('currency_symbol', $link) ?? '₦';
$message = '';
$message_type = '';

// Handle form submissions
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    try {
        // Edit Payment
        if (isset($_POST['edit_payment'])) {
            $payment_id = intval($_POST['payment_id']);
            $amount = floatval($_POST['amount'] ?? 0);
            $description = sanitize_input($_POST['description'] ?? '');
            $payment_status = sanitize_input($_POST['payment_status'] ?? '');
            $payment_date = sanitize_input($_POST['payment_date'] ?? '');

            // Validate date format (YYYY-MM-DD)
            if ($amount <= 0 || empty($payment_status) || empty($payment_date) || !preg_match('/^\d{4}-\d{2}-\d{2}$/', $payment_date) || !strtotime($payment_date)) {
                $message = "Invalid input. Amount, status, and a valid payment date (YYYY-MM-DD) are required.";
                $message_type = 'danger';
            } else {
                $sql = "UPDATE payments SET amount = ?, description = ?, payment_status = ?, payment_date = ? WHERE id = ?";
                if ($stmt = mysqli_prepare($link, $sql)) {
                    mysqli_stmt_bind_param($stmt, "dsssi", $amount, $description, $payment_status, $payment_date, $payment_id);
                    if (mysqli_stmt_execute($stmt)) {
                        $message = "Payment updated successfully!";
                        $message_type = 'success';
                    } else {
                        throw new Exception("Error updating payment: " . mysqli_error($link));
                    }
                    mysqli_stmt_close($stmt);
                } else {
                    throw new Exception("Failed to prepare statement for payment update.");
                }
            }
        }
        // Delete Payment
        elseif (isset($_POST['delete_payment'])) {
            $payment_id = intval($_POST['payment_id']);
            $sql = "DELETE FROM payments WHERE id = ?";
            if ($stmt = mysqli_prepare($link, $sql)) {
                mysqli_stmt_bind_param($stmt, "i", $payment_id);
                if (mysqli_stmt_execute($stmt)) {
                    $message = "Payment deleted successfully!";
                    $message_type = 'success';
                } else {
                    throw new Exception("Error deleting payment: " . mysqli_error($link));
                }
                mysqli_stmt_close($stmt);
            } else {
                throw new Exception("Failed to prepare statement for payment deletion.");
            }
        }
        // Edit Expense
        elseif (isset($_POST['edit_expense'])) {
            $expense_id = intval($_POST['expense_id']);
            $amount = floatval($_POST['amount'] ?? 0);
            $expense_type_id = intval($_POST['expense_type_id'] ?? 0);
            $description = sanitize_input($_POST['description'] ?? '');
            $created_at = sanitize_input($_POST['created_at'] ?? '');

            if ($amount <= 0 || $expense_type_id <= 0 || empty($description) || empty($created_at) || !preg_match('/^\d{4}-\d{2}-\d{2}$/', $created_at) || !strtotime($created_at)) {
                $message = "Invalid input. All fields are required, and date must be valid (YYYY-MM-DD).";
                $message_type = 'danger';
            } else {
                $sql = "UPDATE expenses SET amount = ?, expense_type_id = ?, description = ?, created_at = ? WHERE id = ?";
                if ($stmt = mysqli_prepare($link, $sql)) {
                    mysqli_stmt_bind_param($stmt, "dissi", $amount, $expense_type_id, $description, $created_at, $expense_id);
                    if (mysqli_stmt_execute($stmt)) {
                        $message = "Expense updated successfully!";
                        $message_type = 'success';
                    } else {
                        throw new Exception("Error updating expense: " . mysqli_error($link));
                    }
                    mysqli_stmt_close($stmt);
                } else {
                    throw new Exception("Failed to prepare statement for expense update.");
                }
            }
        }
        // Delete Expense
        elseif (isset($_POST['delete_expense'])) {
            $expense_id = intval($_POST['expense_id']);
            $sql = "DELETE FROM expenses WHERE id = ?";
            if ($stmt = mysqli_prepare($link, $sql)) {
                mysqli_stmt_bind_param($stmt, "i", $expense_id);
                if (mysqli_stmt_execute($stmt)) {
                    $message = "Expense deleted successfully!";
                    $message_type = 'success';
                } else {
                    throw new Exception("Error deleting expense: " . mysqli_error($link));
                }
                mysqli_stmt_close($stmt);
            } else {
                throw new Exception("Failed to prepare statement for expense deletion.");
            }
        }
        // Edit Benefit
        elseif (isset($_POST['edit_benefit'])) {
            $benefit_id = intval($_POST['benefit_id']);
            $member_id = intval($_POST['member_id'] ?? 0);
            $benefit_type_id = intval($_POST['benefit_type_id'] ?? 0);
            $amount = floatval($_POST['amount'] ?? 0);
            $benefit_date = sanitize_input($_POST['benefit_date'] ?? '');
            $benefit_status = sanitize_input($_POST['benefit_status'] ?? '');

            if ($member_id <= 0 || $benefit_type_id <= 0 || $amount <= 0 || empty($benefit_date) || empty($benefit_status) || !preg_match('/^\d{4}-\d{2}-\d{2}$/', $benefit_date) || !strtotime($benefit_date)) {
                $message = "Invalid input. All fields are required, and date must be valid (YYYY-MM-DD).";
                $message_type = 'danger';
            } else {
                $sql = "UPDATE member_benefits SET member_id = ?, benefit_type_id = ?, amount = ?, benefit_date = ?, benefit_status = ? WHERE id = ?";
                if ($stmt = mysqli_prepare($link, $sql)) {
                    mysqli_stmt_bind_param($stmt, "iisdsi", $member_id, $benefit_type_id, $amount, $benefit_date, $benefit_status, $benefit_id);
                    if (mysqli_stmt_execute($stmt)) {
                        $message = "Benefit updated successfully!";
                        $message_type = 'success';
                    } else {
                        throw new Exception("Error updating benefit: " . mysqli_error($link));
                    }
                    mysqli_stmt_close($stmt);
                } else {
                    throw new Exception("Failed to prepare statement for benefit update.");
                }
            }
        }
        // Delete Benefit
        elseif (isset($_POST['delete_benefit'])) {
            $benefit_id = intval($_POST['benefit_id']);
            $sql = "DELETE FROM member_benefits WHERE id = ?";
            if ($stmt = mysqli_prepare($link, $sql)) {
                mysqli_stmt_bind_param($stmt, "i", $benefit_id);
                if (mysqli_stmt_execute($stmt)) {
                    $message = "Benefit deleted successfully!";
                    $message_type = 'success';
                } else {
                    throw new Exception("Error deleting benefit: " . mysqli_error($link));
                }
                mysqli_stmt_close($stmt);
            } else {
                throw new Exception("Failed to prepare statement for benefit deletion.");
            }
        }
    } catch (Exception $e) {
        $message = "An error occurred: " . $e->getMessage();
        $message_type = 'danger';
    }
}

// Fetch data for tables
$payments = [];
if (user_has_permission($link, 'view_payments')) {
    $sql = "SELECT p.id, u.full_name, pt.type_name AS payment_type, p.amount, p.description, p.payment_status, p.payment_date
            FROM payments p
            LEFT JOIN users u ON p.user_id = u.id
            LEFT JOIN payment_types pt ON p.payment_type = pt.type_name
            WHERE pt.deleted_at IS NULL
            ORDER BY p.payment_date DESC";
    $result = mysqli_query($link, $sql);
    if ($result) {
        while ($row = mysqli_fetch_assoc($result)) {
            $payments[] = $row;
        }
    } else {
        $message = "Error fetching payments: " . mysqli_error($link);
        $message_type = 'danger';
    }
}

$expenses = [];
if (user_has_permission($link, 'view_expenses')) {
    $sql = "SELECT e.id, et.type_name AS expense_type, e.amount, e.description, e.created_at
            FROM expenses e
            JOIN expense_types et ON e.expense_type_id = et.id
            WHERE et.deleted_at IS NULL
            ORDER BY e.created_at DESC";
    $result = mysqli_query($link, $sql);
    if ($result) {
        while ($row = mysqli_fetch_assoc($result)) {
            $expenses[] = $row;
        }
    } else {
        $message = "Error fetching expenses: " . mysqli_error($link);
        $message_type = 'danger';
    }
}

$benefits = [];
if (user_has_permission($link, 'view_benefits')) {
    $sql = "SELECT mb.id, u.full_name, bt.type_name AS benefit_type, mb.amount, mb.benefit_date, mb.benefit_status
            FROM member_benefits mb
            JOIN users u ON mb.member_id = u.id
            JOIN benefit_types bt ON mb.benefit_type_id = bt.id
            WHERE bt.deleted_at IS NULL
            ORDER BY mb.benefit_date DESC";
    $result = mysqli_query($link, $sql);
    if ($result) {
        while ($row = mysqli_fetch_assoc($result)) {
            $benefits[] = $row;
        }
    } else {
        $message = "Error fetching benefits: " . mysqli_error($link);
        $message_type = 'danger';
    }
}

// Fetch data for dropdowns in edit forms
$expense_types = [];
$sql = "SELECT id, type_name FROM expense_types WHERE deleted_at IS NULL ORDER BY type_name";
$result = mysqli_query($link, $sql);
if ($result) {
    while ($row = mysqli_fetch_assoc($result)) {
        $expense_types[] = $row;
    }
} else {
    $message = "Error fetching expense types: " . mysqli_error($link);
    $message_type = 'danger';
}

$benefit_types = [];
$sql = "SELECT id, type_name FROM benefit_types WHERE deleted_at IS NULL ORDER BY type_name";
$result = mysqli_query($link, $sql);
if ($result) {
    while ($row = mysqli_fetch_assoc($result)) {
        $benefit_types[] = $row;
    }
} else {
    $message = "Error fetching benefit types: " . mysqli_error($link);
    $message_type = 'danger';
}

$members = [];
$sql = "SELECT id, full_name FROM users WHERE role = 'member' AND (deleted_at IS NULL OR deleted_at > NOW()) ORDER BY full_name";
$result = mysqli_query($link, $sql);
if ($result) {
    while ($row = mysqli_fetch_assoc($result)) {
        $members[] = $row;
    }
} else {
    $message = "Error fetching members: " . mysqli_error($link);
    $message_type = 'danger';
}

ob_end_clean();
?>

<?php require_once 'includes/header.php'; ?>

<script>
// Set dynamic page title
document.title = 'Modify Financial Records - <?php echo htmlspecialchars(get_setting('system_name', $link) ?? 'Club Portal'); ?>';
</script>

<div class="container-fluid">
    <div class="row">
        <?php require_once 'includes/sidebar.php'; ?>
        <main class="col-md-9 col-lg-10 main-content">
            <h2>Modify Financial Records</h2>

            <?php if ($message): ?>
                <div class="alert alert-<?php echo htmlspecialchars($message_type); ?>" role="alert">
                    <?php echo htmlspecialchars($message); ?>
                </div>
            <?php endif; ?>

            <ul class="nav nav-tabs mb-4" id="recordTabs" role="tablist">
                <?php if (user_has_permission($link, 'view_payments')): ?>
                    <li class="nav-item">
                        <a class="nav-link active" id="payments-tab" data-bs-toggle="tab" href="#payments" role="tab" aria-controls="payments" aria-selected="true">Payments</a>
                    </li>
                <?php endif; ?>
                <?php if (user_has_permission($link, 'view_expenses')): ?>
                    <li class="nav-item">
                        <a class="nav-link" id="expenses-tab" data-bs-toggle="tab" href="#expenses" role="tab" aria-controls="expenses" aria-selected="false">Expenses</a>
                    </li>
                <?php endif; ?>
                <?php if (user_has_permission($link, 'view_benefits')): ?>
                    <li class="nav-item">
                        <a class="nav-link" id="benefits-tab" data-bs-toggle="tab" href="#benefits" role="tab" aria-controls="benefits" aria-selected="false">Benefits</a>
                    </li>
                <?php endif; ?>
            </ul>

            <div class="tab-content" id="recordTabsContent">
                <?php if (user_has_permission($link, 'view_payments')): ?>
                    <div class="tab-pane fade show active" id="payments" role="tabpanel" aria-labelledby="payments-tab">
                        <div class="card mb-4">
                            <div class="card-header">Payment Records</div>
                            <div class="card-body">
                                <div class="table-responsive">
                                    <table class="table table-striped table-hover">
                                        <thead>
                                            <tr>
                                                <th>Member Name</th>
                                                <th>Payment Type</th>
                                                <th>Amount (<?php echo htmlspecialchars($currency_symbol); ?>)</th>
                                                <th>Description</th>
                                                <th>Status</th>
                                                <th>Payment Date</th>
                                                <th>Action</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php if (empty($payments)): ?>
                                                <tr><td colspan="7" class="text-center">No payment records found.</td></tr>
                                            <?php else: ?>
                                                <?php foreach ($payments as $payment): ?>
                                                    <tr>
                                                        <td><?php echo htmlspecialchars($payment['full_name'] ?? 'N/A'); ?></td>
                                                        <td><?php echo htmlspecialchars($payment['payment_type'] ?? 'N/A'); ?></td>
                                                        <td><?php echo htmlspecialchars(format_currency($payment['amount'], $currency_symbol)); ?></td>
                                                        <td><?php echo htmlspecialchars(truncate_text($payment['description'] ?? '', 50)); ?></td>
                                                        <td><span class="badge bg-<?php echo ($payment['payment_status'] === 'Paid') ? 'success' : 'danger'; ?>">
                                                            <?php echo htmlspecialchars($payment['payment_status']); ?>
                                                        </span></td>
                                                        <td><?php
                                                            // Handle invalid or NULL payment_date
                                                            if (empty($payment['payment_date']) || $payment['payment_date'] === '0000-00-00' || $payment['payment_date'] === '0000-00-00 00:00:00' || !strtotime($payment['payment_date'])) {
                                                                echo 'N/A';
                                                            } else {
                                                                echo htmlspecialchars(date('Y-m-d', strtotime($payment['payment_date'])));
                                                            }
                                                        ?></td>
                                                        <td>
                                                            <button class="btn btn-primary btn-sm" data-bs-toggle="modal" data-bs-target="#editPaymentModal<?php echo htmlspecialchars($payment['id']); ?>">Edit</button>
                                                            <form action="modify_records.php" method="post" style="display:inline;">
                                                                <input type="hidden" name="payment_id" value="<?php echo htmlspecialchars($payment['id']); ?>">
                                                                <button type="submit" name="delete_payment" class="btn btn-danger btn-sm" onclick="return confirm('Are you sure? This action cannot be undone.');">Delete</button>
                                                            </form>
                                                        </td>
                                                    </tr>
                                                <?php endforeach; ?>
                                            <?php endif; ?>
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        </div>
                    </div>
                <?php endif; ?>
                <?php if (user_has_permission($link, 'view_expenses')): ?>
                    <div class="tab-pane fade" id="expenses" role="tabpanel" aria-labelledby="expenses-tab">
                        <div class="card mb-4">
                            <div class="card-header">Expense Records</div>
                            <div class="card-body">
                                <div class="table-responsive">
                                    <table class="table table-striped table-hover">
                                        <thead>
                                            <tr>
                                                <th>Expense Type</th>
                                                <th>Amount (<?php echo htmlspecialchars($currency_symbol); ?>)</th>
                                                <th>Description</th>
                                                <th>Date</th>
                                                <th>Action</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php if (empty($expenses)): ?>
                                                <tr><td colspan="5" class="text-center">No expense records found.</td></tr>
                                            <?php else: ?>
                                                <?php foreach ($expenses as $expense): ?>
                                                    <tr>
                                                        <td><?php echo htmlspecialchars($expense['expense_type'] ?? 'N/A'); ?></td>
                                                        <td><?php echo htmlspecialchars(format_currency($expense['amount'], $currency_symbol)); ?></td>
                                                        <td><?php echo htmlspecialchars(truncate_text($expense['description'] ?? '', 50)); ?></td>
                                                        <td><?php
                                                            // Handle invalid or NULL created_at
                                                            if (empty($expense['created_at']) || $expense['created_at'] === '0000-00-00' || $expense['created_at'] === '0000-00-00 00:00:00' || !strtotime($expense['created_at'])) {
                                                                echo 'N/A';
                                                            } else {
                                                                echo htmlspecialchars(date('Y-m-d', strtotime($expense['created_at'])));
                                                            }
                                                        ?></td>
                                                        <td>
                                                            <button class="btn btn-primary btn-sm" data-bs-toggle="modal" data-bs-target="#editExpenseModal<?php echo htmlspecialchars($expense['id']); ?>">Edit</button>
                                                            <form action="modify_records.php" method="post" style="display:inline;">
                                                                <input type="hidden" name="expense_id" value="<?php echo htmlspecialchars($expense['id']); ?>">
                                                                <button type="submit" name="delete_expense" class="btn btn-danger btn-sm" onclick="return confirm('Are you sure? This action cannot be undone.');">Delete</button>
                                                            </form>
                                                        </td>
                                                    </tr>
                                                <?php endforeach; ?>
                                            <?php endif; ?>
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        </div>
                    </div>
                <?php endif; ?>
                <?php if (user_has_permission($link, 'view_benefits')): ?>
                    <div class="tab-pane fade" id="benefits" role="tabpanel" aria-labelledby="benefits-tab">
                        <div class="card mb-4">
                            <div class="card-header">Benefit Records</div>
                            <div class="card-body">
                                <div class="table-responsive">
                                    <table class="table table-striped table-hover">
                                        <thead>
                                            <tr>
                                                <th>Member Name</th>
                                                <th>Benefit Type</th>
                                                <th>Amount (<?php echo htmlspecialchars($currency_symbol); ?>)</th>
                                                <th>Date</th>
                                                <th>Status</th>
                                                <th>Action</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php if (empty($benefits)): ?>
                                                <tr><td colspan="6" class="text-center">No benefit records found.</td></tr>
                                            <?php else: ?>
                                                <?php foreach ($benefits as $benefit): ?>
                                                    <tr>
                                                        <td><?php echo htmlspecialchars($benefit['full_name'] ?? 'N/A'); ?></td>
                                                        <td><?php echo htmlspecialchars($benefit['benefit_type'] ?? 'N/A'); ?></td>
                                                        <td><?php echo htmlspecialchars(format_currency($benefit['amount'], $currency_symbol)); ?></td>
                                                        <td><?php
                                                            // Handle invalid or NULL benefit_date
                                                            if (empty($benefit['benefit_date']) || $benefit['benefit_date'] === '0000-00-00' || $benefit['benefit_date'] === '0000-00-00 00:00:00' || !strtotime($benefit['benefit_date'])) {
                                                                echo 'N/A';
                                                            } else {
                                                                echo htmlspecialchars(date('Y-m-d', strtotime($benefit['benefit_date'])));
                                                            }
                                                        ?></td>
                                                        <td><span class="badge bg-<?php echo ($benefit['benefit_status'] === 'Received') ? 'success' : ($benefit['benefit_status'] === 'Rejected' ? 'danger' : 'warning'); ?>">
                                                            <?php echo htmlspecialchars($benefit['benefit_status']); ?>
                                                        </span></td>
                                                        <td>
                                                            <button class="btn btn-primary btn-sm" data-bs-toggle="modal" data-bs-target="#editBenefitModal<?php echo htmlspecialchars($benefit['id']); ?>">Edit</button>
                                                            <form action="modify_records.php" method="post" style="display:inline;">
                                                                <input type="hidden" name="benefit_id" value="<?php echo htmlspecialchars($benefit['id']); ?>">
                                                                <button type="submit" name="delete_benefit" class="btn btn-danger btn-sm" onclick="return confirm('Are you sure? This action cannot be undone.');">Delete</button>
                                                            </form>
                                                        </td>
                                                    </tr>
                                                <?php endforeach; ?>
                                            <?php endif; ?>
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        </div>
                    </div>
                <?php endif; ?>
            </div>

            <!-- Edit Payment Modals -->
            <?php foreach ($payments as $payment): ?>
                <div class="modal fade" id="editPaymentModal<?php echo htmlspecialchars($payment['id']); ?>" tabindex="-1" aria-labelledby="editPaymentModalLabel<?php echo htmlspecialchars($payment['id']); ?>" aria-hidden="true">
                    <div class="modal-dialog">
                        <div class="modal-content">
                            <div class="modal-header">
                                <h5 class="modal-title" id="editPaymentModalLabel<?php echo htmlspecialchars($payment['id']); ?>">Edit Payment</h5>
                                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                            </div>
                            <form action="modify_records.php" method="post">
                                <div class="modal-body">
                                    <input type="hidden" name="edit_payment" value="1">
                                    <input type="hidden" name="payment_id" value="<?php echo htmlspecialchars($payment['id']); ?>">
                                    <div class="mb-3">
                                        <label for="amount_payment_<?php echo htmlspecialchars($payment['id']); ?>" class="form-label">Amount (<?php echo htmlspecialchars($currency_symbol); ?>)</label>
                                        <input type="number" class="form-control" id="amount_payment_<?php echo htmlspecialchars($payment['id']); ?>" name="amount" value="<?php echo htmlspecialchars($payment['amount']); ?>" step="0.01" min="0.01" required>
                                    </div>
                                    <div class="mb-3">
                                        <label for="description_payment_<?php echo htmlspecialchars($payment['id']); ?>" class="form-label">Description</label>
                                        <textarea class="form-control" id="description_payment_<?php echo htmlspecialchars($payment['id']); ?>" name="description" rows="3"><?php echo htmlspecialchars($payment['description'] ?? ''); ?></textarea>
                                    </div>
                                    <div class="mb-3">
                                        <label for="payment_status_<?php echo htmlspecialchars($payment['id']); ?>" class="form-label">Status</label>
                                        <select class="form-select" id="payment_status_<?php echo htmlspecialchars($payment['id']); ?>" name="payment_status" required>
                                            <option value="Paid" <?php echo $payment['payment_status'] === 'Paid' ? 'selected' : ''; ?>>Paid</option>
                                            <option value="Pending" <?php echo $payment['payment_status'] === 'Pending' ? 'selected' : ''; ?>>Pending</option>
                                        </select>
                                    </div>
                                    <div class="mb-3">
                                        <label for="payment_date_<?php echo htmlspecialchars($payment['id']); ?>" class="form-label">Payment Date</label>
                                        <input type="date" class="form-control" id="payment_date_<?php echo htmlspecialchars($payment['id']); ?>" name="payment_date" value="<?php
                                            // Handle invalid or NULL payment_date for input
                                            if (!empty($payment['payment_date']) && $payment['payment_date'] !== '0000-00-00' && $payment['payment_date'] !== '0000-00-00 00:00:00' && strtotime($payment['payment_date'])) {
                                                echo htmlspecialchars(date('Y-m-d', strtotime($payment['payment_date'])));
                                            }
                                        ?>" required>
                                    </div>
                                </div>
                                <div class="modal-footer">
                                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                                    <button type="submit" class="btn btn-primary">Save Changes</button>
                                </div>
                            </form>
                        </div>
                    </div>
                </div>
            <?php endforeach; ?>

            <!-- Edit Expense Modals -->
            <?php foreach ($expenses as $expense): ?>
                <div class="modal fade" id="editExpenseModal<?php echo htmlspecialchars($expense['id']); ?>" tabindex="-1" aria-labelledby="editExpenseModalLabel<?php echo htmlspecialchars($expense['id']); ?>" aria-hidden="true">
                    <div class="modal-dialog">
                        <div class="modal-content">
                            <div class="modal-header">
                                <h5 class="modal-title" id="editExpenseModalLabel<?php echo htmlspecialchars($expense['id']); ?>">Edit Expense</h5>
                                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                            </div>
                            <form action="modify_records.php" method="post">
                                <div class="modal-body">
                                    <input type="hidden" name="edit_expense" value="1">
                                    <input type="hidden" name="expense_id" value="<?php echo htmlspecialchars($expense['id']); ?>">
                                    <div class="mb-3">
                                        <label for="expense_type_id_<?php echo htmlspecialchars($expense['id']); ?>" class="form-label">Expense Type</label>
                                        <select class="form-select" id="expense_type_id_<?php echo htmlspecialchars($expense['id']); ?>" name="expense_type_id" required>
                                            <?php foreach ($expense_types as $type): ?>
                                                <option value="<?php echo htmlspecialchars($type['id']); ?>" <?php echo $type['type_name'] === $expense['expense_type'] ? 'selected' : ''; ?>><?php echo htmlspecialchars($type['type_name']); ?></option>
                                            <?php endforeach; ?>
                                        </select>
                                    </div>
                                    <div class="mb-3">
                                        <label for="amount_expense_<?php echo htmlspecialchars($expense['id']); ?>" class="form-label">Amount (<?php echo htmlspecialchars($currency_symbol); ?>)</label>
                                        <input type="number" class="form-control" id="amount_expense_<?php echo htmlspecialchars($expense['id']); ?>" name="amount" value="<?php echo htmlspecialchars($expense['amount']); ?>" step="0.01" min="0.01" required>
                                    </div>
                                    <div class="mb-3">
                                        <label for="description_expense_<?php echo htmlspecialchars($expense['id']); ?>" class="form-label">Description</label>
                                        <textarea class="form-control" id="description_expense_<?php echo htmlspecialchars($expense['id']); ?>" name="description" rows="3" required><?php echo htmlspecialchars($expense['description']); ?></textarea>
                                    </div>
                                    <div class="mb-3">
                                        <label for="created_at_expense_<?php echo htmlspecialchars($expense['id']); ?>" class="form-label">Date</label>
                                        <input type="date" class="form-control" id="created_at_expense_<?php echo htmlspecialchars($expense['id']); ?>" name="created_at" value="<?php
                                            // Handle invalid or NULL created_at for input
                                            if (!empty($expense['created_at']) && $expense['created_at'] !== '0000-00-00' && $expense['created_at'] !== '0000-00-00 00:00:00' && strtotime($expense['created_at'])) {
                                                echo htmlspecialchars(date('Y-m-d', strtotime($expense['created_at'])));
                                            }
                                        ?>" required>
                                    </div>
                                </div>
                                <div class="modal-footer">
                                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                                    <button type="submit" class="btn btn-primary">Save Changes</button>
                                </div>
                            </form>
                        </div>
                    </div>
                </div>
            <?php endforeach; ?>

            <!-- Edit Benefit Modals -->
            <?php foreach ($benefits as $benefit): ?>
                <div class="modal fade" id="editBenefitModal<?php echo htmlspecialchars($benefit['id']); ?>" tabindex="-1" aria-labelledby="editBenefitModalLabel<?php echo htmlspecialchars($benefit['id']); ?>" aria-hidden="true">
                    <div class="modal-dialog">
                        <div class="modal-content">
                            <div class="modal-header">
                                <h5 class="modal-title" id="editBenefitModalLabel<?php echo htmlspecialchars($benefit['id']); ?>">Edit Benefit</h5>
                                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                            </div>
                            <form action="modify_records.php" method="post">
                                <div class="modal-body">
                                    <input type="hidden" name="edit_benefit" value="1">
                                    <input type="hidden" name="benefit_id" value="<?php echo htmlspecialchars($benefit['id']); ?>">
                                    <div class="mb-3">
                                        <label for="member_id_<?php echo htmlspecialchars($benefit['id']); ?>" class="form-label">Member</label>
                                        <select class="form-select" id="member_id_<?php echo htmlspecialchars($benefit['id']); ?>" name="member_id" required>
                                            <?php foreach ($members as $member): ?>
                                                <option value="<?php echo htmlspecialchars($member['id']); ?>" <?php echo $member['full_name'] === $benefit['full_name'] ? 'selected' : ''; ?>><?php echo htmlspecialchars($member['full_name']); ?></option>
                                            <?php endforeach; ?>
                                        </select>
                                    </div>
                                    <div class="mb-3">
                                        <label for="benefit_type_id_<?php echo htmlspecialchars($benefit['id']); ?>" class="form-label">Benefit Type</label>
                                        <select class="form-select" id="benefit_type_id_<?php echo htmlspecialchars($benefit['id']); ?>" name="benefit_type_id" required>
                                            <?php foreach ($benefit_types as $type): ?>
                                                <option value="<?php echo htmlspecialchars($type['id']); ?>" <?php echo $type['type_name'] === $benefit['benefit_type'] ? 'selected' : ''; ?>><?php echo htmlspecialchars($type['type_name']); ?></option>
                                            <?php endforeach; ?>
                                        </select>
                                    </div>
                                    <div class="mb-3">
                                        <label for="amount_benefit_<?php echo htmlspecialchars($benefit['id']); ?>" class="form-label">Amount (<?php echo htmlspecialchars($currency_symbol); ?>)</label>
                                        <input type="number" class="form-control" id="amount_benefit_<?php echo htmlspecialchars($benefit['id']); ?>" name="amount" value="<?php echo htmlspecialchars($benefit['amount']); ?>" step="0.01" min="0.01" required>
                                    </div>
                                    <div class="mb-3">
                                        <label for="benefit_date_<?php echo htmlspecialchars($benefit['id']); ?>" class="form-label">Date</label>
                                        <input type="date" class="form-control" id="benefit_date_<?php echo htmlspecialchars($benefit['id']); ?>" name="benefit_date" value="<?php
                                            // Handle invalid or NULL benefit_date for input
                                            if (!empty($benefit['benefit_date']) && $benefit['benefit_date'] !== '0000-00-00' && $benefit['benefit_date'] !== '0000-00-00 00:00:00' && strtotime($benefit['benefit_date'])) {
                                                echo htmlspecialchars(date('Y-m-d', strtotime($benefit['benefit_date'])));
                                            }
                                        ?>" required>
                                    </div>
                                    <div class="mb-3">
                                        <label for="benefit_status_<?php echo htmlspecialchars($benefit['id']); ?>" class="form-label">Status</label>
                                        <select class="form-select" id="benefit_status_<?php echo htmlspecialchars($benefit['id']); ?>" name="benefit_status" required>
                                            <option value="Received" <?php echo $benefit['benefit_status'] === 'Received' ? 'selected' : ''; ?>>Received</option>
                                            <option value="Rejected" <?php echo $benefit['benefit_status'] === 'Rejected' ? 'selected' : ''; ?>>Rejected</option>
                                            <option value="Pending" <?php echo $benefit['benefit_status'] === 'Pending' ? 'selected' : ''; ?>>Pending</option>
                                        </select>
                                    </div>
                                </div>
                                <div class="modal-footer">
                                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                                    <button type="submit" class="btn btn-primary">Save Changes</button>
                                </div>
                            </form>
                        </div>
                    </div>
                </div>
            <?php endforeach; ?>
        </main>
    </div>
</div>

<style>
.modal-backdrop.show {
    opacity: 0.5;
    z-index: 1040;
}
.modal {
    z-index: 1050;
}
.main-content {
    padding: 20px;
}
.table-responsive {
    max-height: 500px;
    overflow-y: auto;
}
</style>

<?php require_once 'includes/footer.php'; ?>