<?php
// member_request_benefit.php - Allows a standard club member to request a benefit or fund.
ob_start();
session_start();
ini_set('display_errors',1);
ini_set('display_startup_errors',1);
error_reporting(E_ALL);
mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);
if (function_exists('opcache_reset')) { opcache_reset(); }
require_once 'includes/functions.php';
require_once 'includes/db_config.php';
/* ---------- SESSION / PERMISSIONS ---------- */
$timeout = 1800;
if (isset($_SESSION['last_activity']) && (time() - $_SESSION['last_activity']) > $timeout) {
    session_unset(); session_destroy();
    redirect("login.php?timeout=1");
}
$_SESSION['last_activity'] = time();
if (!isset($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}
// Only Member can access this page
if (!is_logged_in() || !in_array($_SESSION['user_role'], ['member'])) {
    redirect("login.php"); exit;
}
/* ---------- SETTINGS & DATA FETCH ---------- */
$currency_symbol = get_setting('currency_symbol',$link) ?? '₦';
$message = '';
$error = '';
$user_id = $_SESSION['user_id'];
$csrf_token = $_SESSION['csrf_token'];
// Fetch the specific 'Member Benefit' request type ID (Target Role: Member)
$benefit_type_id = 0;
$sql_type = "SELECT id FROM request_types WHERE type_name = 'Member Benefit' AND target_role = 'member' LIMIT 1";
$result_type = mysqli_query($link, $sql_type);
if ($result_type && mysqli_num_rows($result_type) > 0) {
    $benefit_type_id = mysqli_fetch_assoc($result_type)['id'];
} else {
    $error = "The required 'Member Benefit' request type is missing from the database. Please contact an administrator.";
}
/* ---------- HANDLE FORM SUBMISSION ---------- */
if ($_SERVER["REQUEST_METHOD"] == "POST" && $benefit_type_id > 0) {
   
    // CSRF Check
    if (!isset($_POST['csrf_token']) || !hash_equals($csrf_token, $_POST['csrf_token'])) {
        $error = "Security token mismatch. Please try again.";
    } else {
        $amount = isset($_POST['amount']) ? sanitize_input($_POST['amount']) : 0.00;
        $purpose = isset($_POST['purpose']) ? sanitize_input($_POST['purpose']) : '';
        if (!is_numeric($amount) || $amount <= 0) {
            $error = "Please enter a valid requested amount greater than zero.";
        } elseif (empty($purpose)) {
            $error = "Please provide a detailed justification for your benefit request.";
        }
        if (empty($error)) {
            // Insert the new fund request into the database
            $status = 'pending';
            $sql_insert = "INSERT INTO fund_requests (request_type_id, requester_id, amount, purpose, status)
                               VALUES (?, ?, ?, ?, ?)";
           
            $stmt_insert = mysqli_prepare($link, $sql_insert);
            mysqli_stmt_bind_param($stmt_insert, "iidss", $benefit_type_id, $user_id, $amount, $purpose, $status);
           
            if (mysqli_stmt_execute($stmt_insert)) {
                $message = "Your Benefit Request of {$currency_symbol}" . number_format($amount, 2) . " has been submitted successfully for Admin review.";
                // Clear form data
                $amount = '';
                $purpose = '';
            } else {
                $error = "Failed to submit request: " . mysqli_error($link);
            }
            mysqli_stmt_close($stmt_insert);
        }
    }
    // Regenerate CSRF token
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    $csrf_token = $_SESSION['csrf_token'];
}
/* ---------- DATA FETCH FOR DISPLAY (Past Requests) ---------- */
$past_requests = [];
if ($benefit_type_id > 0) {
    $sql_past = "SELECT
                    fr.id, fr.amount, fr.purpose, fr.status, fr.created_at, fr.approval_date, fr.admin_note,
                    u_app.full_name AS approver_name
                   FROM fund_requests fr
                   LEFT JOIN users u_app ON fr.admin_approver_id = u_app.id
                   WHERE fr.requester_id = ? AND fr.request_type_id = ?
                   ORDER BY fr.created_at DESC";
    $stmt_past = mysqli_prepare($link, $sql_past);
    mysqli_stmt_bind_param($stmt_past, "ii", $user_id, $benefit_type_id);
    mysqli_stmt_execute($stmt_past);
    $result_past = mysqli_stmt_get_result($stmt_past);
    while ($row = mysqli_fetch_assoc($result_past)) {
        $past_requests[] = $row;
    }
    mysqli_stmt_close($stmt_past);
}
// Function to determine badge color
function get_status_badge($status) {
    switch ($status) {
        case 'pending': return 'badge bg-warning text-dark';
        case 'approved': return 'badge bg-success';
        case 'rejected': return 'badge bg-danger';
        case 'partially_spent': return 'badge bg-info';
        case 'spent': return 'badge bg-secondary';
        default: return 'badge bg-light text-dark';
    }
}
?>
<?php require_once 'includes/header.php'; ?>
<!-- LAYOUT FIX: Added title script and padding style -->
<script>document.title='Member Benefit Request - <?=htmlspecialchars(get_setting('system_name',$link)??'Club Portal')?>';</script>
<style>.main-content{padding:20px;}</style>
<div class="container-fluid">
    <!-- LAYOUT FIX: Changed from justify-content-center -->
    <div class="row">
        <?php
        // LAYOUT FIX: Explicitly include the sidebar
        require_once 'includes/sidebar.php';
        ?>
       
        <!-- LAYOUT FIX: Adjusted column class and added main-content class -->
        <main class="col-md-9 col-lg-10 main-content">
            <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                <h1 class="h2">Member Benefit Request</h1>
            </div>
            <?php if (!empty($error)): ?>
                <div class="alert alert-danger"><?=htmlspecialchars($error)?></div>
            <?php endif; ?>
            <?php if (!empty($message)): ?>
                <div class="alert alert-success"><?=htmlspecialchars($message)?></div>
            <?php endif; ?>
           
            <!-- Benefit Request Form -->
            <div class="card shadow-sm mb-5">
                <div class="card-header bg-primary text-white">
                    Submit a New Benefit Claim
                </div>
                <div class="card-body">
                    <p class="text-muted">Fill out this form to request access to club benefits, funds, or grants you are entitled to. All requests require Administrator approval.</p>
                    <?php if ($benefit_type_id > 0): ?>
                    <form method="POST" action="member_request_benefit.php">
                        <input type="hidden" name="csrf_token" value="<?=htmlspecialchars($csrf_token)?>">
                        <div class="mb-3">
                            <label for="amount" class="form-label fw-bold">Benefit Amount Requested (<?=$currency_symbol?>)</label>
                            <input type="number" class="form-control" id="amount" name="amount"
                                   step="0.01" min="0.01" required value="<?=htmlspecialchars($amount ?? '')?>"
                                   placeholder="e.g., 2500.00">
                            <div class="form-text">Enter the specific monetary value of the benefit or grant you are applying for.</div>
                        </div>
                        <div class="mb-3">
                            <label for="purpose" class="form-label fw-bold">Detailed Justification/Purpose</label>
                            <textarea class="form-control" id="purpose" name="purpose" rows="4" required
                                    placeholder="Explain which benefit you are claiming and why (e.g., 'Claiming the annual education grant for the purchase of new course materials')."><?=htmlspecialchars($purpose ?? '')?></textarea>
                            <div class="form-text">Be specific. This is what the Admin will use to decide on approval.</div>
                        </div>
                        <button type="submit" class="btn btn-primary mt-3">Submit Benefit Request</button>
                    </form>
                    <?php endif; ?>
                </div>
            </div>
            <!-- Past Requests Table -->
            <div class="card shadow-sm mb-4">
                <div class="card-header bg-secondary text-white">
                    Your Past Benefit Claims
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <?php if (empty($past_requests)): ?>
                            <div class="alert alert-info">You have not submitted any benefit claims yet.</div>
                        <?php else: ?>
                        <table class="table table-striped table-hover align-middle">
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>Amount</th>
                                    <th>Status</th>
                                    <th>Date Submitted</th>
                                    <th>Admin Response</th>
                                    <th>Actioned By</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($past_requests as $r): ?>
                                <tr>
                                    <td><?=htmlspecialchars($r['id'])?></td>
                                    <td class="fw-bold"><?=$currency_symbol . number_format($r['amount'], 2)?></td>
                                    <td><span class="<?=get_status_badge($r['status'])?>"><?=ucfirst(str_replace('_', ' ', $r['status']))?></span></td>
                                    <td><?=date('Y-m-d H:i', strtotime($r['created_at']))?></td>
                                    <td>
                                        <button class="btn btn-link btn-sm" type="button" data-bs-toggle="collapse" data-bs-target="#collapsePurpose<?=htmlspecialchars($r['id'])?>" aria-expanded="false" aria-controls="collapsePurpose<?=htmlspecialchars($r['id'])?>">
                                            View Details
                                        </button>
                                        <div class="collapse mt-2" id="collapsePurpose<?=htmlspecialchars($r['id'])?>">
                                            <div class="card card-body p-2 bg-light border">
                                                <p class="mb-1"><strong>Purpose:</strong> <?=htmlspecialchars($r['purpose'])?></p>
                                                <?php if (isset($r['admin_note']) && !empty($r['admin_note'])): // Now includes the admin note ?>
                                                    <p class="mb-0 text-danger"><strong>Admin Note:</strong> <?=htmlspecialchars($r['admin_note'])?></p>
                                                <?php endif; ?>
                                            </div>
                                        </div>
                                    </td>
                                    <td>
                                        <?php if (!empty($r['approver_name'])): ?>
                                            <?=htmlspecialchars($r['approver_name'])?> (<?=date('Y-m-d', strtotime($r['approval_date']))?>)
                                        <?php else: ?>
                                            N/A
                                        <?php endif; ?>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
           
        </main>
    </div>
</div>
<?php require_once 'includes/footer.php'; ?>
<?php ob_end_flush(); ?>