<?php
// member_management.php
use Dompdf\Dompdf;

// CSV EXPORT
if (isset($_GET['export']) && $_GET['export'] == 'csv') {
    require_once 'includes/functions.php';
    require_once 'includes/db_config.php';
    if (!is_logged_in() || !is_admin()) { redirect("login.php"); exit(); }

    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename=members.csv');
    $out = fopen('php://output', 'w');
    fputcsv($out, ['Member ID','Full Name','Email','Membership Status']);

    $sql = "SELECT u.member_id, u.full_name, u.email, u.membership_status,
                   fd.data AS form_data, f.form_data AS form_structure
            FROM users u
            LEFT JOIN form_data fd ON u.id = fd.user_id
            LEFT JOIN forms f ON fd.form_id = f.id
            WHERE u.role IN ('member','pending')";
    $res = mysqli_query($link, $sql);
    while ($row = mysqli_fetch_assoc($res)) {
        $surname = $other = '';
        $fd = $row['form_data'] ? json_decode($row['form_data'], true) : null;
        $fs = $row['form_structure'] ? json_decode($row['form_structure'], true) : null;
        if ($fd && $fs && isset($fs['fields'])) {
            foreach ($fs['fields'] as $field) {
                $lbl = strtolower($field['label'] ?? '');
                if (strpos($lbl, 'surname') !== false && isset($fd[$field['name']])) $surname = $fd[$field['name']];
                if (strpos($lbl, 'other names') !== false && isset($fd[$field['name']])) $other = $fd[$field['name']];
            }
        }
        $name = trim($surname.' '.$other) ?: $row['full_name'];
        fputcsv($out, [$row['member_id'],$name,$row['email'],$row['membership_status']]);
    }
    fclose($out);
    exit();
}

// PDF ALL EXPORT
if (isset($_GET['export']) && $_GET['export'] == 'pdf_all') {
    require_once 'includes/functions.php';
    if (!is_logged_in() || !is_admin()) { redirect("login.php"); exit(); }
    header("Location: export_all_members_pdf.php");
    exit();
}

// BULK ACTIONS
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['bulk_action'])) {
    require_once 'includes/functions.php';
    require_once 'includes/db_config.php';
    // Start session to access CSRF token
    if (session_status() === PHP_SESSION_NONE) { session_start(); }

    if (!is_logged_in() || !is_admin()) { echo json_encode(['error'=>'Unauthorised']); exit(); }

    if (!isset($_POST['csrf_token']) || !hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {
        echo json_encode(['error'=>'Invalid CSRF token']);
        exit();
    }

    $ids_str = $_POST['ids'] ?? '';
    $ids     = !empty($ids_str) ? explode(',', $ids_str) : [];
    $action  = $_POST['bulk_action'];
    $ids     = array_filter($ids, 'is_numeric'); // Sanitize to ensure all are numbers

    if (empty($ids) || !in_array($action, ['approved','rejected','delete'])) {
        echo json_encode(['error'=>'Invalid request']);
        exit();
    }

    if ($action === 'delete') {
        $placeholders = str_repeat('?,', count($ids)-1).'?';
        $stmt = mysqli_prepare($link, "UPDATE users SET deleted_at = NOW() WHERE id IN ($placeholders)");
        $types = str_repeat('i', count($ids));
        $params = $ids;
    } else {
        $placeholders = str_repeat('?,', count($ids)-1).'?';
        $stmt = mysqli_prepare($link, "UPDATE users SET membership_status = ? WHERE id IN ($placeholders)");
        $types = 's' . str_repeat('i', count($ids));
        $params = array_merge([$action], $ids);
    }

    $ref = [];
    foreach ($params as $k=>$v) $ref[$k] = &$params[$k];
    array_unshift($ref, $types);
    call_user_func_array([$stmt, 'bind_param'], $ref);
    mysqli_stmt_execute($stmt);
    $affected = mysqli_stmt_affected_rows($stmt);
    mysqli_stmt_close($stmt);

    if ($action === 'approved' && $affected) {
        $placeholders = str_repeat('?,', count($ids)-1).'?';
        $stmt2 = mysqli_prepare($link, "SELECT email, full_name FROM users WHERE id IN ($placeholders)");
        $types2 = str_repeat('i', count($ids));
        $ref2 = [];
        foreach ($ids as $k=>$v) $ref2[$k] = &$ids[$k];
        array_unshift($ref2, $types2);
        call_user_func_array([$stmt2, 'bind_param'], $ref2);
        mysqli_stmt_execute($stmt2);
        $res = mysqli_stmt_get_result($stmt2);
        while ($row = mysqli_fetch_assoc($res)) {
            $to = $row['email'];
            $name = $row['full_name'];
            $subject = "Membership Approved";
            $message = "Dear $name,\n\nYour membership has been approved. Welcome!";
            $headers = "From: no-reply@yoursite.com";
            @mail($to, $subject, $message, $headers);
        }
        mysqli_stmt_close($stmt2);
    }

    echo json_encode(['success'=>true, 'affected'=>$affected]);
    exit();
}

// PRINT ID CARD (PDF) - This code is now legacy and no longer linked, but left intact.
if (isset($_GET['print_id_card']) && is_numeric($_GET['id'])) {
    require_once 'includes/functions.php';
    if (!is_logged_in() || !is_admin()) {
        die('Unauthorised. Please log in as an admin.');
    }

    require_once 'includes/db_config.php';
    require_once 'includes/dompdf/autoload.inc.php';
    require_once 'includes/phpqrcode/qrlib.php';

    $id = (int)$_GET['id'];
    $system_name = get_setting('system_name', $link) ?? 'Club Portal';

    $sql = "SELECT u.full_name, u.member_id, u.profile_picture, u.membership_status, u.expiry_date
            FROM users u
            WHERE u.id = ? AND u.role IN ('member','pending')";
    
    $stmt = mysqli_prepare($link, $sql);
    mysqli_stmt_bind_param($stmt, 'i', $id);
    mysqli_stmt_execute($stmt);
    $result = mysqli_stmt_get_result($stmt);
    $row = mysqli_fetch_assoc($result);
    mysqli_stmt_close($stmt);

    if (!$row) { die('Member not found.'); }

    $status = (strtolower($row['membership_status']) === 'approved' && (!$row['expiry_date'] || $row['expiry_date'] > date('Y-m-d')))
        ? 'Active' : 'Expired';

    // PROFILE PICTURE
    $profile = $row['profile_picture'] ?: 'uploads/profile_pictures/default.png';
    $profile_path = $_SERVER['DOCUMENT_ROOT'] . '/' . ltrim($profile, '/'); // ltrim to handle /uploads vs uploads
    $default_path = $_SERVER['DOCUMENT_ROOT'] . '/uploads/profile_pictures/default.png';
    
    if (!file_exists($profile_path)) {
        $profile_path = $default_path;
    }
    
    $profile_data = 'data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAEAAAABCAQAAAC1HAwCAAAAC0lEQVR42mNkYAAAAAYAAjCB0C8AAAAASUVORK5CYII='; // Default transparent pixel
    if (file_exists($profile_path)) {
        $finfo = finfo_open(FILEINFO_MIME_TYPE);
        $mime_type = finfo_file($finfo, $profile_path);
        finfo_close($finfo);
        $profile_data = 'data:' . $mime_type . ';base64,' . base64_encode(file_get_contents($profile_path));
    }

    // QR CODE
    $qr_url = "https://{$_SERVER['HTTP_HOST']}/verify_id.php?id=" . $row['member_id'];
    $qr_file = 'temp/qr_' . $id . '.png';
    if (!is_dir('temp')) mkdir('temp', 0777, true);
    QRcode::png($qr_url, $qr_file, QR_ECLEVEL_L, 8);
    
    $qr_data = 'data:image/png;base64,' . base64_encode(file_get_contents($qr_file));
    @unlink($qr_file); // Clean up temp file

    $html = '
    <!DOCTYPE html>
    <html>
    <head>
        <meta charset="UTF-8">
        <title>ID Card - Print</title>
        <style>
            @page { margin: 0; size: 85.6mm 54mm landscape; }
            body { margin: 0; padding: 0; font-family: Arial, sans-serif; background: #fff; }
            .card {
                width: 100%; height: 100%; background: #fff;
                border-radius: 10px; overflow: hidden; position: relative;
                display: flex; padding: 4mm; box-sizing: border-box; font-size: 9pt;
            }
            .left { flex: 1; display: flex; flex-direction: column; justify-content: center; margin-right: 4mm; }
            .photo { width: 18mm; height: 18mm; border-radius: 50%; border: 2px solid #000; object-fit: cover; margin-bottom: 2mm; }
            .info { font-weight: bold; line-height: 1.4; }
            .info p { margin: 1mm 0 0; white-space: nowrap; overflow: hidden; text-overflow: ellipsis; max-width: 40mm; }
            .right { width: 26mm; display: flex; flex-direction: column; justify-content: space-between; align-items: flex-end; }
            .name { font-size: 13pt; font-weight: bold; color: #28a745; text-align: right; line-height: 1.2; word-wrap: break-word; }
            .qr { width: 16mm; height: 16mm; }
            .footer {
                position: absolute; bottom: 2mm; left: 4mm; font-size: 7pt; color: #28a745; font-weight: bold;
            }
        </style>
    </head>
    <body onload="window.print(); window.close();">
        <div class="card">
            <div class="left">
                <img src="' . $profile_data . '" class="photo">
                <div class="info">
                    <p>ID: ' . htmlspecialchars($row['member_id']) . '</p>
                    <p>Status: <span style="color:' . ($status==='Active'?'#28a745':'#dc3545') . '">' . $status . '</span></p>
                </div>
            </div>
            <div class="right">
                <div class="name">' . htmlspecialchars($row['full_name']) . '</div>
                <img src="' . $qr_data . '" class="qr">
            </div>
            <div class="footer">Membership ID Card - ' . htmlspecialchars($system_name) . '</div>
        </div>
    </body>
    </html>';

    $dompdf = new Dompdf();
    $dompdf->getOptions()->set('isRemoteEnabled', true); // Allow loading inline base64 images
    $dompdf->loadHtml($html);
    $dompdf->setPaper([0, 0, 241.44, 152.88], 'landscape'); // Credit card size
    $dompdf->render();
    $dompdf->stream("id_card_{$row['member_id']}.pdf", ['Attachment' => false]);
    exit();
}

// MAIN PAGE
require_once 'includes/functions.php';
require_once 'includes/db_config.php';
if (!is_logged_in() || !is_admin()) { redirect("login.php"); exit(); }

$system_name = get_setting('system_name', $link) ?? 'Club Portal';
$_SESSION['csrf_token'] = $_SESSION['csrf_token'] ?? bin2hex(random_bytes(32));

$perPage = 10;
$page    = max(1, (int)($_GET['page'] ?? 1));
$search  = trim($_GET['search'] ?? '');
$offset  = ($page-1) * $perPage;

$where = "u.role IN ('member','pending') AND (u.deleted_at IS NULL OR u.deleted_at > NOW())";
$params = [];
$types  = '';

if ($search !== '') {
    $where .= " AND (u.member_id LIKE ? OR u.full_name LIKE ? OR u.email LIKE ?)";
    $like = "%$search%";
    $params = [$like,$like,$like];
    $types  = 'sss';
}

$countSql = "SELECT COUNT(*) FROM users u WHERE $where";
if ($types) {
    $stmt = mysqli_prepare($link, $countSql);
    mysqli_stmt_bind_param($stmt, $types, ...$params);
    mysqli_stmt_execute($stmt);
    $total = mysqli_fetch_row(mysqli_stmt_get_result($stmt))[0];
    mysqli_stmt_close($stmt);
} else {
    $total = mysqli_fetch_row(mysqli_query($link, $countSql))[0];
}
$totalPages = max(1, ceil($total / $perPage));

$sql = "SELECT u.id, u.member_id, u.full_name, u.email, u.membership_status,
               u.profile_picture, u.role, u.expiry_date
        FROM users u
        WHERE $where
        ORDER BY u.id DESC
        LIMIT ? OFFSET ?";
$limitParams = array_merge($params, [$perPage, $offset]);
$limitTypes  = $types . 'ii';

$stmt = mysqli_prepare($link, $sql);
mysqli_stmt_bind_param($stmt, $limitTypes, ...$limitParams);
mysqli_stmt_execute($stmt);
$result = mysqli_stmt_get_result($stmt);
$members = [];
while ($row = mysqli_fetch_assoc($result)) $members[] = $row;
mysqli_stmt_close($stmt);
?>
<?php require_once 'includes/header.php'; ?>
<script>document.title = 'Member Management - <?php echo htmlspecialchars($system_name); ?>';</script>

<style>
.main-content{padding:20px;}
.modal-lg{max-width:900px;}
.profile-img{width:120px;height:120px;border-radius:50%;object-fit:cover;border:2px solid #ddd;}
.badge{font-size:.9em;padding:.4em .6em;}
.modal-dialog-scrollable .modal-body{max-height:70vh !important;overflow-y:auto !important;padding-right:10px;}
.table-green-border{border:3px solid #28a745 !important;border-collapse:separate;border-spacing:0;border-radius:8px;overflow:hidden;table-layout:fixed;width:100%;}
.table-green-border th,.table-green-border td{border:1px solid #28a745 !important;padding:.75rem;vertical-align:top;word-wrap:break-word;overflow-wrap:break-word;}
.table-green-border th{width:30%;background:#d4edda;color:#155724;}
.table-green-border tbody tr:hover{background:#f1f8f3;}
.modal-open{overflow:hidden !important;}
.modal-backdrop{position:fixed;top:0;left:0;z-index:1040;width:100vw;height:100vh;background:#000;opacity:.5;}
#memberDetailsModal{z-index:1055 !important;}
.pagination .page-link { color: #28a745; }
.pagination .page-item.active .page-link { background-color: #28a745; border-color: #28a745; }
.pagination .page-item.disabled .page-link { color: #6c757d; }

/* ID CARD MODAL */
.id-card-modal {
    width: 80mm; height: 50mm; background: #fff; border: 2px solid #000;
    border-radius: 10px; overflow: hidden; position: relative; box-shadow: 0 0 15px rgba(0,0,0,0.4);
    display: flex; padding: 4mm; margin: 20px auto; font-family: Arial, sans-serif; font-size: 9pt;
}
.id-card-modal .left { flex: 1; display: flex; flex-direction: column; justify-content: center; margin-right: 4mm; }
.id-card-modal .photo { width: 18mm; height: 18mm; border-radius: 50%; border: 2px solid #000; object-fit: cover; margin-bottom: 2mm; }
.id-card-modal .info { font-weight: bold; line-height: 1.4; }
.id-card-modal .info p { margin: 1mm 0 0; white-space: nowrap; overflow: hidden; text-overflow: ellipsis; max-width: 40mm; }
.id-card-modal .right { width: 26mm; display: flex; flex-direction: column; justify-content: space-between; align-items: flex-end; }
.id-card-modal .name { font-size: 14pt; font-weight: bold; color: #28a745; text-align: right; line-height: 1.2; word-wrap: break-word; }
.id-card-modal .qr { width: 16mm; height: 16mm; }
.id-card-modal .footer { position: absolute; bottom: 2mm; left: 4mm; font-size: 7pt; color: #28a745; font-weight: bold; }
</style>

<div class="container-fluid">
    <div class="row">
        <?php require_once 'includes/sidebar.php'; ?>
        <main class="col-md-9 col-lg-10 main-content">
            <h2>Member Management</h2>

            <div class="card mb-3">
                <div class="card-body d-flex flex-wrap gap-2 align-items-center">
                    <form method="get" class="d-flex flex-grow-1">
                        <input type="text" name="search" class="form-control me-2" placeholder="Search by ID, Name or Email..." value="<?=htmlspecialchars($search)?>" style="max-width:300px;">
                        <input type="hidden" name="page" value="<?= $page ?>">
                        <button type="submit" class="btn btn-outline-primary">Search</button>
                    </form>

                    <select id="bulkActionSelect" class="form-select me-2" style="width:auto;">
                        <option value="">Bulk Action</option>
                        <option value="approved">Approve Selected</option>
                        <option value="rejected">Reject Selected</option>
                        <option value="delete">Delete Selected</option>
                    </select>
                    <button id="applyBulk" class="btn btn-warning btn-sm" disabled>Apply</button>

                    <div class="ms-auto">
                        <a href="member_management.php?export=csv" class="btn btn-secondary btn-sm me-2">Export CSV</a>
                        <a href="member_management.php?export=pdf_all" class="btn btn-success btn-sm">Export All to PDF</a>
                    </div>
                </div>
            </div>

            <div class="card">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <span>Members (<?= $total ?> total)</span>
                    <small class="text-muted">Page <?= $page ?> of <?= $totalPages ?></small>
                </div>
                <div class="card-body table-responsive">
                    <form id="membersForm">
                    <table class="table table-striped align-middle">
                        <thead>
                            <tr>
                                <th><input type="checkbox" id="selectAll"></th>
                                <th>Member ID</th>
                                <th>Full Name</th>
                                <th>Email</th>
                                <th>Status</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                        <?php if (empty($members)): ?>
                            <tr><td colspan="6" class="text-center">No members found.</td></tr>
                        <?php else: foreach ($members as $m):
                            $status = strtolower($m['membership_status']);
                            $badge  = $status==='approved' ? 'bg-success' : ($status==='rejected' ? 'bg-danger' : 'bg-secondary');
                        ?>
                            <tr>
                                <td><input type="checkbox" name="member_id[]" value="<?= $m['id'] ?>"></td>
                                <td><?= htmlspecialchars($m['member_id']) ?></td>
                                <td>
                                    <a href="javascript:void(0);" onclick="viewMemberDetails(<?= $m['id'] ?>)">
                                        <?= htmlspecialchars($m['full_name']) ?>
                                    </a>
                                </td>
                                <td><?= htmlspecialchars($m['email']) ?></td>
                                <td>
                                    <?php if ($status === 'pending'): ?>
                                        <select class="form-select form-select-sm status-dropdown" data-id="<?= $m['id'] ?>">
                                            <option value="pending" selected>Pending</option>
                                            <option value="approved">Approved</option>
                                            <option value="rejected">Rejected</option>
                                        </select>
                                    <?php else: ?>
                                        <span class="badge <?= $badge ?>"><?= ucfirst($status) ?></span>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <button type="button" class="btn btn-info btn-sm" onclick="viewMemberDetails(<?= $m['id'] ?>)">View</button>
                                    <button type="button" class="btn btn-success btn-sm ms-1" onclick="showIdCard(<?= $m['id'] ?>)">ID Card</button>
                                    <!-- **** FIXED BUTTON **** -->
                                    <!-- Changed href to point to member_id_cards.php PNG download -->
                                    <!-- Changed text from "Print ID" to "Download PNG" -->
                                    <a href="member_id_cards.php?download_single_png=1&id=<?= $m['id'] ?>" class="btn btn-primary btn-sm ms-1" target="_blank">Download PNG</a>
                                </td>
                            </tr>
                        <?php endforeach; endif; ?>
                        </tbody>
                    </table>
                    </form>
                </div>

                <?php if ($totalPages > 1): ?>
                <nav aria-label="Page navigation" class="mt-3">
                    <ul class="pagination justify-content-center mb-0">
                        <li class="page-item <?= $page <= 1 ? 'disabled' : '' ?>">
                            <a class="page-link" href="?page=<?= $page-1 ?>&search=<?= urlencode($search) ?>">Previous</a>
                        </li>
                        <?php if ($page > 3): ?>
                            <li class="page-item"><a class="page-link" href="?page=1&search=<?= urlencode($search) ?>">1</a></li>
                            <?php if ($page > 4): ?><li class="page-item disabled"><span class="page-link">...</span></li><?php endif; ?>
                        <?php endif; ?>
                        <?php for ($i = max(1, $page - 2); $i <= min($totalPages, $page + 2); $i++): ?>
                            <li class="page-item <?= $i == $page ? 'active' : '' ?>">
                                <a class="page-link" href="?page=<?= $i ?>&search=<?= urlencode($search) ?>"><?= $i ?></a>
                            </li>
                        <?php endfor; ?>
                        <?php if ($page < $totalPages - 2): ?>
                            <?php if ($page < $totalPages - 3): ?><li class="page-item disabled"><span class="page-link">...</span></li><?php endif; ?>
                            <li class="page-item"><a class="page-link" href="?page=<?= $totalPages ?>&search=<?= urlencode($search) ?>"><?= $totalPages ?></a></li>
                        <?php endif; ?>
                        <li class="page-item <?= $page >= $totalPages ? 'disabled' : '' ?>">
                            <a class="page-link" href="?page=<?= $page+1 ?>&search=<?= urlencode($search) ?>">Next</a>
                        </li>
                    </ul>
                </nav>
                <div class="text-center mt-2"><small class="text-muted">Showing <?= count($members) ?> of <?= $total ?> members</small></div>
                <?php endif; ?>
            </div>
        </main>
    </div>
</div>

<!-- MODAL -->
<div class="modal fade" id="memberDetailsModal" tabindex="-1">
  <div class="modal-dialog modal-lg modal-dialog-scrollable">
    <div class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title" id="memberDetailsLabel">Member Details</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
      </div>
      <div class="modal-body" id="member-details-body"><div class="text-center"><em>Loading...</em></div></div>
      <div class="modal-footer">
        <div class="me-auto" id="approvalButtons" style="display:none;">
            <button type="button" id="approveButton" class="btn btn-success">Approve</button>
            <button type="button" id="rejectButton" class="btn btn-danger">Reject</button>
        </div>
        <button type="button" id="editButton" class="btn btn-primary">Edit</button>
        <button type="button" id="exportPdfButton" class="btn btn-warning">Export to PDF</button>
        <button type="button" id="printIdCardBtn" class="btn btn-success d-none">Print ID Card</button>
        <button type="button" id="saveButton" class="btn btn-success d-none">Save Changes</button>
      </div>
    </div>
  </div>
</div>

<script>
let memberModal = null;
let isModalOpening = false;
let currentMemberId = null;
const csrfToken = '<?php echo $_SESSION['csrf_token']; ?>';

function viewMemberDetails(id) { loadMember(id, 'details'); }
function showIdCard(id)       { loadMember(id, 'idcard'); }

function loadMember(id, mode) {
    if (isModalOpening) return;
    isModalOpening = true;
    
    if (memberModal) { memberModal.hide(); }

    fetch('fetch_member_details.php?id=' + id)
        .then(r => r.json())
        .then(d => {
            if (d.error) { alert(d.error); isModalOpening = false; return; }
            window.currentMemberData = d;
            currentMemberId = id;

            let html = '';
            
            const modalTitle = document.getElementById('memberDetailsLabel');
            const modalBody = document.getElementById('member-details-body');
            const editBtn = document.getElementById('editButton');
            const saveBtn = document.getElementById('saveButton');
            const exportBtn = document.getElementById('exportPdfButton');
            const printBtn = document.getElementById('printIdCardBtn');
            const approvalBox = document.getElementById('approvalButtons');

            if (mode === 'idcard') {
                const status = (d.membership_status.toLowerCase() === 'approved') ? 'Active' : 'Expired';
                const qrUrl = `https://${location.host}/verify_id.php?id=${d.member_id}`;

                // **** FIXED BUTTON ****
                // Changed the button's onclick to point to member_id_cards.php PNG download
                // Changed text from "Print ID Card (PDF)" to "Download ID Card (PNG)"
                html = `
                <div class="text-center">
                    <div class="id-card-modal">
                        <div class="left">
                            <img src="${d.profile_picture}" class="photo" onerror="this.src='uploads/profile_pictures/default.png'">
                            <div class="info">
                                <p>ID: ${d.member_id}</p>
                                <p>Status: <span style="color:${status==='Active'?'#28a745':'#dc3545'}">${status}</span></p>
                            </div>
                        </div>
                        <div class="right">
                            <div class="name">${d.full_name}</div>
                            <img src="https://api.qrserver.com/v1/create-qr-code/?size=160x160&data=${encodeURIComponent(qrUrl)}" class="qr">
                        </div>
                        <div class="footer">Membership ID Card - ${d.system_name}</div>
                    </div>
                    <p class="mt-3"><small>Scan QR to verify membership</small></p>
                    <button onclick="window.location.href='member_id_cards.php?download_single_png=1&id=${id}'" class="btn btn-success btn-sm mt-2">Download ID Card (PNG)</button>
                </div>`;
                
                modalTitle.innerText = 'Member ID Card';
                modalBody.innerHTML = html;
                
                editBtn.classList.add('d-none');
                saveBtn.classList.add('d-none');
                exportBtn.classList.add('d-none');
                printBtn.classList.add('d-none');
                approvalBox.style.display = 'none';

            } else { 
                // --- DETAILS MODE ---
                html = `<div class="text-center mb-4">
                    <img src="${d.profile_picture}" class="profile-img mb-2" id="profilePreview" onerror="this.src='uploads/profile_pictures/default.png'">
                    <input type="file" id="profileInput" class="form-control w-50 mx-auto d-none" accept="image/*">
                </div>
                <h5 class="mb-3">Core Information</h5>
                <table class="table table-green-border mb-5"><tbody>
                    <tr><th>Member ID</th><td>${d.member_id}</td></tr>
                    <tr><th>Full Name</th><td><input class="form-control" readonly id="full_name" value="${d.full_name.replace(/"/g,'&quot;')}"></td></tr>
                    <tr><th>Email</th><td><input class="form-control" readonly id="email" value="${d.email.replace(/"/g,'&quot;')}"></td></tr>
                    <tr><th>Membership Status</th><td>${d.membership_status}</td></tr>
                </tbody></table>
                <h5 class="mb-3">Registration Data</h5>
                <table class="table table-green-border" id="customFieldsTable"><tbody>`;
                if (d.form_data && d.form_data.length) {
                    d.form_data.forEach(f => {
                        const esc = f.value.replace(/&/g,'&amp;').replace(/</g,'&lt;').replace(/>/g,'&gt;').replace(/"/g,'&quot;');
                        html += `<tr><th>${f.label}</th><td><input type="text" name="${f.name}" class="form-control" readonly value="${esc}" style="width:100%;"></td></tr>`;
                    });
                } else {
                    html += `<tr><td colspan="2" class="text-center text-muted">No registration data.</td></tr>`;
                }
                html += `</tbody></table>`;

                modalTitle.innerText = 'Member Details';
                modalBody.innerHTML = html;

                exportBtn.classList.remove('d-none');
                printBtn.classList.add('d-none');
                
                if (d.membership_status.toLowerCase() === 'pending') {
                    approvalBox.style.display = 'block';
                    document.getElementById('approveButton').onclick = () => updateMemberStatus(id, 'approved');
                    document.getElementById('rejectButton').onclick = () => updateMemberStatus(id, 'rejected');
                } else {
                    approvalBox.style.display = 'none';
                }
                
                toggleEditMode(false);
            }

            const modalEl = document.getElementById('memberDetailsModal');
            memberModal = bootstrap.Modal.getOrCreateInstance(modalEl);
            
            modalEl.addEventListener('shown.bs.modal', () => { 
                isModalOpening = false; 
                
                if (mode === 'details') {
                    document.getElementById('exportPdfButton').onclick = () => exportSingleMemberToPdf(window.currentMemberData);
                    document.getElementById('editButton').onclick = () => toggleEditMode(true);
                    document.getElementById('saveButton').onclick = () => saveMemberChanges(currentMemberId);
                } else {
                    document.getElementById('exportPdfButton').onclick = null;
                    document.getElementById('editButton').onclick = null;
                    document.getElementById('saveButton').onclick = null;
                }
            }, { once: true });
            
            return memberModal.show();
        })
        .catch(err => { alert('Failed to load.'); console.error(err); isModalOpening = false; });
}

function cleanupModal() {
    if (memberModal) { 
        document.getElementById('exportPdfButton').onclick = null;
        document.getElementById('editButton').onclick = null;
        document.getElementById('saveButton').onclick = null;
        document.getElementById('approveButton').onclick = null;
        document.getElementById('rejectButton').onclick = null;
        memberModal.dispose(); 
        memberModal = null; 
    }
    document.querySelectorAll('.modal-backdrop').forEach(b => b.remove());
    document.body.classList.remove('modal-open');
    document.body.style.overflow = ''; document.body.style.paddingRight = '';
}
document.getElementById('memberDetailsModal').addEventListener('hidden.bs.modal', () => { cleanupModal(); isModalOpening = false; });

document.querySelectorAll('.status-dropdown').forEach(sel => {
    sel.addEventListener('change', function() {
        const id = this.dataset.id;
        const status = this.value;
        if (confirm(`Change status to ${status}?`)) {
            updateMemberStatus(id, status);
        } else {
            this.value = 'pending'; // Revert change if cancelled
        }
    });
});

document.getElementById('selectAll').addEventListener('change', function() {
    document.querySelectorAll('input[name="member_id[]"]').forEach(cb => cb.checked = this.checked);
    toggleBulkButton();
});
document.querySelectorAll('input[name="member_id[]"]').forEach(cb => cb.addEventListener('change', toggleBulkButton));

function toggleBulkButton() {
    const checked = document.querySelectorAll('input[name="member_id[]"]:checked').length;
    document.getElementById('applyBulk').disabled = checked === 0;
}
document.getElementById('applyBulk').addEventListener('click', function() {
    const action = document.getElementById('bulkActionSelect').value;
    if (!action) return;
    const ids = Array.from(document.querySelectorAll('input[name="member_id[]"]:checked')).map(c => c.value);
    let msg = `You are about to ${action} ${ids.length} member(s). Continue?`;
    if (action === 'delete') msg = `DELETE ${ids.length} member(s)? This action is permanent.`;
    
    if (!confirm(msg)) return;

    fetch('', { // Post to self
        method: 'POST',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: `bulk_action=${action}&ids=${ids.join(',')}&csrf_token=${csrfToken}`
    })
    .then(r => r.json())
    .then(res => {
        if (res.error) alert(res.error);
        else { alert(`${action === 'delete' ? 'Deleted' : 'Updated'} ${res.affected} member(s)`); location.reload(); }
    });
});

function updateMemberStatus(id, status) {
    fetch('update_member_status.php', {
        method: 'POST',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: `id=${id}&status=${encodeURIComponent(status)}&csrf_token=${csrfToken}`
    })
    .then(r => r.json())
    .then(d => { 
        if(d.error) {
            alert(d.error);
        } else {
            alert(d.message || 'Status updated');
        }
        location.reload(); 
    });
}

function exportSingleMemberToPdf(data) {
    if (typeof window.jspdf === 'undefined') {
        alert('PDF library is not loaded. Please contact support.');
        return;
    }
    const { jsPDF } = window.jspdf;
    const doc = new jsPDF(); let y = 20; const line = 7; const ph = 280;
    doc.setFontSize(16); doc.text('Member Details', 20, y); y += 15;
    
    const img = new Image();
    img.setAttribute('crossOrigin', 'anonymous');
    img.onload = function() {
        let canvas = document.createElement('canvas');
        canvas.width = this.width;
        canvas.height = this.height;
        let ctx = canvas.getContext('2d');
        ctx.drawImage(this, 0, 0);
        let dataURL = canvas.toDataURL('image/png');

        if (dataURL.length > 10) {
            try { doc.addImage(dataURL, 'PNG', 20, y, 30, 30); } catch (e) { console.error("Error adding image to PDF: ", e); }
        }
        y += 35;
        renderPdfText();
    };
    img.onerror = function() {
        console.error("Could not load profile image for PDF.");
        y += 35;
        renderPdfText();
    };
    img.src = document.getElementById('profilePreview').src;

    function renderPdfText() {
        doc.setFontSize(12);
        doc.text('Core Information', 20, y); y += 10;
        ['Member ID','Full Name','Email','Status'].forEach((t, i) => {
            const v = [data.member_id, data.full_name, data.email, data.membership_status][i];
            doc.text(`${t}: ${v}`, 20, y); y += line;
        });
        y += 5; doc.text('Registration Data', 20, y); y += 10;
        if (data.form_data && data.form_data.length) {
            data.form_data.forEach(f => {
                if (y > ph) { doc.addPage(); y = 20; }
                const lines = doc.splitTextToSize(`${f.label}: ${f.value}`, 170);
                doc.text(lines, 20, y); y += lines.length * line;
            });
        } else doc.text('No registration data.', 20, y);
        doc.save(`member_${data.member_id}.pdf`);
    }
}


function toggleEditMode(enable) {
    document.querySelectorAll('#member-details-body input.form-control').forEach(i => i.readOnly = !enable);
    document.getElementById('profileInput').classList.toggle('d-none', !enable);
    document.getElementById('editButton').classList.toggle('d-none', enable);
    document.getElementById('saveButton').classList.toggle('d-none', !enable);
    
    if (enable) {
        document.getElementById('profileInput').onchange = e => {
            const f = e.target.files[0];
            if (f) {
                const r = new FileReader();
                r.onload = ev => document.getElementById('profilePreview').src = ev.target.result;
                r.readAsDataURL(f);
            }
        };
    } else {
        document.getElementById('profileInput').onchange = null;
    }
}

function saveMemberChanges(id) {
    const fd = new FormData();
    fd.append('id', id);
    fd.append('full_name', document.getElementById('full_name').value);
    fd.append('email', document.getElementById('email').value);
    
    const cust = document.querySelectorAll('#customFieldsTable input[name]');
    const obj = {}; cust.forEach(i => obj[i.name] = i.value);
    fd.append('form_data', JSON.stringify(obj));
    
    const file = document.getElementById('profileInput').files[0];
    if (file) fd.append('profile_picture', file);
    
    fd.append('csrf_token', csrfToken);

    fetch('update_member_details.php', { method: 'POST', body: fd })
        .then(r => r.json())
        .then(res => { 
            alert(res.message || res.error); 
            if (!res.error) { 
                memberModal.hide();
                location.reload();
            } 
        });
}
</script>

<?php require_once 'includes/footer.php'; ?>
