<?php
require_once 'includes/functions.php';
require_once 'includes/db_config.php';

if (!is_logged_in() || !is_admin()) {
    redirect("login.php");
}

function getBenefitStatusColor($status) {
    switch ($status) {
        case 'Received':
            return 'success';
        case 'Rejected':
            return 'danger';
        case 'Pending':
            return 'warning';
        default:
            return 'secondary';
    }
}

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['log_benefit'])) {
    $member_id = $_POST['member_id'];
    $benefit_type_id = $_POST['benefit_type_id'];
    $amount = $_POST['amount'];
    $benefit_date = $_POST['benefit_date'];
    $benefit_status = $_POST['benefit_status'];

    if (!empty($member_id) && !empty($benefit_type_id) && !empty($amount) && !empty($benefit_date)) {
        $stmt = $link->prepare("INSERT INTO member_benefits (member_id, benefit_type_id, amount, benefit_date, benefit_status) VALUES (?, ?, ?, ?, ?)");
        $stmt->bind_param("iisss", $member_id, $benefit_type_id, $amount, $benefit_date, $benefit_status);
        if ($stmt->execute()) {
            $message = "Benefit logged successfully.";
            $message_type = "success";
        } else {
            $message = "Error: " . $stmt->error;
            $message_type = "danger";
        }
        $stmt->close();
    }
}

$members = [];
$sql_members = "SELECT u.id AS member_id, u.full_name, fd.data AS form_data, f.form_data AS form_structure
                FROM users u
                LEFT JOIN form_data fd ON u.id = fd.user_id
                LEFT JOIN forms f ON fd.form_id = f.id
                WHERE u.role = 'member'
                ORDER BY u.full_name ASC";
$result_members = $link->query($sql_members);
if ($result_members) {
    while ($row = $result_members->fetch_assoc()) {
        $members[] = $row;
    }
}

$benefit_types = [];
$result_types = $link->query("SELECT * FROM benefit_types ORDER BY type_name ASC");
if ($result_types) {
    while ($row = $result_types->fetch_assoc()) {
        $benefit_types[] = $row;
    }
}

$records_per_page = isset($_GET['per_page']) && is_numeric($_GET['per_page']) ? intval($_GET['per_page']) : 10;
$current_page = isset($_GET['page']) && is_numeric($_GET['page']) ? intval($_GET['page']) : 1;
$offset = ($current_page - 1) * $records_per_page;

$total_sql = "SELECT COUNT(*) FROM member_benefits";
$total_result = $link->query($total_sql);
$total_records = $total_result->fetch_row()[0];
$total_pages = ceil($total_records / $records_per_page);

$sql_benefits = "SELECT 
                    mb.*, 
                    u.full_name, 
                    u.member_id AS user_member_id,
                    bt.type_name AS benefit_type_name,
                    fd.data AS form_data,
                    f.form_data AS form_structure
                   FROM member_benefits mb
                   JOIN users u ON mb.member_id = u.id
                   LEFT JOIN form_data fd ON u.id = fd.user_id
                   LEFT JOIN forms f ON fd.form_id = f.id
                   JOIN benefit_types bt ON mb.benefit_type_id = bt.id
                   ORDER BY mb.benefit_date DESC
                   LIMIT $records_per_page OFFSET $offset";
$result_benefits = $link->query($sql_benefits);
$logged_benefits = [];
if ($result_benefits) {
    while ($row = $result_benefits->fetch_assoc()) {
        $logged_benefits[] = $row;
    }
}
?>

<?php require_once 'includes/header.php'; ?>
<div class="container-fluid">
    <div class="row">
        <?php require_once 'includes/sidebar.php'; ?>
        <main class="col-md-9 col-lg-10 main-content">
            <h2>Manage Member Benefits</h2>

            <?php if (isset($message)): ?>
                <div class="alert alert-<?php echo $message_type; ?>" role="alert">
                    <?php echo htmlspecialchars($message); ?>
                </div>
            <?php endif; ?>

            <div class="card mb-4">
                <div class="card-header">Log a New Benefit</div>
                <div class="card-body">
                    <form method="POST" action="manage_benefits.php">
                        <div class="mb-3">
                            <label for="member_id" class="form-label">Member</label>
                            <select class="form-select" id="member_id" name="member_id" required>
                                <option value="">Select a Member</option>
                                <?php foreach ($members as $member): ?>
                                    <?php
                                    $member_full_name = $member['full_name'];
                                    $form_data_array = json_decode($member['form_data'], true);
                                    $form_structure_array = json_decode($member['form_structure'], true);
                                    
                                    if ($form_data_array && $form_structure_array && isset($form_structure_array['fields'])) {
                                        $surname = '';
                                        $other_names = '';
                                        foreach ($form_structure_array['fields'] as $field) {
                                            if (strpos(strtolower($field['label']), 'surname') !== false) {
                                                $surname = $form_data_array[$field['name']] ?? '';
                                            } elseif (strpos(strtolower($field['label']), 'other names') !== false) {
                                                $other_names = $form_data_array[$field['name']] ?? '';
                                            }
                                        }
                                        $combined_name = trim($surname . ' ' . $other_names);
                                        if (!empty($combined_name)) {
                                            $member_full_name = $combined_name;
                                        }
                                    }
                                    ?>
                                    <option value="<?php echo htmlspecialchars($member['member_id']); ?>">
                                        <?php echo htmlspecialchars($member_full_name); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="mb-3">
                            <label for="benefit_type_id" class="form-label">Benefit Type</label>
                            <select class="form-select" id="benefit_type_id" name="benefit_type_id" required>
                                <option value="">Select Benefit Type</option>
                                <?php foreach ($benefit_types as $type): ?>
                                    <option value="<?php echo htmlspecialchars($type['id']); ?>">
                                        <?php echo htmlspecialchars($type['type_name']); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="mb-3">
                            <label for="amount" class="form-label">Amount</label>
                            <input type="number" class="form-control" id="amount" name="amount" step="0.01" required>
                        </div>
                        <div class="mb-3">
                            <label for="benefit_date" class="form-label">Date</label>
                            <input type="date" class="form-control" id="benefit_date" name="benefit_date" required>
                        </div>
                        <div class="mb-3">
                            <label for="benefit_status" class="form-label">Status</label>
                            <select class="form-select" id="benefit_status" name="benefit_status" required>
                                <option value="Received">Received</option>
                                <option value="Rejected">Rejected</option>
                                <option value="Pending">Pending</option>
                            </select>
                        </div>
                        <button type="submit" name="log_benefit" class="btn btn-success">Log Benefit</button>
                    </form>
                </div>
            </div>

            <div class="card">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <span>Member Beneficiaries List</span>
                    <div>
                        <a href="export_benefits.php?export=csv" class="btn btn-primary btn-sm me-2">Export to CSV</a>
                        <button class="btn btn-secondary btn-sm me-2" onclick="window.print()">Print</button>
                        <div class="d-inline-block">
                            <label for="records_per_page" class="me-2">Show</label>
                            <select id="records_per_page" class="form-select form-select-sm" onchange="window.location.href='?page=1&per_page=' + this.value">
                                <option value="10" <?php echo $records_per_page == 10 ? 'selected' : ''; ?>>10</option>
                                <option value="25" <?php echo $records_per_page == 25 ? 'selected' : ''; ?>>25</option>
                                <option value="50" <?php echo $records_per_page == 50 ? 'selected' : ''; ?>>50</option>
                                <option value="100" <?php echo $records_per_page == 100 ? 'selected' : ''; ?>>100</option>
                            </select>
                        </div>
                    </div>
                </div>
                <div class="card-body printable-area">
                    <div class="table-responsive">
                        <table class="table table-striped table-hover">
                            <thead>
                                <tr>
                                    <th>Member</th>
                                    <th>Benefit Type</th>
                                    <th>Amount</th>
                                    <th>Date</th>
                                    <th>Status</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (!empty($logged_benefits)): ?>
                                    <?php foreach ($logged_benefits as $benefit): ?>
                                        <?php
                                        $member_full_name = $benefit['full_name'];
                                        $form_data_array = json_decode($benefit['form_data'], true);
                                        $form_structure_array = json_decode($benefit['form_structure'], true);
                                        
                                        if ($form_data_array && $form_structure_array && isset($form_structure_array['fields'])) {
                                            $surname = '';
                                            $other_names = '';
                                            foreach ($form_structure_array['fields'] as $field) {
                                                if (strpos(strtolower($field['label']), 'surname') !== false) {
                                                    $surname = $form_data_array[$field['name']] ?? '';
                                                } elseif (strpos(strtolower($field['label']), 'other names') !== false) {
                                                    $other_names = $form_data_array[$field['name']] ?? '';
                                                }
                                            }
                                            $combined_name = trim($surname . ' ' . $other_names);
                                            if (!empty($combined_name)) {
                                                $member_full_name = $combined_name;
                                            }
                                        }
                                        ?>
                                        <tr>
                                            <td><?php echo htmlspecialchars($member_full_name); ?></td>
                                            <td><?php echo htmlspecialchars($benefit['benefit_type_name']); ?></td>
                                            <td>₦<?php echo number_format($benefit['amount'], 2); ?></td>
                                            <td><?php echo htmlspecialchars($benefit['benefit_date']); ?></td>
                                            <td><span class="badge bg-<?php echo getBenefitStatusColor($benefit['benefit_status']); ?>"><?php echo htmlspecialchars($benefit['benefit_status']); ?></td>
                                        </tr>
                                    <?php endforeach; ?>
                                <?php else: ?>
                                    <tr>
                                        <td colspan="5" class="text-center">No benefits have been logged.</td>
                                    </tr>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
                
                <?php if ($total_pages > 1): ?>
                    <div class="card-footer d-flex justify-content-center">
                        <nav>
                            <ul class="pagination mb-0">
                                <li class="page-item <?php echo $current_page <= 1 ? 'disabled' : ''; ?>">
                                    <a class="page-link" href="?page=<?php echo $current_page - 1; ?>&per_page=<?php echo $records_per_page; ?>">Previous</a>
                                </li>
                                <?php for ($i = 1; $i <= $total_pages; $i++): ?>
                                    <li class="page-item <?php echo $i == $current_page ? 'active' : ''; ?>">
                                        <a class="page-link" href="?page=<?php echo $i; ?>&per_page=<?php echo $records_per_page; ?>"><?php echo $i; ?></a>
                                    </li>
                                <?php endfor; ?>
                                <li class="page-item <?php echo $current_page >= $total_pages ? 'disabled' : ''; ?>">
                                    <a class="page-link" href="?page=<?php echo $current_page + 1; ?>&per_page=<?php echo $records_per_page; ?>">Next</a>
                                </li>
                            </ul>
                        </nav>
                    </div>
                <?php endif; ?>
            </div>
        </main>
    </div>
</div>

<style>
    @media print {
        body * {
            visibility: hidden;
        }
        .printable-area, .printable-area * {
            visibility: visible;
        }
        .printable-area {
            position: absolute;
            left: 0;
            top: 0;
        }
    }
</style>

<?php require_once 'includes/footer.php'; ?>