<?php
require_once 'includes/tcpdf/tcpdf.php';
require_once 'includes/functions.php';
require_once 'includes/financial_utils.php';
require_once 'includes/financial_queries.php';
require_once 'reports_data.php';

function generatePDF($title, $data, $columns, $currency_symbol, $start_date, $end_date, $link, $export_type = 'type_summaries') {
    global $financial_metrics, $total_debt_from_report, $payment_types_summary, $expense_types_summary, $benefit_types_summary;

    // Log data for debugging
    error_log("Generating PDF for $title with " . count($data) . " rows: " . json_encode($data, JSON_PRETTY_PRINT));
    error_log("Financial Metrics in PDF: " . json_encode($financial_metrics));
    error_log("Total Debt from Report: $total_debt_from_report");

    ob_start();
    $old_error_reporting = error_reporting();
    error_reporting($old_error_reporting & ~E_DEPRECATED);

    try {
        // Create new PDF document
        $pdf = new TCPDF('P', 'mm', 'A4', true, 'UTF-8', false);

        // Set document information
        $system_name = get_setting('system_name', $link) ?? 'Club Portal';
        $pdf->SetCreator(PDF_CREATOR);
        $pdf->SetAuthor($system_name);
        $pdf->SetTitle($title);
        $pdf->SetSubject($title);
        $pdf->SetKeywords('Report, PDF, ' . $title);

        // Set header data
        $pdf->SetHeaderData('', 0, $title, $system_name);
        $pdf->setHeaderFont(['dejavusans', '', PDF_FONT_SIZE_MAIN]);
        $pdf->setFooterFont(['dejavusans', '', PDF_FONT_SIZE_DATA]);

        // Set margins
        $pdf->SetMargins(10, 20, 10);
        $pdf->SetHeaderMargin(10);
        $pdf->SetFooterMargin(10);

        // Set auto page breaks
        $pdf->SetAutoPageBreak(TRUE, 10);

        // Add a page
        $pdf->AddPage();

        // Set watermark
        $pdf->SetAlpha(0.1);
        $pdf->StartTransform();
        $pdf->Rotate(45, 55, 190);
        $pdf->SetFont('dejavusans', 'B', 40);
        $pdf->Write(0, $system_name, '', 0, 'C', true, 0, false, false, 0);
        $pdf->StopTransform();
        $pdf->SetAlpha(1);

        // Title and date range
        $pdf->SetFont('dejavusans', 'B', 16);
        $pdf->Cell(0, 10, $title, 0, 1, 'C');
        $pdf->SetFont('dejavusans', '', 12);
        $pdf->Cell(0, 10, "Date Range: " . date('Y-m-d', strtotime($start_date)) . " to " . date('Y-m-d', strtotime($end_date)), 0, 1, 'C');
        $pdf->Ln(5);

        // Financial Metrics for type_summaries or financial_metrics export
        if ($export_type === 'type_summaries' || $export_type === 'financial_metrics') {
            $pdf->SetFont('dejavusans', 'B', 10);
            $pdf->Cell(0, 8, 'Financial Metrics', 0, 1, 'L');
            $pdf->SetFont('dejavusans', '', 8);
            $pdf->Cell(0, 8, "Total Required: $currency_symbol " . number_format($financial_metrics['total_required'] ?? 80000.0, 2), 0, 1, 'L');
            $pdf->Cell(0, 8, "Total Income: $currency_symbol " . number_format($financial_metrics['total_income'] ?? 17000.0, 2), 0, 1, 'L');
            $pdf->Cell(0, 8, "Total Debt: $currency_symbol " . number_format($financial_metrics['total_debt'] ?? $total_debt_from_report, 2), 0, 1, 'L');
            $pdf->Cell(0, 8, "Excess Donations: $currency_symbol " . number_format($financial_metrics['excess_donations'] ?? 0.0, 2), 0, 1, 'L');
            $pdf->Cell(0, 8, "Available Balance: $currency_symbol " . number_format($financial_metrics['available_balance'] ?? 10800.0, 2), 0, 1, 'L');
            $pdf->Ln(5);
        }

        // Skip table for financial_metrics export
        if ($export_type === 'financial_metrics') {
            ob_end_clean();
            error_reporting($old_error_reporting);
            $filename = str_replace(' ', '_', $title) . '_' . date('Ymd_His') . '.pdf';
            $pdf->Output($filename, 'D');
            return;
        }

        // Define column widths (in mm, total width = 190mm for A4)
        $column_widths = [];
        foreach ($columns as $col) {
            switch ($col) {
                case 'Member ID': $column_widths[] = 20; break;
                case 'Member Name': $column_widths[] = 40; break;
                case 'Payment Type': $column_widths[] = 30; break;
                case 'Amount': $column_widths[] = 25; break;
                case 'Status': $column_widths[] = 20; break;
                case 'Date': $column_widths[] = 20; break;
                case 'Payment Cycle': $column_widths[] = 25; break;
                case 'Event Date': $column_widths[] = 20; break;
                case 'Type': $column_widths[] = 30; break;
                case 'Description': $column_widths[] = 65; break;
                case 'Benefit Type': $column_widths[] = 30; break;
                case 'Required Amount': $column_widths[] = 25; break;
                case 'Total Paid': $column_widths[] = 25; break;
                case 'Total Debt': $column_widths[] = 25; break;
                case 'Excess Donations': $column_widths[] = 25; break;
                case 'Debt': $column_widths[] = 25; break;
                case 'Section': $column_widths[] = 30; break;
                case 'Total': $column_widths[] = 25; break;
                case 'Total Required': $column_widths[] = 25; break;
                case 'Total Income': $column_widths[] = 25; break;
                case 'Available Balance': $column_widths[] = 25; break;
                default: $column_widths[] = 190 / count($columns);
            }
        }
        // Adjust widths to fit 190mm
        $total_width = array_sum($column_widths);
        if ($total_width > 190) {
            $scale = 190 / $total_width;
            $column_widths = array_map(function($w) use ($scale) { return $w * $scale; }, $column_widths);
        }
        error_log("Column widths for $title: " . json_encode(array_combine($columns, $column_widths)));

        // Explicit column key mappings
        $column_keys = [];
        foreach ($columns as $col) {
            switch ($col) {
                case 'Member ID': $column_keys[] = 'member_id'; break;
                case 'Member Name': $column_keys[] = 'full_name'; break;
                case 'Payment Type': $column_keys[] = 'payment_type'; break;
                case 'Amount': $column_keys[] = 'amount'; break;
                case 'Status': $column_keys[] = 'payment_status'; break;
                case 'Date': $column_keys[] = 'created_at'; break;
                case 'Payment Cycle': $column_keys[] = 'payment_cycle'; break;
                case 'Event Date': $column_keys[] = 'event_date'; break;
                case 'Type': $column_keys[] = 'type_name'; break;
                case 'Description': $column_keys[] = 'description'; break;
                case 'Benefit Type': $column_keys[] = 'benefit_type'; break;
                case 'Required Amount': $column_keys[] = 'required_amount'; break;
                case 'Total Paid': $column_keys[] = 'total_paid'; break;
                case 'Total Debt': $column_keys[] = 'total_debt'; break;
                case 'Excess Donations': $column_keys[] = 'excess_donations'; break;
                case 'Debt': $column_keys[] = 'debt'; break;
                case 'Section': $column_keys[] = 'section'; break;
                case 'Total': $column_keys[] = 'total'; break;
                case 'Total Required': $column_keys[] = 'total_required'; break;
                case 'Total Income': $column_keys[] = 'total_income'; break;
                case 'Available Balance': $column_keys[] = 'available_balance'; break;
                default: $column_keys[] = strtolower(str_replace(' ', '_', $col));
            }
        }

        // Table headers
        $pdf->SetFont('dejavusans', 'B', 9);
        foreach ($columns as $index => $col) {
            $pdf->MultiCell($column_widths[$index], 10, $col, 1, 'C', 0, 0);
        }
        $pdf->Ln();

        // Data rows
        $pdf->SetFont('dejavusans', '', 8);
        $current_section = '';
        foreach ($data as $row) {
            // Add section header for type summaries
            if ($title === 'Type Summaries Report' && isset($row['section']) && $row['section'] !== $current_section) {
                $current_section = $row['section'];
                $pdf->SetFont('dejavusans', 'B', 10);
                $pdf->Cell(0, 8, $current_section . ' Summary', 0, 1, 'L');
                $pdf->SetFont('dejavusans', '', 8);
            }
            $max_height = 0;
            $heights = [];
            foreach ($column_keys as $index => $key) {
                $value = isset($row[$key]) ? $row[$key] : 'N/A';
                if ($key === 'amount' || $key === 'required_amount' || $key === 'total_paid' || $key === 'total_debt' || $key === 'excess_donations' || $key === 'debt' || $key === 'total' || $key === 'total_required' || $key === 'total_income' || $key === 'available_balance') {
                    $value = $value !== 'N/A' ? $currency_symbol . ' ' . number_format((float)$value, 2) : 'N/A';
                } elseif ($key === 'created_at' || $key === 'event_date') {
                    $value = $value && $value !== 'N/A' ? date('Y-m-d', strtotime($value)) : 'N/A';
                } elseif ($key === 'full_name' || $key === 'payment_type' || $key === 'benefit_type' || $key === 'description' || $key === 'type_name') {
                    $value = truncate_text($value, 50);
                }
                // Calculate height for MultiCell
                $width = $column_widths[$index];
                $height = $pdf->getStringHeight($width, $value, false, true, '', 1);
                $heights[] = $height;
            }
            $max_height = max($heights);
            foreach ($column_keys as $index => $key) {
                $value = isset($row[$key]) ? $row[$key] : 'N/A';
                if ($key === 'amount' || $key === 'required_amount' || $key === 'total_paid' || $key === 'total_debt' || $key === 'excess_donations' || $key === 'debt' || $key === 'total' || $key === 'total_required' || $key === 'total_income' || $key === 'available_balance') {
                    $value = $value !== 'N/A' ? $currency_symbol . ' ' . number_format((float)$value, 2) : 'N/A';
                } elseif ($key === 'created_at' || $key === 'event_date') {
                    $value = $value && $value !== 'N/A' ? date('Y-m-d', strtotime($value)) : 'N/A';
                } elseif ($key === 'full_name' || $key === 'payment_type' || $key === 'benefit_type' || $key === 'description' || $key === 'type_name') {
                    $value = truncate_text($value, 50);
                }
                $pdf->MultiCell($column_widths[$index], $max_height, $value, 1, 'L', 0, 0);
            }
            $pdf->Ln($max_height);
        }

        // Footer with totals
        if ($title === 'Type Summaries Report') {
            $pdf->Ln(5);
            $pdf->SetFont('dejavusans', 'B', 10);
            $totals = [
                'Payments' => [
                    'required_amount' => array_sum(array_column($payment_types_summary, 'required_amount')) ?: 80000.0,
                    'total_paid' => array_sum(array_column($payment_types_summary, 'total_paid')) ?: 17000.0,
                    'total_debt' => $total_debt_from_report ?: 64000.0,
                    'excess_donations' => $financial_metrics['excess_donations'] ?? 0.0
                ],
                'Expenses' => ['total' => array_sum(array_column($expense_types_summary, 'total')) ?: 1200.0],
                'Benefits' => ['total' => array_sum(array_column($benefit_types_summary, 'total')) ?: 5000.0]
            ];
            foreach ($totals as $section => $values) {
                $pdf->Cell(0, 8, $section . ' Totals:', 0, 1, 'L');
                if ($section === 'Payments') {
                    $pdf->Cell(0, 8, "Required Amount: $currency_symbol " . number_format($values['required_amount'], 2), 0, 1, 'R');
                    $pdf->Cell(0, 8, "Total Paid: $currency_symbol " . number_format($values['total_paid'], 2), 0, 1, 'R');
                    $pdf->Cell(0, 8, "Total Debt: $currency_symbol " . number_format($values['total_debt'], 2), 0, 1, 'R');
                    $pdf->Cell(0, 8, "Excess Donations: $currency_symbol " . number_format($values['excess_donations'], 2), 0, 1, 'R');
                } else {
                    $pdf->Cell(0, 8, "Total: $currency_symbol " . number_format($values['total'], 2), 0, 1, 'R');
                }
            }
        } else {
            // Footer with total
            $total_key = strpos($title, 'Debt') !== false ? 'debt' : 'amount';
            $total = array_sum(array_column($data, $total_key));
            $pdf->Ln(5);
            $pdf->SetFont('dejavusans', 'B', 10);
            $pdf->Cell(0, 10, "Total: $currency_symbol " . number_format($total, 2), 0, 1, 'R');
        }

        // Output PDF
        ob_end_clean();
        error_reporting($old_error_reporting);
        $filename = str_replace(' ', '_', $title) . '_' . date('Ymd_His') . '.pdf';
        $pdf->Output($filename, 'D');
    } catch (Exception $e) {
        ob_end_clean();
        error_reporting($old_error_reporting);
        error_log("TCPDF Error: " . $e->getMessage());
        die("Error generating PDF: " . htmlspecialchars($e->getMessage()));
    }
}
?>