<?php
// ledger_functions.php

/**
 * Executes a double-entry transaction to the financial_ledger.
 * This is the core function for recording all revenue, expenses, and transfers.
 *
 * It ensures the integrity of the ledger by recording both a Debit and a Credit
 * entry, and linking them back to the originating table and ID.
 *
 * @param mysqli $link The database connection object.
 * @param int $debit_account_id The ID of the account to be DEBITED (increase in Asset/Expense, decrease in Liability/Equity).
 * @param int $credit_account_id The ID of the account to be CREDITED (decrease in Asset/Expense, increase in Liability/Equity).
 * @param float $amount The amount of the transaction.
 * @param string $description The transaction description.
 * @param string $source_table The originating table (e.g., 'payments', 'expenses', 'fund_requests', 'member_benefits').
 * @param int $source_id The ID of the record in the source table.
 * @return bool True on success, false on failure.
 */
function record_to_ledger(mysqli $link, $debit_account_id, $credit_account_id, $amount, $description, $source_table, $source_id) {
    // 1. Basic validation
    if ($amount <= 0 || !is_numeric($amount)) {
        error_log("Attempted to record invalid amount: $amount for $source_table ID $source_id");
        return false;
    }

    // Start a transaction to ensure both entries (Debit and Credit) succeed or fail together.
    mysqli_begin_transaction($link);
    $success = true;

    // Current timestamp for date_recorded
    $date_recorded = date('Y-m-d H:i:s');

    // --- 2. Debit Entry ---
    // The account to be debited gets the full amount. Credit is 0.00.
    $sql_debit = "INSERT INTO financial_ledger 
                  (account_id, debit, credit, description, source_table, source_id, date_recorded)
                  VALUES (?, ?, 0.00, ?, ?, ?, ?)";
    
    if ($stmt = mysqli_prepare($link, $sql_debit)) {
        mysqli_stmt_bind_param($stmt, "idssis", $debit_account_id, $amount, $description, $source_table, $source_id, $date_recorded);
        if (!mysqli_stmt_execute($stmt)) {
            $success = false;
        }
        mysqli_stmt_close($stmt);
    } else {
        $success = false;
    }

    // --- 3. Credit Entry ---
    // The account to be credited gets the full amount. Debit is 0.00.
    if ($success) {
        $sql_credit = "INSERT INTO financial_ledger 
                       (account_id, debit, credit, description, source_table, source_id, date_recorded)
                       VALUES (?, 0.00, ?, ?, ?, ?, ?)";
        
        if ($stmt = mysqli_prepare($link, $sql_credit)) {
            mysqli_stmt_bind_param($stmt, "idssis", $credit_account_id, $amount, $description, $source_table, $source_id, $date_recorded);
            if (!mysqli_stmt_execute($stmt)) {
                $success = false;
            }
            mysqli_stmt_close($stmt);
        } else {
            $success = false;
        }
    }

    // --- 4. Commit or Rollback ---
    if ($success) {
        mysqli_commit($link);
        return true;
    } else {
        mysqli_rollback($link);
        // Log the error detail for debugging
        error_log("Ledger entry failed. Transaction rolled back for $source_table ID $source_id. DB Error: " . mysqli_error($link));
        return false;
    }
}

/**
 * Updates the current_balance in the bank_accounts table.
 * Assumes this is called AFTER a successful ledger entry.
 * NOTE: This function only updates the denormalized balance, 
 * the true balance is always calculated from the financial_ledger.
 *
 * @param mysqli $link The database connection object.
 * @param int $account_id The ID of the bank account to update.
 * @param float $change The amount to add (positive) or subtract (negative).
 * @return bool True on success, false on failure.
 */
function update_account_balance(mysqli $link, $account_id, $change) {
    // Basic validation
    if (!is_numeric($change) || !is_numeric($account_id) || $account_id <= 0) {
        error_log("Invalid input for update_account_balance. Account ID: $account_id, Change: $change");
        return false;
    }

    $sql = "UPDATE bank_accounts SET current_balance = current_balance + ? WHERE id = ?";
    if ($stmt = mysqli_prepare($link, $sql)) {
        mysqli_stmt_bind_param($stmt, "di", $change, $account_id);
        if (mysqli_stmt_execute($stmt)) {
            mysqli_stmt_close($stmt);
            return true;
        }
        mysqli_stmt_close($stmt);
    }
    error_log("Failed to update account balance for ID: $account_id. DB Error: " . mysqli_error($link));
    return false;
}

?>
