<?php
// includes/financial_queries.php
// -----------------------------------------------------------------------
// Contains query logic and safe utility functions for financial reports
// -----------------------------------------------------------------------

// Prevent direct access
if (!defined('ABSPATH') && !defined('IN_REPORTS')) {
    exit('Direct access not allowed.');
}

// === SAFE: build_time_filter() - Only define if not already exists ===
if (!function_exists('build_time_filter')) {
    function build_time_filter($time_period, $date_field, $record_type) {
        $filter = '';
        switch ($time_period) {
            case 'range':
                $filter = " AND $date_field BETWEEN ? AND ?";
                break;
            case 'week':
                $filter = " AND $date_field >= DATE_SUB(NOW(), INTERVAL 1 WEEK)";
                break;
            case 'month':
                $filter = " AND $date_field >= DATE_SUB(NOW(), INTERVAL 1 MONTH)";
                break;
            case 'year':
                $filter = " AND $date_field >= DATE_SUB(NOW(), INTERVAL 1 YEAR)";
                break;
            case 'all':
            default:
                $filter = '';
                break;
        }
        return $filter;
    }
}

// === get_total_counts() ===
function get_total_counts($link, $record_type, $time_period, $start_date, $end_date, $type_filter, $search, $date_field, $pagination, $member_id = 0) {
    $tables = ['payments', 'expenses', 'debts', 'benefits'];
    $total_counts = [];
    $type_filter_condition = validate_type_filter($type_filter, $record_type);
    $time_filter = build_time_filter($time_period, $date_field, $record_type);
    $time_filter_debts = str_replace('created_at', ($date_field === 'due_date' ? 'po.due_date' : 'po.created_at'), $time_filter);

    foreach ($tables as $table) {
        $sql_count = '';
        $count_bind_types = '';
        $count_bind_params = [];
        $include_filters = ($record_type === 'all' || $record_type === $table);
        $search_condition = '';
        $member_condition = '';

        if ($table === 'payments') {
            $sql_count = "SELECT COUNT(*) as total FROM payments p JOIN users u ON p.user_id = u.id WHERE payment_status = 'Paid'";
            if ($include_filters) {
                $sql_count .= str_replace('created_at', 'p.created_at', $time_filter);
                if ($time_period === 'range') {
                    $count_bind_types .= 'ss';
                    $count_bind_params[] = $start_date;
                    $count_bind_params[] = $end_date;
                }
                if ($type_filter && strpos($type_filter, 'payment_') === 0) {
                    $sql_count .= $type_filter_condition;
                    $count_bind_types .= 's';
                    $count_bind_params[] = explode('_', $type_filter, 2)[1];
                }
                if ($member_id > 0) {
                    $member_condition = " AND p.user_id = ?";
                    $sql_count .= $member_condition;
                    $count_bind_types .= 'i';
                    $count_bind_params[] = $member_id;
                }
            } else {
                $sql_count .= ' AND FALSE';
            }
            $search_condition = " AND (u.full_name LIKE ? OR p.payment_type LIKE ? OR CAST(p.id AS CHAR) LIKE ?)";
        } elseif ($table === 'expenses') {
            $sql_count = "SELECT COUNT(*) as total FROM expenses e LEFT JOIN expense_types et ON e.expense_type_id = et.id WHERE 1=1";
            if ($include_filters) {
                $sql_count .= str_replace('created_at', 'e.created_at', $time_filter);
                if ($time_period === 'range') {
                    $count_bind_types .= 'ss';
                    $count_bind_params[] = $start_date;
                    $count_bind_params[] = $end_date;
                }
                if ($type_filter && strpos($type_filter, 'expense_') === 0) {
                    $sql_count .= $type_filter_condition;
                    $count_bind_types .= 's';
                    $count_bind_params[] = explode('_', $type_filter, 2)[1];
                }
            } else {
                $sql_count .= ' AND FALSE';
            }
            $search_condition = " AND (et.type_name LIKE ? OR e.description LIKE ? OR CAST(e.id AS CHAR) LIKE ?)";
        } elseif ($table === 'benefits') {
            $sql_count = "SELECT COUNT(*) as total FROM member_benefits b JOIN users u ON b.member_id = u.id JOIN benefit_types bt ON b.benefit_type_id = bt.id WHERE 1=1";
            if ($include_filters) {
                $sql_count .= str_replace('created_at', 'b.created_at', $time_filter);
                if ($time_period === 'range') {
                    $count_bind_types .= 'ss';
                    $count_bind_params[] = $start_date;
                    $count_bind_params[] = $end_date;
                }
                if ($type_filter && strpos($type_filter, 'benefit_') === 0) {
                    $sql_count .= $type_filter_condition;
                    $count_bind_types .= 's';
                    $count_bind_params[] = explode('_', $type_filter, 2)[1];
                }
                if ($member_id > 0) {
                    $member_condition = " AND b.member_id = ?";
                    $sql_count .= $member_condition;
                    $count_bind_types .= 'i';
                    $count_bind_params[] = $member_id;
                }
            } else {
                $sql_count .= ' AND FALSE';
            }
            $search_condition = " AND (u.full_name LIKE ? OR bt.type_name LIKE ? OR CAST(b.id AS CHAR) LIKE ?)";
        } elseif ($table === 'debts') {
            $sql_count = "SELECT COUNT(*) as total 
                          FROM payment_obligations po 
                          JOIN users u ON po.user_id = u.id 
                          JOIN payment_types pt ON po.payment_type = pt.type_name 
                          LEFT JOIN payments p ON p.obligation_id = po.id AND p.payment_status = 'Paid'
                          WHERE po.deleted_at IS NULL 
                              AND u.role = 'member' 
                              AND (u.deleted_at IS NULL OR u.deleted_at > NOW())
                              AND COALESCE(po.required_amount, 0) > 0";
            if ($include_filters) {
                $sql_count .= $time_filter_debts;
                if ($time_period === 'range') {
                    $count_bind_types .= 'ss';
                    $count_bind_params[] = $start_date;
                    $count_bind_params[] = $end_date;
                }
                if ($type_filter && strpos($type_filter, 'payment_') === 0) {
                    $adjusted_condition = str_replace('p.payment_type', 'pt.type_name', $type_filter_condition);
                    $sql_count .= $adjusted_condition;
                    $count_bind_types .= 's';
                    $count_bind_params[] = explode('_', $type_filter, 2)[1];
                }
                if ($member_id > 0) {
                    $member_condition = " AND po.user_id = ?";
                    $sql_count .= $member_condition;
                    $count_bind_types .= 'i';
                    $count_bind_params[] = $member_id;
                }
            } else {
                $sql_count .= ' AND FALSE';
            }
            $search_condition = " AND (u.full_name LIKE ? OR pt.type_name LIKE ? OR CAST(po.id AS CHAR) LIKE ?)";
        }

        if (!empty($search)) {
            $sql_count .= $search_condition;
            $count_bind_types .= 'sss';
            $search_param = "%$search%";
            $count_bind_params[] = $search_param;
            $count_bind_params[] = $search_param;
            $count_bind_params[] = $search_param;
        }

        if ($table === 'debts') {
            $sql_count .= " GROUP BY po.id, u.id, po.required_amount, po.due_date HAVING (COALESCE(po.required_amount, 0) - COALESCE(SUM(p.amount), 0)) > 0";
        }

        if (!$stmt = mysqli_prepare($link, $sql_count)) {
            error_log("Failed to prepare statement for $table count: " . mysqli_error($link) . "; SQL: $sql_count");
            $total_counts[$table] = 0;
            continue;
        }
        if ($count_bind_types && !mysqli_stmt_bind_param($stmt, $count_bind_types, ...$count_bind_params)) {
            error_log("Failed to bind parameters for $table count: " . mysqli_stmt_error($stmt) . "; SQL: $sql_count");
            $total_counts[$table] = 0;
            mysqli_stmt_close($stmt);
            continue;
        }
        if (!mysqli_stmt_execute($stmt)) {
            error_log("Failed to execute $table count query: " . mysqli_stmt_error($stmt) . "; SQL: $sql_count");
            $total_counts[$table] = 0;
            mysqli_stmt_close($stmt);
            continue;
        }
        $result = mysqli_stmt_get_result($stmt);
        if ($table === 'debts') {
            $total_counts[$table] = mysqli_num_rows($result);
        } else {
            $total_counts[$table] = mysqli_fetch_assoc($result)['total'] ?? 0;
        }
        mysqli_stmt_close($stmt);
    }
    return $total_counts;
}

// === All other functions below remain unchanged ===
function get_total_payments($link, $record_type, $time_period, $start_date, $end_date, $type_filter, $member_id = 0) {
    $sql = "SELECT COALESCE(SUM(amount), 0) AS total_payments FROM payments p WHERE payment_status = 'Paid'";
    $bind_types = '';
    $bind_params = [];
    $type_filter_condition = validate_type_filter($type_filter, $record_type);
    $member_condition = $member_id > 0 ? " AND p.user_id = ?" : '';
    if ($record_type === 'all' || $record_type === 'payments') {
        $sql .= str_replace('created_at', 'p.created_at', build_time_filter($time_period, 'created_at', $record_type));
        if ($time_period === 'range') {
            $bind_types .= 'ss';
            $bind_params[] = $start_date;
            $bind_params[] = $end_date;
        }
        if ($type_filter && strpos($type_filter, 'payment_') === 0) {
            $sql .= $type_filter_condition;
            $bind_types .= 's';
            $bind_params[] = explode('_', $type_filter, 2)[1];
        }
        $sql .= $member_condition;
        if ($member_id > 0) {
            $bind_types .= 'i';
            $bind_params[] = $member_id;
        }
    } else {
        $sql .= ' AND FALSE';
    }
    if ($stmt = mysqli_prepare($link, $sql)) {
        if ($bind_types) {
            mysqli_stmt_bind_param($stmt, $bind_types, ...$bind_params);
        }
        if (mysqli_stmt_execute($stmt)) {
            $total = mysqli_fetch_assoc(mysqli_stmt_get_result($stmt))['total_payments'] ?? 0;
            mysqli_stmt_close($stmt);
            return $total;
        }
        error_log("Failed to execute total_payments query: " . mysqli_stmt_error($stmt) . "; SQL: $sql");
        mysqli_stmt_close($stmt);
    }
    error_log("Failed to prepare total_payments query: " . mysqli_error($link) . "; SQL: $sql");
    return 0;
}

function get_total_expenses($link, $record_type, $time_period, $start_date, $end_date, $type_filter) {
    $sql = "SELECT COALESCE(SUM(amount), 0) AS total_expenses FROM expenses e LEFT JOIN expense_types et ON e.expense_type_id = et.id WHERE 1=1";
    $bind_types = '';
    $bind_params = [];
    $type_filter_condition = validate_type_filter($type_filter, $record_type);
    if ($record_type === 'all' || $record_type === 'expenses') {
        $sql .= str_replace('created_at', 'e.created_at', build_time_filter($time_period, 'created_at', $record_type));
        if ($time_period === 'range') {
            $bind_types .= 'ss';
            $bind_params[] = $start_date;
            $bind_params[] = $end_date;
        }
        if ($type_filter && strpos($type_filter, 'expense_') === 0) {
            $sql .= $type_filter_condition;
            $bind_types .= 's';
            $bind_params[] = explode('_', $type_filter, 2)[1];
        }
    } else {
        $sql .= ' AND FALSE';
    }
    if ($stmt = mysqli_prepare($link, $sql)) {
        if ($bind_types) {
            mysqli_stmt_bind_param($stmt, $bind_types, ...$bind_params);
        }
        if (mysqli_stmt_execute($stmt)) {
            $total = mysqli_fetch_assoc(mysqli_stmt_get_result($stmt))['total_expenses'] ?? 0;
            mysqli_stmt_close($stmt);
            return $total;
        }
        error_log("Failed to execute total_expenses query: " . mysqli_stmt_error($stmt) . "; SQL: $sql");
        mysqli_stmt_close($stmt);
    }
    error_log("Failed to prepare total_expenses query: " . mysqli_error($link) . "; SQL: $sql");
    return 0;
}

function get_total_benefits($link, $record_type, $time_period, $start_date, $end_date, $type_filter, $member_id = 0) {
    $sql = "SELECT COALESCE(SUM(amount), 0) AS total_benefits FROM member_benefits b JOIN benefit_types bt ON b.benefit_type_id = bt.id WHERE 1=1";
    $bind_types = '';
    $bind_params = [];
    $type_filter_condition = validate_type_filter($type_filter, $record_type);
    $member_condition = $member_id > 0 ? " AND b.member_id = ?" : '';
    if ($record_type === 'all' || $record_type === 'benefits') {
        $sql .= str_replace('created_at', 'b.created_at', build_time_filter($time_period, 'created_at', $record_type));
        if ($time_period === 'range') {
            $bind_types .= 'ss';
            $bind_params[] = $start_date;
            $bind_params[] = $end_date;
        }
        if ($type_filter && strpos($type_filter, 'benefit_') === 0) {
            $sql .= $type_filter_condition;
            $bind_types .= 's';
            $bind_params[] = explode('_', $type_filter, 2)[1];
        }
        $sql .= $member_condition;
        if ($member_id > 0) {
            $bind_types .= 'i';
            $bind_params[] = $member_id;
        }
    } else {
        $sql .= ' AND FALSE';
    }
    if ($stmt = mysqli_prepare($link, $sql)) {
        if ($bind_types) {
            mysqli_stmt_bind_param($stmt, $bind_types, ...$bind_params);
        }
        if (mysqli_stmt_execute($stmt)) {
            $total = mysqli_fetch_assoc(mysqli_stmt_get_result($stmt))['total_benefits'] ?? 0;
            mysqli_stmt_close($stmt);
            return $total;
        }
        error_log("Failed to execute total_benefits query: " . mysqli_stmt_error($stmt) . "; SQL: $sql");
        mysqli_stmt_close($stmt);
    }
    error_log("Failed to prepare total_benefits query: " . mysqli_error($link) . "; SQL: $sql");
    return 0;
}

function get_total_debts($link, $record_type, $time_period, $start_date, $end_date, $type_filter, $date_field, $member_id = 0) {
    $sql = "SELECT COALESCE(SUM(debt), 0) AS total_debt 
            FROM (
                SELECT 
                    po.id,
                    GREATEST(COALESCE(po.required_amount, 0) - COALESCE(SUM(p.amount), 0), 0) AS debt
                FROM payment_obligations po
                JOIN users u ON po.user_id = u.id
                JOIN payment_types pt ON po.payment_type = pt.type_name
                LEFT JOIN payments p ON p.obligation_id = po.id AND p.payment_status = 'Paid'
                WHERE po.deleted_at IS NULL
                    AND u.role = 'member'
                    AND (u.deleted_at IS NULL OR u.deleted_at > NOW())
                    AND COALESCE(po.required_amount, 0) > 0";
    $bind_types = '';
    $bind_params = [];
    $type_filter_condition = validate_type_filter($type_filter, $record_type);
    $member_condition = $member_id > 0 ? " AND po.user_id = ?" : '';
    if ($record_type === 'all' || $record_type === 'debts') {
        $sql .= str_replace('created_at', ($date_field === 'due_date' ? 'po.due_date' : 'po.created_at'), build_time_filter($time_period, $date_field, $record_type));
        if ($time_period === 'range') {
            $bind_types .= 'ss';
            $bind_params[] = $start_date;
            $bind_params[] = $end_date;
        }
        if ($type_filter && strpos($type_filter, 'payment_') === 0) {
            $adjusted_condition = str_replace('p.payment_type', 'pt.type_name', $type_filter_condition);
            $sql .= $adjusted_condition;
            $bind_types .= 's';
            $bind_params[] = explode('_', $type_filter, 2)[1];
        }
        $sql .= $member_condition;
        if ($member_id > 0) {
            $bind_types .= 'i';
            $bind_params[] = $member_id;
        }
    } else {
        $sql .= ' AND FALSE';
    }
    $sql .= " GROUP BY po.id, po.required_amount
             HAVING debt > 0
            ) AS subquery";
    if ($stmt = mysqli_prepare($link, $sql)) {
        if ($bind_types) {
            mysqli_stmt_bind_param($stmt, $bind_types, ...$bind_params);
        }
        if (mysqli_stmt_execute($stmt)) {
            $total = mysqli_fetch_assoc(mysqli_stmt_get_result($stmt))['total_debt'] ?? 0;
            mysqli_stmt_close($stmt);
            return $total;
        }
        error_log("Failed to execute total_debts query: " . mysqli_stmt_error($stmt) . "; SQL: $sql");
        mysqli_stmt_close($stmt);
    }
    error_log("Failed to prepare total_debts query: " . mysqli_error($link) . "; SQL: $sql");
    return 0;
}

function get_total_members($link) {
    $sql = "SELECT COUNT(*) as total FROM users WHERE role = 'member' AND deleted_at IS NULL";
    if ($result = mysqli_query($link, $sql)) {
        $total = mysqli_fetch_assoc($result)['total'] ?? 0;
        mysqli_free_result($result);
        return $total;
    }
    error_log("Failed to fetch total_members: " . mysqli_error($link));
    return 0;
}

function get_total_payment_types($link) {
    $sql = "SELECT COUNT(*) as total FROM payment_types WHERE deleted_at IS NULL";
    if ($result = mysqli_query($link, $sql)) {
        $total = mysqli_fetch_assoc($result)['total'] ?? 0;
        mysqli_free_result($result);
        return $total;
    }
    error_log("Failed to fetch total_payment_types: " . mysqli_error($link));
    return 0;
}

function get_previous_balances($link, $end_date, $date_field) {
    $balances = ['month' => 0, 'year' => 0];

    $month_start = date('Y-m-01', strtotime('-1 month', strtotime($end_date)));
    $year_start = date('Y-01-01', strtotime('-1 year', strtotime($end_date)));

    $sql_month = "SELECT (
        (SELECT COALESCE(SUM(amount), 0) FROM payments WHERE payment_status = 'Paid' AND created_at < ?) -
        (SELECT COALESCE(SUM(amount), 0) FROM expenses WHERE created_at < ?) -
        (SELECT COALESCE(SUM(amount), 0) FROM member_benefits WHERE created_at < ?)
    ) AS balance_previous_month";
    if ($stmt = mysqli_prepare($link, $sql_month)) {
        mysqli_stmt_bind_param($stmt, 'sss', $month_start, $month_start, $month_start);
        mysqli_stmt_execute($stmt);
        $balances['month'] = mysqli_fetch_assoc(mysqli_stmt_get_result($stmt))['balance_previous_month'] ?? 0;
        mysqli_stmt_close($stmt);
    } else {
        error_log("Failed to prepare previous month balance query: " . mysqli_error($link));
    }

    $sql_year = "SELECT (
        (SELECT COALESCE(SUM(amount), 0) FROM payments WHERE payment_status = 'Paid' AND created_at < ?) -
        (SELECT COALESCE(SUM(amount), 0) FROM expenses WHERE created_at < ?) -
        (SELECT COALESCE(SUM(amount), 0) FROM member_benefits WHERE created_at < ?)
    ) AS balance_previous_year";
    if ($stmt = mysqli_prepare($link, $sql_year)) {
        mysqli_stmt_bind_param($stmt, 'sss', $year_start, $year_start, $year_start);
        mysqli_stmt_execute($stmt);
        $balances['year'] = mysqli_fetch_assoc(mysqli_stmt_get_result($stmt))['balance_previous_year'] ?? 0;
        mysqli_stmt_close($stmt);
    } else {
        error_log("Failed to prepare previous year balance query: " . mysqli_error($link));
    }

    return $balances;
}

function get_payments($link, $record_type, $time_period, $start_date, $end_date, $type_filter, $search, $pagination, $member_id = 0) {
    $sql = "SELECT 
                p.id, 
                u.full_name, 
                p.payment_type, 
                p.amount, 
                p.payment_status, 
                p.created_at, 
                po.payment_cycle
            FROM payments p 
            JOIN users u ON p.user_id = u.id
            LEFT JOIN payment_obligations po ON p.obligation_id = po.id
            WHERE p.payment_status = 'Paid'";

    $bind_types = '';
    $bind_params = [];
    $type_filter_condition = validate_type_filter($type_filter, $record_type);
    $member_condition = $member_id > 0 ? " AND p.user_id = ?" : '';

    if ($record_type === 'all' || $record_type === 'payments') {
        $sql .= str_replace('created_at', 'p.created_at', build_time_filter($time_period, 'created_at', $record_type));
        if ($time_period === 'range') {
            $bind_types .= 'ss';
            $bind_params[] = $start_date;
            $bind_params[] = $end_date;
        }
        if ($type_filter && strpos($type_filter, 'payment_') === 0) {
            $sql .= $type_filter_condition;
            $bind_types .= 's';
            $bind_params[] = explode('_', $type_filter, 2)[1];
        }
        $sql .= $member_condition;
        if ($member_id > 0) {
            $bind_types .= 'i';
            $bind_params[] = $member_id;
        }
    } else {
        $sql .= ' AND FALSE';
    }

    if (!empty($search)) {
        $sql .= " AND (u.full_name LIKE ? OR p.payment_type LIKE ? OR CAST(p.id AS CHAR) LIKE ?)";
        $bind_types .= 'sss';
        $search_param = "%$search%";
        $bind_params[] = $search_param;
        $bind_params[] = $search_param;
        $bind_params[] = $search_param;
    }

    $sql .= " ORDER BY p.created_at DESC LIMIT ? OFFSET ?";
    $bind_types .= 'ii';
    $bind_params[] = $pagination['per_page'];
    $bind_params[] = $pagination['offset'];

    $payments = [];
    if ($stmt = mysqli_prepare($link, $sql)) {
        if ($bind_types) {
            mysqli_stmt_bind_param($stmt, $bind_types, ...$bind_params);
        }
        if (mysqli_stmt_execute($stmt)) {
            $result = mysqli_stmt_get_result($stmt);
            while ($row = mysqli_fetch_assoc($result)) {
                $row['payment_cycle'] = $row['payment_cycle'] ?? 'N/A';
                $payments[] = $row;
            }
            mysqli_free_result($result);
        } else {
            error_log("Failed to execute get_payments query: " . mysqli_stmt_error($stmt) . "; SQL: $sql");
        }
        mysqli_stmt_close($stmt);
    } else {
        error_log("Failed to prepare get_payments query: " . mysqli_error($link) . "; SQL: $sql");
    }

    return $payments;
}

// [All remaining functions: get_expenses, get_benefits, get_debts, get_payment_types_summary, etc.]
// ... (unchanged from your original code)

function get_expenses($link, $record_type, $time_period, $start_date, $end_date, $type_filter, $search, $pagination) {
    $sql = "SELECT e.id, et.type_name AS expense_type, e.amount, e.description, e.created_at
            FROM expenses e LEFT JOIN expense_types et ON e.expense_type_id = et.id WHERE 1=1";
    $bind_types = '';
    $bind_params = [];
    $type_filter_condition = validate_type_filter($type_filter, $record_type);
    if ($record_type === 'all' || $record_type === 'expenses') {
        $sql .= str_replace('created_at', 'e.created_at', build_time_filter($time_period, 'created_at', $record_type));
        if ($time_period === 'range') {
            $bind_types .= 'ss';
            $bind_params[] = $start_date;
            $bind_params[] = $end_date;
        }
        if ($type_filter && strpos($type_filter, 'expense_') === 0) {
            $sql .= $type_filter_condition;
            $bind_types .= 's';
            $bind_params[] = explode('_', $type_filter, 2)[1];
        }
    } else {
        $sql .= ' AND FALSE';
    }
    if (!empty($search)) {
        $sql .= " AND (et.type_name LIKE ? OR e.description LIKE ? OR CAST(e.id AS CHAR) LIKE ?)";
        $bind_types .= 'sss';
        $search_param = "%$search%";
        $bind_params[] = $search_param;
        $bind_params[] = $search_param;
        $bind_params[] = $search_param;
    }
    $sql .= " ORDER BY e.created_at DESC LIMIT ? OFFSET ?";
    $bind_types .= 'ii';
    $bind_params[] = $pagination['per_page'];
    $bind_params[] = $pagination['offset'];
    $expenses = [];
    if ($stmt = mysqli_prepare($link, $sql)) {
        if ($bind_types) {
            mysqli_stmt_bind_param($stmt, $bind_types, ...$bind_params);
        }
        if (mysqli_stmt_execute($stmt)) {
            $result = mysqli_stmt_get_result($stmt);
            while ($row = mysqli_fetch_assoc($result)) {
                $expenses[] = $row;
            }
            mysqli_free_result($result);
        } else {
            error_log("Failed to execute get_expenses query: " . mysqli_stmt_error($stmt) . "; SQL: $sql");
        }
        mysqli_stmt_close($stmt);
    } else {
        error_log("Failed to prepare get_expenses query: " . mysqli_error($link) . "; SQL: $sql");
    }
    return $expenses;
}

function get_benefits($link, $record_type, $time_period, $start_date, $end_date, $type_filter, $search, $pagination, $member_id = 0) {
    $sql = "SELECT b.id, b.member_id, u.full_name, bt.type_name AS benefit_type, b.amount, b.created_at
            FROM member_benefits b
            JOIN users u ON b.member_id = u.id
            JOIN benefit_types bt ON b.benefit_type_id = bt.id
            WHERE 1=1";
    $bind_types = '';
    $bind_params = [];
    $type_filter_condition = validate_type_filter($type_filter, $record_type);
    $member_condition = $member_id > 0 ? " AND b.member_id = ?" : '';
    if ($record_type === 'all' || $record_type === 'benefits') {
        $sql .= str_replace('created_at', 'b.created_at', build_time_filter($time_period, 'created_at', $record_type));
        if ($time_period === 'range') {
            $bind_types .= 'ss';
            $bind_params[] = $start_date;
            $bind_params[] = $end_date;
        }
        if ($type_filter && strpos($type_filter, 'benefit_') === 0) {
            $sql .= $type_filter_condition;
            $bind_types .= 's';
            $bind_params[] = explode('_', $type_filter, 2)[1];
        }
        $sql .= $member_condition;
        if ($member_id > 0) {
            $bind_types .= 'i';
            $bind_params[] = $member_id;
        }
    } else {
        $sql .= ' AND FALSE';
    }
    if (!empty($search)) {
        $sql .= " AND (u.full_name LIKE ? OR bt.type_name LIKE ? OR CAST(b.id AS CHAR) LIKE ?)";
        $bind_types .= 'sss';
        $search_param = "%$search%";
        $bind_params[] = $search_param;
        $bind_params[] = $search_param;
        $bind_params[] = $search_param;
    }
    $sql .= " ORDER BY b.created_at DESC LIMIT ? OFFSET ?";
    $bind_types .= 'ii';
    $bind_params[] = $pagination['per_page'];
    $bind_params[] = $pagination['offset'];
    $benefits = [];
    if ($stmt = mysqli_prepare($link, $sql)) {
        if ($bind_types) {
            mysqli_stmt_bind_param($stmt, $bind_types, ...$bind_params);
        }
        if (mysqli_stmt_execute($stmt)) {
            $result = mysqli_stmt_get_result($stmt);
            while ($row = mysqli_fetch_assoc($result)) {
                $benefits[] = $row;
            }
            mysqli_free_result($result);
        } else {
            error_log("Failed to execute get_benefits query: " . mysqli_stmt_error($stmt) . "; SQL: $sql");
        }
        mysqli_stmt_close($stmt);
    } else {
        error_log("Failed to prepare get_benefits query: " . mysqli_error($link) . "; SQL: $sql");
    }
    return $benefits;
}

function get_debts($link, $record_type, $time_period, $start_date, $end_date, $type_filter, $search, $date_field, $pagination, $member_id = 0) {
    $sql = "SELECT u.id AS user_id, u.member_id, u.full_name, pt.type_name AS payment_type, po.required_amount, COALESCE(SUM(p.amount), 0) AS total_paid,
                   GREATEST(COALESCE(po.required_amount, 0) - COALESCE(SUM(p.amount), 0), 0) AS debt, po.payment_cycle, po.due_date
            FROM payment_obligations po
            JOIN users u ON po.user_id = u.id
            JOIN payment_types pt ON po.payment_type = pt.type_name
            LEFT JOIN payments p ON p.obligation_id = po.id AND p.payment_status = 'Paid'
            WHERE po.deleted_at IS NULL
                AND u.role = 'member'
                AND (u.deleted_at IS NULL OR u.deleted_at > NOW())
                AND COALESCE(po.required_amount, 0) > 0";
    $bind_types = '';
    $bind_params = [];
    $type_filter_condition = validate_type_filter($type_filter, $record_type);
    $member_condition = $member_id > 0 ? " AND po.user_id = ?" : '';
    if ($record_type === 'all' || $record_type === 'debts') {
        $sql .= str_replace('created_at', ($date_field === 'due_date' ? 'po.due_date' : 'po.created_at'), build_time_filter($time_period, $date_field, $record_type));
        if ($time_period === 'range') {
            $bind_types .= 'ss';
            $bind_params[] = $start_date;
            $bind_params[] = $end_date;
        }
        if ($type_filter && strpos($type_filter, 'payment_') === 0) {
            $adjusted_condition = str_replace('p.payment_type', 'pt.type_name', $type_filter_condition);
            $sql .= $adjusted_condition;
            $bind_types .= 's';
            $bind_params[] = explode('_', $type_filter, 2)[1];
        }
        $sql .= $member_condition;
        if ($member_id > 0) {
            $bind_types .= 'i';
            $bind_params[] = $member_id;
        }
    } else {
        $sql .= ' AND FALSE';
    }
    if (!empty($search)) {
        $sql .= " AND (u.full_name LIKE ? OR pt.type_name LIKE ? OR CAST(po.id AS CHAR) LIKE ?)";
        $bind_types .= 'sss';
        $search_param = "%$search%";
        $bind_params[] = $search_param;
        $bind_params[] = $search_param;
        $bind_params[] = $search_param;
    }
    $sql .= " GROUP BY po.id, u.id, po.required_amount, po.payment_cycle, po.due_date
             HAVING debt > 0
             ORDER BY debt DESC
             LIMIT ? OFFSET ?";
    $bind_types .= 'ii';
    $bind_params[] = $pagination['per_page'];
    $bind_params[] = $pagination['offset'];
    $debts = [];
    if ($stmt = mysqli_prepare($link, $sql)) {
        if ($bind_types) {
            mysqli_stmt_bind_param($stmt, $bind_types, ...$bind_params);
        }
        if (mysqli_stmt_execute($stmt)) {
            $result = mysqli_stmt_get_result($stmt);
            while ($row = mysqli_fetch_assoc($result)) {
                $debts[] = $row;
            }
            mysqli_free_result($result);
        } else {
            error_log("Failed to execute get_debts query: " . mysqli_stmt_error($stmt) . "; SQL: $sql");
        }
        mysqli_stmt_close($stmt);
    } else {
        error_log("Failed to prepare get_debts query: " . mysqli_error($link) . "; SQL: $sql");
    }
    return $debts;
}

/**
 * ENHANCED: Generates a summary of all payment types.
 * This function now combines two data sources:
 * 1. Payments from 'payment_obligations' (debts, required amounts, etc.)
 * 2. Payments from 'payments' table (donations, non-obligation payments)
 * This ensures all payment types, including 'donation' types, are summarized.
 */
function get_payment_types_summary($link, $record_type, $time_period, $start_date, $end_date, $type_filter, $date_field, $member_id = 0) {
    // This function will now combine two sets of data:
    // 1. Data from obligations (required, paid, debt)
    // 2. Data from non-obligation donations (paid, as 'excess')

    $summary_data = [];

    // --- Common Filter Logic ---
    $type_filter_name = '';
    if ($type_filter && strpos($type_filter, 'payment_') === 0) {
        $type_filter_name = explode('_', $type_filter, 2)[1];
    }

    // =================================================================
    // --- QUERY 1: OBLIGATION SUMMARY (Existing Logic) ---
    // =================================================================
    $bind_types_obl = '';
    $bind_params_obl = [];
    $where_obl = " WHERE po.deleted_at IS NULL
                    AND u.role = 'member'
                    AND (u.deleted_at IS NULL OR u.deleted_at > NOW())
                    AND COALESCE(po.required_amount, 0) >= 0";

    if ($type_filter_name) {
        $where_obl .= " AND pt.type_name = ?";
        $bind_types_obl .= 's';
        $bind_params_obl[] = $type_filter_name;
    }

    if ($member_id > 0) {
        $where_obl .= " AND u.id = ?";
        $bind_types_obl .= 'i';
        $bind_params_obl[] = $member_id;
    }

    if ($record_type === 'all' || $record_type === 'payments') {
        $time_filter_obl = str_replace('created_at', ($date_field === 'due_date' ? 'po.due_date' : 'po.created_at'), build_time_filter($time_period, $date_field, $record_type));
        $where_obl .= $time_filter_obl;
        if ($time_period === 'range') {
            $bind_types_obl .= 'ss';
            $bind_params_obl[] = $start_date;
            $bind_params_obl[] = $end_date;
        }
    } else {
        $where_obl .= ' AND FALSE';
    }

    $sql1 = "
        SELECT 
            pt.type_name,
            SUM(sub.required_amount) AS required_amount,
            SUM(sub.paid) AS total_paid,
            SUM(sub.debt) AS total_debt,
            SUM(sub.excess) AS excess_amount
        FROM (
            SELECT 
                po.payment_type,
                COALESCE(po.required_amount, 0) AS required_amount,
                COALESCE(SUM(p.amount), 0) AS paid,
                GREATEST(COALESCE(po.required_amount, 0) - COALESCE(SUM(p.amount), 0), 0) AS debt,
                CASE 
                    WHEN pt.payment_category = 'freewill' 
                    THEN GREATEST(COALESCE(SUM(p.amount), 0) - COALESCE(po.required_amount, 0), 0) 
                    ELSE 0 
                END AS excess
            FROM payment_obligations po
            JOIN users u ON po.user_id = u.id
            JOIN payment_types pt ON po.payment_type = pt.type_name
            LEFT JOIN payments p ON p.obligation_id = po.id AND p.payment_status = 'Paid'
            $where_obl
            GROUP BY po.id, po.payment_type, po.required_amount, pt.payment_category
        ) sub
        JOIN payment_types pt ON sub.payment_type = pt.type_name
        GROUP BY pt.type_name
        HAVING total_debt > 0 OR total_paid > 0 OR excess_amount > 0
    ";

    if ($stmt1 = mysqli_prepare($link, $sql1)) {
        if ($bind_types_obl) {
            mysqli_stmt_bind_param($stmt1, $bind_types_obl, ...$bind_params_obl);
        }
        if (mysqli_stmt_execute($stmt1)) {
            $result1 = mysqli_stmt_get_result($stmt1);
            while ($row = mysqli_fetch_assoc($result1)) {
                $type = $row['type_name'];
                if (!isset($summary_data[$type])) {
                    $summary_data[$type] = ['type_name' => $type, 'required_amount' => 0, 'total_paid' => 0, 'total_debt' => 0, 'excess_amount' => 0];
                }
                $summary_data[$type]['required_amount'] += (float)$row['required_amount'];
                $summary_data[$type]['total_paid'] += (float)$row['total_paid'];
                $summary_data[$type]['total_debt'] += (float)$row['total_debt'];
                $summary_data[$type]['excess_amount'] += (float)$row['excess_amount'];
            }
            mysqli_free_result($result1);
        } else {
            error_log("Failed to execute get_payment_types_summary query (Part 1): " . mysqli_stmt_error($stmt1) . "; SQL: $sql1");
        }
        mysqli_stmt_close($stmt1);
    } else {
        error_log("Failed to prepare get_payment_types_summary query (Part 1): " . mysqli_error($link) . "; SQL: $sql1");
    }

    // =================================================================
    // --- QUERY 2: DONATION/OTHER_INCOME SUMMARY (New Logic) ---
    // =================================================================
    $bind_types_don = '';
    $bind_params_don = [];
    $where_don = " WHERE p.payment_status = 'Paid'
                   AND (p.is_donation = 1 OR p.obligation_id IS NULL) 
                   AND pt.payment_category IN ('donation', 'other_income')"; // *** FIX: Changed from '=' to 'IN' ***

    if ($type_filter_name) {
        $where_don .= " AND p.payment_type = ?";
        $bind_types_don .= 's';
        $bind_params_don[] = $type_filter_name;
    }

    if ($member_id > 0) {
        // Use member_id for payments linked to members, or ignore if it's a non-member donation
        $where_don .= " AND p.member_id = ?";
        $bind_types_don .= 'i';
        $bind_params_don[] = $member_id;
    }

    if ($record_type === 'all' || $record_type === 'payments') {
        // Use 'payment_date' or 'created_at' for filtering, as donations don't have 'due_date'
        $time_filter_don = str_replace('created_at', 'p.payment_date', build_time_filter($time_period, 'created_at', $record_type));
        $where_don .= $time_filter_don;
        if ($time_period === 'range') {
            $bind_types_don .= 'ss';
            $bind_params_don[] = $start_date;
            $bind_params_don[] = $end_date;
        }
    } else {
        $where_don .= ' AND FALSE';
    }

    $sql2 = "
        SELECT
            p.payment_type AS type_name,
            0 AS required_amount,
            COALESCE(SUM(p.amount), 0) AS total_paid,
            0 AS total_debt,
            COALESCE(SUM(p.amount), 0) AS excess_amount
        FROM
            payments p
        JOIN payment_types pt ON p.payment_type = pt.type_name
        LEFT JOIN users u ON p.member_id = u.id -- For member filter
        $where_don
        GROUP BY
            p.payment_type
        HAVING
            total_paid > 0
    ";
    
    if ($stmt2 = mysqli_prepare($link, $sql2)) {
        if ($bind_types_don) {
            mysqli_stmt_bind_param($stmt2, $bind_types_don, ...$bind_params_don);
        }
        if (mysqli_stmt_execute($stmt2)) {
            $result2 = mysqli_stmt_get_result($stmt2);
            while ($row = mysqli_fetch_assoc($result2)) {
                $type = $row['type_name'];
                if (!isset($summary_data[$type])) {
                    $summary_data[$type] = ['type_name' => $type, 'required_amount' => 0, 'total_paid' => 0, 'total_debt' => 0, 'excess_amount' => 0];
                }
                // Add to 'total_paid' and 'excess_amount'
                $summary_data[$type]['total_paid'] += (float)$row['total_paid'];
                $summary_data[$type]['excess_amount'] += (float)$row['excess_amount'];
            }
            mysqli_free_result($result2);
        } else {
            error_log("Failed to execute get_payment_types_summary query (Part 2): " . mysqli_stmt_error($stmt2) . "; SQL: $sql2");
        }
        mysqli_stmt_close($stmt2);
    } else {
        error_log("Failed to prepare get_payment_types_summary query (Part 2): " . mysqli_error($link) . "; SQL: $sql2");
    }

    // Convert the associative array back to a simple array
    return array_values($summary_data);
}

function get_expense_types_summary($link, $record_type, $time_period, $start_date, $end_date, $type_filter) {
    // --- FIX: Include all types from expense_types, even if they have 0 transactions ---
    $sql = "SELECT et.type_name, COALESCE(SUM(e.amount), 0) AS total
            FROM expense_types et
            LEFT JOIN expenses e ON e.expense_type_id = et.id";
    
    $bind_types = '';
    $bind_params = [];
    $type_filter_condition = validate_type_filter($type_filter, $record_type);
    
    // Build WHERE clause for the LEFT JOIN
    $where = " WHERE et.deleted_at IS NULL "; // Start with base condition
    
    if ($record_type === 'all' || $record_type === 'expenses') {
        // Add time filter only if there's an expense record (e.id IS NOT NULL)
        $time_filter = str_replace('created_at', 'e.created_at', build_time_filter($time_period, 'created_at', $record_type));
        if (!empty($time_filter)) {
             // Apply time filter only to the 'e' table
            $sql .= " AND (e.id IS NULL OR " . ltrim($time_filter, " AND") . ")";
        }

        if ($time_period === 'range') {
            $bind_types .= 'ss';
            $bind_params[] = $start_date;
            $bind_params[] = $end_date;
        }

        if ($type_filter && strpos($type_filter, 'expense_') === 0) {
            $where .= $type_filter_condition; // This filters et.type_name
            $bind_types .= 's';
            $bind_params[] = explode('_', $type_filter, 2)[1];
        }
    } else {
        // If filtering for other types, still show expense types but with 0 total
        $where .= " AND e.id IS NULL"; 
    }

    $sql .= $where . " GROUP BY et.type_name ORDER BY et.type_name ASC"; // Added ORDER BY
    
    $summary = [];
    if ($stmt = mysqli_prepare($link, $sql)) {
        if ($bind_types) {
            mysqli_stmt_bind_param($stmt, $bind_types, ...$bind_params);
        }
        if (mysqli_stmt_execute($stmt)) {
            $result = mysqli_stmt_get_result($stmt);
            while ($row = mysqli_fetch_assoc($result)) {
                $summary[] = $row;
            }
            mysqli_free_result($result);
        } else {
            error_log("Failed to execute get_expense_types_summary query: " . mysqli_stmt_error($stmt) . "; SQL: $sql");
        }
        mysqli_stmt_close($stmt);
    } else {
        error_log("Failed to prepare get_expense_types_summary query: " . mysqli_error($link) . "; SQL: $sql");
    }
    return $summary;
}

function get_benefit_types_summary($link, $record_type, $time_period, $start_date, $end_date, $type_filter, $member_id = 0) {
    // --- FIX: Include all types from benefit_types, even if they have 0 transactions ---
    $sql = "SELECT bt.type_name, COALESCE(SUM(b.amount), 0) AS total
            FROM benefit_types bt
            LEFT JOIN member_benefits b ON b.benefit_type_id = bt.id";
            
    $bind_types = '';
    $bind_params = [];
    $type_filter_condition = validate_type_filter($type_filter, $record_type);
    
    // Build WHERE clause for the LEFT JOIN
    $where = " WHERE bt.deleted_at IS NULL "; // Start with base condition
    
    $member_condition = $member_id > 0 ? " AND b.member_id = ?" : '';

    if ($record_type === 'all' || $record_type === 'benefits') {
        // Add time filter only if there's a benefit record (b.id IS NOT NULL)
        $time_filter = str_replace('created_at', 'b.created_at', build_time_filter($time_period, 'created_at', $record_type));
        if(!empty($time_filter)) {
            $sql .= " AND (b.id IS NULL OR " . ltrim($time_filter, " AND") . ")";
        }

        if ($time_period === 'range') {
            $bind_types .= 'ss';
            $bind_params[] = $start_date;
            $bind_params[] = $end_date;
        }
        if ($type_filter && strpos($type_filter, 'benefit_') === 0) {
            $where .= $type_filter_condition; // This filters bt.type_name
            $bind_types .= 's';
            $bind_params[] = explode('_', $type_filter, 2)[1];
        }
        
        $sql .= $member_condition;
        if ($member_id > 0) {
            $bind_types .= 'i';
            $bind_params[] = $member_id;
        }
    } else {
        // If filtering for other types, still show benefit types but with 0 total
        $where .= " AND b.id IS NULL";
    }

    $sql .= $where . " GROUP BY bt.type_name ORDER BY bt.type_name ASC"; // Added ORDER BY

    $summary = [];
    if ($stmt = mysqli_prepare($link, $sql)) {
        if ($bind_types) {
            mysqli_stmt_bind_param($stmt, $bind_types, ...$bind_params);
        }
        if (mysqli_stmt_execute($stmt)) {
            $result = mysqli_stmt_get_result($stmt);
            while ($row = mysqli_fetch_assoc($result)) {
                $summary[] = $row;
            }
            mysqli_free_result($result);
        } else {
            error_log("Failed to execute get_benefit_types_summary query: " . mysqli_stmt_error($stmt) . "; SQL: $sql");
        }
        mysqli_stmt_close($stmt);
    } else {
        error_log("Failed to prepare get_benefit_types_summary query: " . mysqli_error($link) . "; SQL: $sql");
    }
    return $summary;
}

function get_line_chart_data($link, $end_date, $date_field) {
    $data = ['labels' => [], 'payments' => [], 'expenses' => [], 'benefits' => [], 'debts' => []];
    $current = new DateTime(date('Y-m-01', strtotime('-11 months', strtotime($end_date))));
    $end = new DateTime(date('Y-m-t', strtotime($end_date)));
    while ($current <= $end) {
        $month_start = $current->format('Y-m-01');
        $month_end = $current->format('Y-m-t');
        $data['labels'][] = $current->format('M Y');

        $sql = "SELECT COALESCE(SUM(amount), 0) FROM payments WHERE payment_status = 'Paid' AND DATE(created_at) BETWEEN ? AND ?";
        if ($stmt = mysqli_prepare($link, $sql)) {
            mysqli_stmt_bind_param($stmt, 'ss', $month_start, $month_end);
            mysqli_stmt_execute($stmt);
            $data['payments'][] = mysqli_fetch_array(mysqli_stmt_get_result($stmt))[0] ?? 0;
            mysqli_stmt_close($stmt);
        } else {
            error_log("Failed to prepare get_line_chart_data payments query: " . mysqli_error($link) . "; SQL: $sql");
        }

        $sql = "SELECT COALESCE(SUM(amount), 0) FROM expenses WHERE DATE(created_at) BETWEEN ? AND ?";
        if ($stmt = mysqli_prepare($link, $sql)) {
            mysqli_stmt_bind_param($stmt, 'ss', $month_start, $month_end);
            mysqli_stmt_execute($stmt);
            $data['expenses'][] = mysqli_fetch_array(mysqli_stmt_get_result($stmt))[0] ?? 0;
            mysqli_stmt_close($stmt);
        } else {
            error_log("Failed to prepare get_line_chart_data expenses query: " . mysqli_error($link) . "; SQL: $sql");
        }

        $sql = "SELECT COALESCE(SUM(amount), 0) FROM member_benefits WHERE DATE(created_at) BETWEEN ? AND ?";
        if ($stmt = mysqli_prepare($link, $sql)) {
            mysqli_stmt_bind_param($stmt, 'ss', $month_start, $month_end);
            mysqli_stmt_execute($stmt);
            $data['benefits'][] = mysqli_fetch_array(mysqli_stmt_get_result($stmt))[0] ?? 0;
            mysqli_stmt_close($stmt);
        } else {
            error_log("Failed to prepare get_line_chart_data benefits query: " . mysqli_error($link) . "; SQL: $sql");
        }

        $sql = "SELECT COALESCE(SUM(debt), 0) AS total_debt 
                FROM (
                    SELECT 
                        po.id,
                        GREATEST(COALESCE(po.required_amount, 0) - COALESCE(SUM(p.amount), 0), 0) AS debt
                    FROM payment_obligations po
                    JOIN users u ON po.user_id = u.id
                    JOIN payment_types pt ON po.payment_type = pt.type_name
                    LEFT JOIN payments p ON p.obligation_id = po.id AND p.payment_status = 'Paid'
                    WHERE po.deleted_at IS NULL
                        AND u.role = 'member'
                        AND (u.deleted_at IS NULL OR u.deleted_at > NOW())
                        AND COALESCE(po.required_amount, 0) > 0
                        AND DATE(" . ($date_field === 'due_date' ? 'po.due_date' : 'po.created_at') . ") BETWEEN ? AND ?
                    GROUP BY po.id, po.required_amount
                    HAVING debt > 0
                ) AS subquery";
        if ($stmt = mysqli_prepare($link, $sql)) {
            mysqli_stmt_bind_param($stmt, 'ss', $month_start, $month_end);
            mysqli_stmt_execute($stmt);
            $data['debts'][] = mysqli_fetch_array(mysqli_stmt_get_result($stmt))[0] ?? 0;
            mysqli_stmt_close($stmt);
        } else {
            error_log("Failed to prepare get_line_chart_data debts query: " . mysqli_error($link) . "; SQL: $sql");
        }

        $current->modify('+1 month');
    }
    return $data;
}

function test_payment_type_column($link) {
    $sql = "SELECT po.payment_type FROM payment_obligations po LIMIT 1";
    if ($stmt = mysqli_prepare($link, $sql)) {
        if (mysqli_stmt_execute($stmt)) {
            $result = mysqli_stmt_get_result($stmt);
            $row = mysqli_fetch_assoc($result);
            mysqli_free_result($result);
            mysqli_stmt_close($stmt);
            return $row['payment_type'] ?? 'No data';
        } else {
            error_log("Test query execution failed: " . mysqli_stmt_error($stmt) . "; SQL: $sql");
            mysqli_stmt_close($stmt);
            return 'Execution failed';
        }
    } else {
        error_log("Test query preparation failed: " . mysqli_error($link) . "; SQL: $sql");
        return 'Preparation failed';
    }
}

// Include validate_type_filter if not already defined
if (!function_exists('validate_type_filter')) {
    function validate_type_filter($type_filter, $record_type) {
        if ($type_filter === '') return '';
        $parts = explode('_', $type_filter, 2);
        $category = $parts[0] ?? '';
        $name = $parts[1] ?? '';
        $allowed = ['income', 'expense', 'benefit', 'payment'];
        if (in_array($category, $allowed)) {
            $table_map = [
                'income' => 'p.payment_type',
                'payment' => 'p.payment_type',
                'expense' => 'et.type_name',
                'benefit' => 'bt.type_name'
            ];
            return " AND {$table_map[$category]} = ?";
        }
        return '';
    }
}
?>