<?php
define('IN_REPORTS', true);
/* --------------------------------------------------------------
  financial_summary.php
  (Combined Transaction & Admin Ledger Report)
  -------------------------------------------------------------- */
session_start();

require_once 'includes/functions.php';
require_once 'includes/db_config.php';
require_once 'includes/financial_queries.php';
require_once 'includes/financial_utils.php';

/* ---------- Development error reporting (remove in prod) ---------- */
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

/* ---------- DB connection check ---------- */
if (!$link) {
    die("ERROR: Could not connect. " . mysqli_connect_error());
}

/* ---------- Auth ---------- */
if (!is_logged_in()) {
    redirect('login.php');
}
$is_admin = is_admin();

$system_name       = get_setting('system_name', $link) ?? 'Club Portal';
$currency_symbol   = get_setting('currency_symbol', $link) ?? '₦';
$currency_symbol_pdf = 'NGN '; // PDF Safe currency symbol

/*
 * ==================================================================
 * !! UNIFIED FILTER & REPORTING VARIABLES !!
 * ==================================================================
 */

/* ---------- Sanitize & defaults (Transaction Report) ---------- */
$valid_record_types = ['all', 'payments', 'expenses', 'debts', 'benefits'];
$valid_time_periods = ['all', 'range', 'week', 'month', 'year'];
$valid_per_page     = [5, 10, 25, 50, 100];

// --- Filters for Transaction Report ---
$record_type = sanitize_filter($_GET['record_type'] ?? 'all', $valid_record_types, 'all');
$time_period = sanitize_filter($_GET['time_period'] ?? 'all', $valid_time_periods, 'all');
$start_date = $_GET['start_date'] ?? date('Y-m-01');
$end_date   = $_GET['end_date']   ?? date('Y-m-d');
$search = $_GET['search'] ?? '';
$member_id = (int)($_GET['member_id'] ?? 0);
$date_field = ($_GET['date_field'] ?? '') === 'due_date' ? 'due_date' : 'created_at';
$type_filter_payments = $_GET['type_filter_payments'] ?? '';
$type_filter_expenses = $_GET['type_filter_expenses'] ?? '';
$type_filter_benefits = $_GET['type_filter_benefits'] ?? '';


// --- Filters for Admin Report ---
// (We use different variable names to keep the two filter sets separate)
$admin_filter_status = sanitize_input($_GET['admin_status'] ?? 'all');
$admin_from_date = sanitize_input($_GET['admin_from_date'] ?? date('Y-m-d', strtotime('-30 days')));
$admin_to_date = sanitize_input($_GET['admin_to_date'] ?? date('Y-m-d'));


/*
 * ==================================================================
 * !! ADMIN-ONLY LEDGER REPORT DATA FETCH !!
 * (This section fetches data from admin_financial_report.php)
 * (FIXED: Queries now use 'amount', 'transaction_type', and 'bank_account_id'
 * to match your database schema, fixing the 0.00 and N/A errors)
 * ==================================================================
 */

/* ---------- ADMIN REPORT VARS (Initialize) ---------- */
$admin_kpis = ['inflows' => 0, 'outflows' => 0, 'net' => 0];
$admin_account_balances = [];
$admin_total_liquid_assets = 0;
$admin_opening_balance = 0;
$admin_closing_balance = 0;
$admin_net_change = 0;
$admin_fund_requests = [];
$admin_ledger_history = [];
$admin_pie_chart_data = '{"labels": [], "datasets": []}';
$admin_bar_chart_data = '{"labels": [], "datasets": []}';
$admin_message = '';


if ($is_admin) {
    // Ensure date format is correct for SQL
    $admin_from_date_sql = mysqli_real_escape_string($link, $admin_from_date . ' 00:00:00');
    $admin_to_date_sql   = mysqli_real_escape_string($link, $admin_to_date . ' 23:59:59');

    // SQL filter clauses for the admin report
    // FIX 1: Removed COALESCE and fl.created_at. Now only uses fl.date_recorded.
    $admin_where_date = " WHERE fl.date_recorded BETWEEN '$admin_from_date_sql' AND '$admin_to_date_sql' ";
    $admin_where_status_fund_requests = ($admin_filter_status !== 'all' && in_array($admin_filter_status, ['pending', 'approved', 'rejected']))
        ? " AND fr.status = '" . mysqli_real_escape_string($link, $admin_filter_status) . "' "
        : '';


    // --- 1. Fetch Bank Account Balances (Current) ---
    $sql_balances = "SELECT account_name, current_balance FROM bank_accounts WHERE deleted_at IS NULL AND is_active = 1 ORDER BY account_name";
    if ($result = mysqli_query($link, $sql_balances)) {
        while ($row = mysqli_fetch_assoc($result)) {
            $admin_account_balances[] = $row;
            $admin_total_liquid_assets += $row['current_balance'];
        }
    } else {
        $admin_message .= "Error fetching balances: " . mysqli_error($link) . "<br>";
    }

    // --- 2. Calculate Opening/Closing Balances (Based on Period) ---
    // FIX 2: (THE ERROR LINE) Removed COALESCE and fl.created_at. Now only uses fl.date_recorded.
    $sql_opening = "SELECT COALESCE(SUM(CASE WHEN fl.transaction_type = 'income' THEN fl.amount ELSE -fl.amount END), 0) AS opening_balance
                    FROM financial_ledger AS fl
                    WHERE fl.date_recorded < '$admin_from_date_sql'";

    if ($result_open = mysqli_query($link, $sql_opening)) {
        $admin_opening_balance = (float)mysqli_fetch_assoc($result_open)['opening_balance'];
    } else {
         $admin_message .= "Error fetching opening balance: " . mysqli_error($link) . "<br>";
    }

    // FIX: Reverted to 'amount' and 'transaction_type' logic
    $sql_net_change = "SELECT COALESCE(SUM(CASE WHEN fl.transaction_type = 'income' THEN fl.amount ELSE -fl.amount END), 0) AS net_change
                         FROM financial_ledger AS fl $admin_where_date";

    if ($result_net = mysqli_query($link, $sql_net_change)) {
        $admin_net_change = (float)mysqli_fetch_assoc($result_net)['net_change'];
        $admin_closing_balance = $admin_opening_balance + $admin_net_change;
    } else {
        $admin_message .= "Error fetching net change: " . mysqli_error($link) . "<br>";
    }

    // --- 3. KPIs (Inflows, Outflows, Net) for the period ---
    // FIX: Reverted to 'amount' and 'transaction_type' logic
    $sql_kpis = "SELECT
                    COALESCE(SUM(CASE WHEN fl.transaction_type = 'income' THEN fl.amount ELSE 0 END), 0) AS inflows,
                    COALESCE(SUM(CASE WHEN fl.transaction_type != 'income' THEN fl.amount ELSE 0 END), 0) AS outflows
                 FROM financial_ledger AS fl $admin_where_date";

    if ($result_kpi = mysqli_query($link, $sql_kpis)) {
        $kpi_row = mysqli_fetch_assoc($result_kpi);
        $admin_kpis['inflows']  = (float)$kpi_row['inflows'];
        $admin_kpis['outflows'] = (float)$kpi_row['outflows'];
        $admin_kpis['net']      = $admin_kpis['inflows'] - $admin_kpis['outflows'];
    } else {
        $admin_message .= "Error fetching KPIs: " . mysqli_error($link) . "<br>";
    }


    // --- 4. Fetch Fund Requests (Filterable) ---
    $sql_requests = "
        SELECT
            fr.id, fr.amount, fr.purpose AS reason, fr.status,
            COALESCE(fr.request_date, fr.created_at) AS display_date,
            COALESCE(u.full_name, u.email) AS user_display_name
        FROM fund_requests fr
        LEFT JOIN users u ON fr.requester_id = u.id
        WHERE COALESCE(fr.request_date, fr.created_at) BETWEEN '$admin_from_date_sql' AND '$admin_to_date_sql' $admin_where_status_fund_requests
        ORDER BY display_date DESC
    ";
    if ($result = mysqli_query($link, $sql_requests)) {
        while ($row = mysqli_fetch_assoc($result)) {
            $admin_fund_requests[] = $row;
        }
    } else {
        $admin_message .= "Error fetching fund requests: " . mysqli_error($link) . "<br>";
    }

    // --- 5. Fetch Financial Ledger History (Filterable) ---
    // FIX 3: Removed COALESCE and fl.created_at. Now only uses fl.date_recorded.
    $sql_ledger = "
        SELECT
            fl.id, fl.date_recorded, fl.description,
            fl.amount, fl.transaction_type,
            ba.account_name,
            fl.source_table, fl.source_id
        FROM financial_ledger fl
        LEFT JOIN bank_accounts ba ON fl.bank_account_id = ba.id
        $admin_where_date
        ORDER BY fl.date_recorded DESC, fl.id DESC
        LIMIT 200 -- Limit for performance
    ";
    if ($result = mysqli_query($link, $sql_ledger)) {
        while ($row = mysqli_fetch_assoc($result)) {
            $admin_ledger_history[] = $row;
        }
    } else {
        $admin_message .= "Error fetching ledger history: " . mysqli_error($link) . ".<br>";
    }

    // --- 6. Chart Data ---
    // 6a. Pie Chart (Inflows vs Outflows)
    $admin_pie_chart_data = json_encode([
        'labels' => ['Inflows', 'Outflows'],
        'datasets' => [[
            'data' => [$admin_kpis['inflows'], $admin_kpis['outflows']],
            'backgroundColor' => ['#28a745', '#dc3545']
        ]]
    ]);

    // 6b. Bar Chart (Trend) Data
    // FIX 4: Removed COALESCE and fl.created_at. Now only uses fl.date_recorded.
    $admin_bar_chart_labels = [];
    $admin_bar_chart_inflows = [];
    $admin_bar_chart_outflows = [];
    $sql_bar_chart = "
        SELECT
            DATE(fl.date_recorded) AS transaction_date,
            COALESCE(SUM(CASE WHEN fl.transaction_type = 'income' THEN fl.amount ELSE 0 END), 0) AS daily_inflow,
            COALESCE(SUM(CASE WHEN fl.transaction_type != 'income' THEN fl.amount ELSE 0 END), 0) AS daily_outflow
        FROM financial_ledger AS fl
        $admin_where_date
        GROUP BY DATE(transaction_date)
        ORDER BY transaction_date ASC
    ";
    if ($result_bar = mysqli_query($link, $sql_bar_chart)) {
        while ($row = mysqli_fetch_assoc($result_bar)) {
            $admin_bar_chart_labels[] = date('M d', strtotime($row['transaction_date']));
            $admin_bar_chart_inflows[] = (float)$row['daily_inflow'];
            $admin_bar_chart_outflows[] = (float)$row['daily_outflow'];
        }
    } else {
        $admin_message .= "Error fetching chart data: " . mysqli_error($link) . "<br>";
    }
    // FIX: Updated labels
    $admin_bar_chart_data = json_encode([
        'labels' => $admin_bar_chart_labels,
        'datasets' => [
            [
                'label' => 'Inflows',
                'data' => $admin_bar_chart_inflows,
                'backgroundColor' => 'rgba(40, 167, 69, 0.7)',
                'borderColor' => '#28a745',
                'borderWidth' => 1
            ],
            [
                'label' => 'Outflows',
                'data' => $admin_bar_chart_outflows,
                'backgroundColor' => 'rgba(220, 53, 69, 0.7)',
                'borderColor' => '#dc3545',
                'borderWidth' => 1
            ]
        ]
    ]);

} // --- END OF if ($is_admin) BLOCK ---


/*
 * ==================================================================
 * !! ORIGINAL TRANSACTION REPORT DATA FETCH !!
 * (This data is now used for the unified KPI cards and the
 * detailed transaction tables)
 * ==================================================================
 */

/* ---------- Pagination (Transaction Report) ---------- */
$tables = ['payments', 'expenses', 'debts', 'benefits'];
$pagination = [];

foreach ($tables as $table) {
    $per_page = (int)($_GET["per_page_$table"] ?? 10);
    $per_page = in_array($per_page, $valid_per_page) ? $per_page : 10;
    $page = max(1, (int)($_GET["page_$table"] ?? 1));
    $offset = ($page - 1) * $per_page;
    $pagination[$table] = compact('per_page', 'page', 'offset');
}

/* ---------- Members for filter (Transaction Report) ---------- */
$members = [];
$sql = "SELECT id, full_name FROM users WHERE role = 'member' AND (deleted_at IS NULL OR deleted_at > NOW()) ORDER BY full_name";
$res = mysqli_query($link, $sql);
while ($row = mysqli_fetch_assoc($res)) {
    $members[] = $row;
}
mysqli_free_result($res);

/* ---------- CSV Export (Transaction Report) ---------- */
if (isset($_GET['export']) && in_array($_GET['export'], $tables)) {
    $export_table = $_GET['export'];
    $full_pagination = ['per_page' => PHP_INT_MAX, 'page' => 1, 'offset' => 0];
    $data = [];

    switch ($export_table) {
        case 'payments':
            $data = get_payments($link, $record_type, $time_period, $start_date, $end_date,
                                  $type_filter_payments, $search, $full_pagination, $member_id);
            break;
        case 'expenses':
            $data = get_expenses($link, $record_type, $time_period, $start_date, $end_date,
                                  $type_filter_expenses, $search, $full_pagination);
            break;
        case 'benefits':
            $data = get_benefits($link, $record_type, $time_period, $start_date, $end_date,
                                  $type_filter_benefits, $search, $full_pagination, $member_id);
            break;
        case 'debts':
            $data = get_debts($link, $record_type, $time_period, $start_date, $end_date,
                                  '', $search, $date_field, $full_pagination, $member_id);
            break;
    }

    header('Content-Type: text/csv');
    header('Content-Disposition: attachment; filename="' . $export_table . '.csv"');
    $out = fopen('php://output', 'w');
    
    // Dynamically get headers
    if (!empty($data)) {
        fputcsv($out, array_keys($data[0]));
        foreach ($data as $r) {
            fputcsv($out, $r);
        }
    } elseif ($export_table === 'payments') {
         fputcsv($out, ['ID','Full Name','Payment Type','Amount','Status','Created At','Payment Cycle']);
    } // etc. for other tables
    
    fclose($out);
    exit;
}

/* ---------- Totals & counts (Transaction Report) ---------- */
$total_counts = get_total_counts($link, $record_type, $time_period, $start_date, $end_date,
                                  '', $search, $date_field, $pagination, $member_id);

$total_pages = [];
foreach ($tables as $t) {
    $total_pages[$t] = max(1, (int)ceil($total_counts[$t] / $pagination[$t]['per_page']));
}

// NOTE: These totals are based on the TRANSACTION tables, not the ledger.
// They are used for the *second* part of the report.
$total_payments = get_total_payments($link, $record_type, $time_period, $start_date, $end_date,
                                     $type_filter_payments, $member_id);
$total_expenses = get_total_expenses($link, $record_type, $time_period, $start_date, $end_date,
                                     $type_filter_expenses);
$total_benefits = get_total_benefits($link, $record_type, $time_period, $start_date, $end_date,
                                     $type_filter_benefits, $member_id);
// This 'Total Debts' is used in the new Unified KPI card row
$total_debts    = get_total_debts($link, $record_type, $time_period, $start_date, $end_date,
                                    '', $date_field, $member_id);

/* ---------- Type option dropdowns (Transaction Report) ---------- */
$type_options_payments = get_type_options($link, 'income',   $type_filter_payments);
$type_options_expenses = get_type_options($link, 'expenses',$type_filter_expenses);
$type_options_benefits = get_type_options($link, 'benefits',$type_filter_benefits);

/* ---------- Paginated data (Transaction Report) ---------- */
$payments = get_payments($link, $record_type, $time_period, $start_date, $end_date,
                          $type_filter_payments, $search, $pagination['payments'], $member_id);
$expenses = get_expenses($link, $record_type, $time_period, $start_date, $end_date,
                          $type_filter_expenses, $search, $pagination['expenses']);
$benefits = get_benefits($link, $record_type, $time_period, $start_date, $end_date,
                          $type_filter_benefits, $search, $pagination['benefits'], $member_id);
$debts    = get_debts($link, $record_type, $time_period, $start_date, $end_date,
                          '', $search, $date_field, $pagination['debts'], $member_id);

/* ---------- Summary data for tables (Transaction Report) ---------- */
$payment_types_summary = get_payment_types_summary($link, $record_type, $time_period,
                                                 $start_date, $end_date, $type_filter_payments,
                                                 $date_field, $member_id);
$expense_types_summary = get_expense_types_summary($link, $record_type, $time_period,
                                                   $start_date, $end_date, $type_filter_expenses);
$benefit_types_summary = get_benefit_types_summary($link, $record_type, $time_period,
                                                   $start_date, $end_date, $type_filter_benefits,
                                                   $member_id);


/* ---------- Include template ---------- */
require_once 'templates/financial_summary_template.php';
?>
