<?php
// export_all_members_pdf.php
// DOMPDF 2.0.8 - FULLY WORKING WITH LONG TEXT WRAPPING

require_once 'includes/functions.php';
require_once 'includes/db_config.php';

// SECURITY
if (!is_logged_in() || !is_admin()) {
    header('Location: login.php');
    exit();
}

// LOAD DOMPDF
require_once 'dompdf/autoload.inc.php';
use Dompdf\Dompdf;
use Dompdf\Options;

// CONFIG
$options = new Options();
$options->set('isRemoteEnabled', true);
$options->set('defaultFont', 'DejaVu Sans');
$options->set('isHtml5ParserEnabled', true);
$options->set('isPhpEnabled', true);

$dompdf = new Dompdf($options);
$dompdf->setPaper('A4', 'portrait');

// QUERY
$sql = "SELECT 
            u.id, u.member_id, u.full_name, u.email, u.membership_status, u.profile_picture,
            fd.data AS form_data, f.form_data AS form_structure
        FROM users u
        LEFT JOIN form_data fd ON u.id = fd.user_id
        LEFT JOIN forms f ON fd.form_id = f.id
        WHERE u.role IN ('member','pending') 
          AND (u.deleted_at IS NULL OR u.deleted_at > NOW())
        ORDER BY u.full_name";

$result = mysqli_query($link, $sql);
if (!$result || mysqli_num_rows($result) == 0) {
    die('No members found.');
}

// BASE URL
$protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https://' : 'http://';
$base_url = $protocol . $_SERVER['HTTP_HOST'] . dirname($_SERVER['SCRIPT_NAME']);
$base_url = rtrim($base_url, '/') . '/';

// BUILD HTML
$html = '<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <style>
        body { 
            font-family: "DejaVu Sans", Arial, sans-serif; 
            margin: 20px; 
            font-size: 12px; 
            line-height: 1.5;
            color: #333;
        }
        .header { 
            text-align: center; 
            margin-bottom: 30px; 
            border-bottom: 2px solid #28a745; 
            padding-bottom: 10px;
        }
        .member { 
            page-break-inside: avoid; 
            margin-bottom: 40px; 
            border: 1px solid #ddd; 
            padding: 15px; 
            border-radius: 5px;
            background: #f9f9f9;
            overflow: hidden;
        }
        .profile-img { 
            width: 60px; 
            height: 60px; 
            border-radius: 50%; 
            float: left; 
            margin-right: 15px; 
            border: 2px solid #28a745;
            object-fit: cover;
        }
        h1 { color: #28a745; margin: 0; font-size: 18px; }
        h2 { color: #28a745; margin: 0 0 10px 0; font-size: 16px; }
        h3 { 
            color: #155724; 
            font-size: 14px; 
            margin: 15px 0 8px 0; 
            border-bottom: 1px solid #28a745; 
            padding-bottom: 3px;
        }
        table { 
            width: 100%; 
            border-collapse: collapse; 
            margin: 8px 0;
            table-layout: fixed;
        }
        th, td { 
            border: 1px solid #28a745; 
            padding: 8px; 
            text-align: left;
            vertical-align: top;
            word-wrap: break-word;
            overflow-wrap: break-word;
            hyphens: auto;
        }
        th { 
            background-color: #d4edda; 
            color: #155724; 
            width: 35%; 
            font-weight: bold;
        }
        td { 
            width: 65%;
        }
        .status-approved { color: #28a745; font-weight: bold; }
        .status-pending { color: #ffc107; font-weight: bold; }
        .status-rejected { color: #dc3545; font-weight: bold; }
        .clearfix { clear: both; }
        @page { margin: 1cm; }
    </style>
</head>
<body>
    <div class="header">
        <h1>MEMBERS DIRECTORY</h1>
        <p>Generated on ' . date('F j, Y \a\t g:i A') . '</p>
    </div>';

while ($row = mysqli_fetch_assoc($result)) {
    $form_data = json_decode($row['form_data'] ?? '', true) ?: [];
    $form_structure = json_decode($row['form_structure'] ?? '', true) ?: [];
    $fields = [];

    if (isset($form_structure['fields'])) {
        foreach ($form_structure['fields'] as $f) {
            $name = $f['name'] ?? '';
            $label = $f['label'] ?? $name;
            if ($name && isset($form_data[$name])) {
                $fields[] = [
                    'label' => htmlspecialchars($label, ENT_HTML5 | ENT_QUOTES),
                    'value' => nl2br(htmlspecialchars($form_data[$name], ENT_HTML5 | ENT_QUOTES))
                ];
            }
        }
    }

    $status_class = match(strtolower($row['membership_status'])) {
        'approved' => 'status-approved',
        'rejected' => 'status-rejected',
        default => 'status-pending'
    };

    $img = !empty($row['profile_picture']) 
        ? $base_url . $row['profile_picture']
        : $base_url . 'uploads/profile_pictures/default.png';

    $html .= "
    <div class='member'>
        <h2>" . htmlspecialchars($row['full_name']) . " 
            <small>(ID: " . htmlspecialchars($row['member_id']) . ")</small>
        </h2>
        <img src='$img' class='profile-img' alt='Profile'>
        
        <h3>Core Information</h3>
        <table>
            <tr><th>Full Name</th><td>" . htmlspecialchars($row['full_name']) . "</td></tr>
            <tr><th>Email</th><td>" . htmlspecialchars($row['email']) . "</td></tr>
            <tr><th>Member ID</th><td>" . htmlspecialchars($row['member_id']) . "</td></tr>
            <tr><th>Status</th><td class='$status_class'>" . htmlspecialchars($row['membership_status']) . "</td></tr>
        </table>

        <h3>Registration Data</h3>
        <table>";

    if ($fields) {
        foreach ($fields as $f) {
            $html .= "<tr><th>{$f['label']}</th><td>{$f['value']}</td></tr>";
        }
    } else {
        $html .= "<tr><td colspan='2'><em>No additional data</em></td></tr>";
    }

    $html .= "</table>
        <div class='clearfix'></div>
    </div>";
}

$html .= '</body></html>';

// RENDER & OUTPUT
$dompdf->loadHtml($html);
$dompdf->render();
$dompdf->stream('members_directory_' . date('Y-m-d') . '.pdf', ['Attachment' => true]);
exit();
?>