/**
 * form_renderer.js
 * Handles conditional logic for the public-facing registration form.
 *
 * FIXES:
 * 1. Uses the unique class hook (.form-field-FIELDNAME) from register.php
 * to reliably target the entire form group container.
 * 2. Reworked source value retrieval for radio and checkbox/consent fields.
 */

$(document).ready(function() {
    // Check if the logic data is available
    if (typeof FORM_CONDITIONAL_LOGIC === 'undefined' || $.isEmptyObject(FORM_CONDITIONAL_LOGIC)) {
        console.log("No conditional logic defined for this form.");
        return;
    }

    // --- Core Logic Execution ---
    
    // Process a single rule set for a target field
    const checkFieldLogic = (targetFieldId, logicConfig) => {
        const rules = logicConfig.rules;
        const matchType = logicConfig.match; // 'all' or 'any'
        
        let conditionsMetCount = 0;

        rules.forEach(rule => {
            // Check for the source field using both standard and array notation
            const sourceFieldName = rule.source_field_name;
            const sourceFieldSelector = `[name="${sourceFieldName}"], [name="${sourceFieldName}[]"]`;
            const sourceFieldElement = $(sourceFieldSelector).first(); // Get the primary element

            if (sourceFieldElement.length === 0) return; // Skip if source field doesn't exist

            let sourceValue;

            // Determine the value of the source field
            if (sourceFieldElement.is(':radio')) {
                // Radio buttons need the checked one
                sourceValue = $(`[name="${sourceFieldName}"]:checked`).val() || '';
            } else if (sourceFieldElement.is(':checkbox') || sourceFieldElement.closest('.form-group').data('type') === 'consent') {
                // Handle checkbox and consent fields (which behave like single checkboxes or groups)
                const checkedValues = $(sourceFieldSelector + ':checked').map(function() {
                    return this.value;
                }).get();
                
                if (checkedValues.length > 1 || sourceFieldElement.attr('name').endsWith('[]')) {
                    // This is a checkbox group or a multi-select simulation
                    sourceValue = checkedValues;
                } else { 
                    // Single checkbox or consent: value is 'checked' value or empty string
                    sourceValue = checkedValues.length > 0 ? checkedValues[0] : '';
                }
            } else {
                // Text, Email, Date, Select, Textarea
                sourceValue = sourceFieldElement.val() || '';
            }

            // Standardize source value for comparison
            sourceValue = Array.isArray(sourceValue) ? sourceValue : String(sourceValue).toLowerCase().trim();
            const ruleValue = String(rule.value).toLowerCase().trim();

            let conditionIsMet = false;

            // Execute the rule comparison
            switch (rule.operator) {
                case 'equals':
                    if (Array.isArray(sourceValue)) {
                        // Check if the array contains the rule value (for multi-select/checkbox groups)
                        conditionIsMet = sourceValue.includes(ruleValue);
                    } else {
                        conditionIsMet = (sourceValue === ruleValue);
                    }
                    break;
                case 'not_equals':
                    if (Array.isArray(sourceValue)) {
                        // Check if the array does NOT contain the rule value
                        conditionIsMet = !sourceValue.includes(ruleValue);
                    } else {
                        conditionIsMet = (sourceValue !== ruleValue);
                    }
                    break;
                case 'is_empty':
                    conditionIsMet = (sourceValue === '' || (Array.isArray(sourceValue) && sourceValue.length === 0));
                    break;
                case 'is_not_empty':
                    conditionIsMet = (sourceValue !== '' && (!Array.isArray(sourceValue) || sourceValue.length > 0));
                    break;
                case 'contains':
                    conditionIsMet = String(sourceValue).toLowerCase().includes(ruleValue);
                    break;
                case 'greater_than':
                    conditionIsMet = parseFloat(sourceValue) > parseFloat(ruleValue);
                    break;
                case 'less_than':
                    conditionIsMet = parseFloat(sourceValue) < parseFloat(ruleValue);
                    break;
            }

            if (conditionIsMet) {
                conditionsMetCount++;
            }
        });

        // --- Apply Action ---
        let shouldApplyRuleAction;

        if (matchType === 'all') {
            shouldApplyRuleAction = (conditionsMetCount === rules.length && rules.length > 0);
        } else { // matchType === 'any'
            shouldApplyRuleAction = (conditionsMetCount > 0);
        }

        // *** FIX: Reliably target the entire form group using the unique class hook ***
        const targetElement = $(`.form-field-${targetFieldId}`).first();
        if (targetElement.length === 0) {
            console.warn(`Conditional logic failed to find target element: .form-field-${targetFieldId}`);
            return;
        }

        const isActionShow = logicConfig.action === 'show';

        // Determine final visibility based on rule and action
        let finalVisibility = false; // Default to hidden/default
        
        if (isActionShow) {
            finalVisibility = shouldApplyRuleAction; // Show if rules match
        } else {
            finalVisibility = !shouldApplyRuleAction; // Show if rules *don't* match
        }

        if (finalVisibility) {
            targetElement.show();
        } else {
            targetElement.hide();
        }
        
        // --- Required Field Toggle ---
        targetElement.find('input, select, textarea').each(function() {
            const field = $(this);
            const originalRequired = field.data('original-required') === true; // Check stored attribute

            if (targetElement.is(':hidden')) {
                // If the field is hidden, remove 'required' and store its original state
                if (field.prop('required')) {
                    field.prop('required', false).data('original-required', true);
                }
            } else {
                // If the field is visible and was originally required, restore 'required'
                if (originalRequired) {
                    field.prop('required', true).removeData('original-required');
                }
            }
        });
    };


    // --- Initialization and Event Binding ---

    const allSourceFields = new Set();

    // 1. Initial check on load, and collect all source field names
    for (const targetFieldId in FORM_CONDITIONAL_LOGIC) {
        const config = FORM_CONDITIONAL_LOGIC[targetFieldId];
        
        // *** FIX: Use class hook for initial hide ***
        $(`.form-field-${targetFieldId}`).first().hide();
        
        // Collect all unique source fields
        config.rules.forEach(rule => {
            // Note: The rule.field_id in the builder maps to rule.source_field_name here
            allSourceFields.add(rule.source_field_name);
        });

        // Perform the initial check
        checkFieldLogic(targetFieldId, config);
    }
    
    // 2. Attach event listeners to all unique source fields
    allSourceFields.forEach(fieldName => {
        // Selector covers single and group inputs (e.g., name="field" and name="field[]")
        const fieldSelector = `[name="${fieldName}"], [name="${fieldName}[]"]`;
        
        // Attach to 'change' for all fields (select, radio, checkbox, text)
        $(fieldSelector).on('change', function() {
            for (const targetFieldId in FORM_CONDITIONAL_LOGIC) {
                checkFieldLogic(targetFieldId, FORM_CONDITIONAL_LOGIC[targetFieldId]);
            }
        });
        
        // Attach 'keyup' only to text/number/textarea fields for real-time validation
        $(`input[type="text"], input[type="email"], input[type="number"], input[type="date"], textarea`).filter(fieldSelector).on('keyup', function() {
             for (const targetFieldId in FORM_CONDITIONAL_LOGIC) {
                checkFieldLogic(targetFieldId, FORM_CONDITIONAL_LOGIC[targetFieldId]);
            }
        });
    });

});