<?php
require_once 'includes/functions.php';
require_once 'includes/db_config.php';
session_start();

// Check if user is logged in and has permission to manage payments
if (!is_logged_in() || !user_has_permission($link, 'view_payments')) {
    redirect("login.php");
}

$message = '';
$message_type = '';

// Handle form submissions for adding/removing obligations
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    if (isset($_POST['add_obligation'])) {
        $user_id = intval($_POST['user_id']);
        $payment_type_id = intval($_POST['payment_type_id']);
        $payment_cycle = intval($_POST['payment_cycle']);

        $sql = "INSERT INTO payment_obligations (user_id, payment_type_id, payment_cycle) VALUES (?, ?, ?)";
        if ($stmt = mysqli_prepare($link, $sql)) {
            mysqli_stmt_bind_param($stmt, "iii", $user_id, $payment_type_id, $payment_cycle);
            if (mysqli_stmt_execute($stmt)) {
                $message = "Payment obligation added successfully!";
                $message_type = 'success';
            } else {
                $message = "Error adding obligation: " . mysqli_error($link);
                $message_type = 'danger';
            }
            mysqli_stmt_close($stmt);
        }
    } elseif (isset($_POST['delete_obligation'])) {
        $obligation_id = intval($_POST['obligation_id']);

        $sql = "DELETE FROM payment_obligations WHERE id = ?";
        if ($stmt = mysqli_prepare($link, $sql)) {
            mysqli_stmt_bind_param($stmt, "i", $obligation_id);
            if (mysqli_stmt_execute($stmt)) {
                $message = "Payment obligation deleted successfully!";
                $message_type = 'success';
            } else {
                $message = "Error deleting obligation: " . mysqli_error($link);
                $message_type = 'danger';
            }
            mysqli_stmt_close($stmt);
        }
    }
}

// Fetch all members
$members_sql = "SELECT id, full_name, member_id, role, membership_status FROM users ORDER BY full_name ASC";
$members_result = mysqli_query($link, $members_sql);
$all_members = mysqli_fetch_all($members_result, MYSQLI_ASSOC);

// Fetch all payment types
$types_sql = "SELECT id, type_name FROM payment_types ORDER BY type_name ASC";
$types_result = mysqli_query($link, $types_sql);
$all_types = mysqli_fetch_all($types_result, MYSQLI_ASSOC);

$user_id = isset($_GET['user_id']) ? intval($_GET['user_id']) : null;
$selected_member = null;
$obligations = [];

if ($user_id) {
    // Fetch details for the selected member
    $member_details_sql = "SELECT id, full_name, member_id FROM users WHERE id = ?";
    $member_details_stmt = mysqli_prepare($link, $member_details_sql);
    mysqli_stmt_bind_param($member_details_stmt, "i", $user_id);
    mysqli_stmt_execute($member_details_stmt);
    $member_details_result = mysqli_stmt_get_result($member_details_stmt);
    $selected_member = mysqli_fetch_assoc($member_details_result);
    mysqli_stmt_close($member_details_stmt);

    // Fetch existing obligations for the selected member
    $obligations_sql = "SELECT po.id, pt.type_name, po.payment_cycle 
                        FROM payment_obligations po
                        JOIN payment_types pt ON po.payment_type_id = pt.id
                        WHERE po.user_id = ?";
    $obligations_stmt = mysqli_prepare($link, $obligations_sql);
    mysqli_stmt_bind_param($obligations_stmt, "i", $user_id);
    mysqli_stmt_execute($obligations_stmt);
    $obligations_result = mysqli_stmt_get_result($obligations_stmt);
    $obligations = mysqli_fetch_all($obligations_result, MYSQLI_ASSOC);
    mysqli_stmt_close($obligations_stmt);
}
?>

<?php require_once 'includes/header.php'; ?>
<div class="container-fluid">
    <div class="row">
        <?php require_once 'includes/sidebar.php'; ?>
        <main class="col-md-9 col-lg-10 main-content">
            <h2>Manage Member Obligations</h2>
            <?php if ($message): ?>
                <div class="alert alert-<?php echo htmlspecialchars($message_type); ?>">
                    <?php echo htmlspecialchars($message); ?>
                </div>
            <?php endif; ?>

            <div class="card mb-4">
                <div class="card-header">Select a Member</div>
                <div class="card-body">
                    <form action="manage_obligations.php" method="get">
                        <div class="form-group mb-3">
                            <select name="user_id" class="form-control" onchange="this.form.submit()">
                                <option value="">-- Select Member --</option>
                                <?php foreach ($all_members as $member): ?>
                                    <option value="<?php echo htmlspecialchars($member['id']); ?>"
                                        <?php echo ($user_id == $member['id']) ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($member['full_name'] . ' (' . $member['member_id'] . ')'); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                    </form>
                </div>
            </div>

            <?php if ($selected_member): ?>
            <div class="card mb-4">
                <div class="card-header">
                    Obligations for: <?php echo htmlspecialchars($selected_member['full_name']); ?>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table table-striped table-hover">
                            <thead>
                                <tr>
                                    <th>Payment Type</th>
                                    <th>Cycle</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (empty($obligations)): ?>
                                    <tr>
                                        <td colspan="3" class="text-center">No obligations found for this member.</td>
                                    </tr>
                                <?php else: ?>
                                    <?php foreach ($obligations as $obligation): ?>
                                    <tr>
                                        <td><?php echo htmlspecialchars($obligation['type_name']); ?></td>
                                        <td><?php echo htmlspecialchars($obligation['payment_cycle']); ?></td>
                                        <td>
                                            <form method="post" action="manage_obligations.php?user_id=<?php echo $user_id; ?>" style="display:inline;">
                                                <input type="hidden" name="obligation_id" value="<?php echo $obligation['id']; ?>">
                                                <input type="hidden" name="delete_obligation" value="1">
                                                <button type="submit" class="btn btn-danger btn-sm" onclick="return confirm('Are you sure you want to delete this obligation?');">
                                                    Delete
                                                </button>
                                            </form>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>

            <div class="card mb-4">
                <div class="card-header">Add New Obligation</div>
                <div class="card-body">
                    <form method="post" action="manage_obligations.php?user_id=<?php echo $user_id; ?>">
                        <input type="hidden" name="user_id" value="<?php echo $user_id; ?>">
                        <div class="form-group mb-3">
                            <label for="payment_type_id">Payment Type</label>
                            <select name="payment_type_id" id="payment_type_id" class="form-control" required>
                                <option value="">-- Select Payment Type --</option>
                                <?php foreach ($all_types as $type): ?>
                                    <option value="<?php echo htmlspecialchars($type['id']); ?>">
$(document).ready(function() {
    const formDataInput = $('#form-data-input');
    const dynamicFormContainer = $('#dynamic-form-container');
    const prevButton = $('.prev-step');
    const nextButton = $('.next-step');
    const registerButton = $('button[name="register"]');

    if (!formDataInput.length) {
        return;
    }

    const formSchema = JSON.parse(formDataInput.val());
    const fields = formSchema.fields;
    let currentStep = 0;
    const fieldsPerPage = 5;

    // Add the captcha as the very first field
    const captchaField = {
        label: captcha_question,
        name: 'captcha_input',
        type: 'text',
        required: true,
        help_text: 'Please answer the security question.'
    };
    fields.unshift(captchaField);
    
    // Group fields into steps
    const steps = [];
    for (let i = 0; i < fields.length; i += fieldsPerPage) {
        steps.push(fields.slice(i, i + fieldsPerPage));
    }

    function renderStep(stepIndex) {
        dynamicFormContainer.empty();
        const stepFields = steps[stepIndex];

        stepFields.forEach(field => {
            let formGroup = `
                <div class="form-group mb-3">
                    <label for="${field.name}">
                        ${field.label}
                        ${field.required ? '<span class="text-danger">*</span>' : ''}
                    </label>
            `;

            switch (field.type) {
                case 'text':
                case 'email':
                case 'password':
                case 'number':
                case 'tel':
                case 'url':
                    formGroup += `<input type="${field.type}" class="form-control" id="${field.name}" name="${field.name}" ${field.required ? 'required' : ''}>`;
                    break;
                case 'textarea':
                    formGroup += `<textarea class="form-control" id="${field.name}" name="${field.name}" ${field.required ? 'required' : ''}></textarea>`;
                    break;
                case 'select':
                    formGroup += `<select class="form-control" id="${field.name}" name="${field.name}" ${field.required ? 'required' : ''}>`;
                    field.options.forEach(option => {
                        formGroup += `<option value="${option}">${option}</option>`;
                    });
                    formGroup += `</select>`;
                    break;
                case 'radio':
                    field.options.forEach(option => {
                        formGroup += `
                            <div class="form-check form-check-inline">
                                <input class="form-check-input" type="radio" name="${field.name}" id="${field.name}-${option}" value="${option}" ${field.required ? 'required' : ''}>
                                <label class="form-check-label" for="${field.name}-${option}">${option}</label>
                            </div>
                        `;
                    });
                    break;
                case 'checkbox':
                    field.options.forEach(option => {
                        formGroup += `
                            <div class="form-check">
                                <input class="form-check-input" type="checkbox" name="${field.name}[]" id="${field.name}-${option}" value="${option}">
                                <label class="form-check-label" for="${field.name}-${option}">${option}</label>
                            </div>
                        `;
                    });
                    break;
                case 'date':
                    formGroup += `<input type="date" class="form-control" id="${field.name}" name="${field.name}" ${field.required ? 'required' : ''}>`;
                    break;
            }

            if (field.help_text) {
                formGroup += `<small class="form-text text-muted">${field.help_text}</small>`;
            }
            
            formGroup += `</div>`;
            dynamicFormContainer.append(formGroup);
        });

        updateButtons();
    }

    function updateButtons() {
        prevButton.toggle(currentStep > 0);
        nextButton.toggle(currentStep < steps.length - 1);
        registerButton.toggle(currentStep === steps.length - 1);
    }

    function validateStep(stepIndex) {
        let isValid = true;
        const stepFields = steps[stepIndex];
        stepFields.forEach(field => {
            if (field.required) {
                const input = $(`#${field.name}`);
                if (input.is(':checkbox') || input.is(':radio')) {
                    if ($(`input[name="${field.name}"]:checked`).length === 0) {
                        isValid = false;
                    }
                } else if (!input.val()) {
                    isValid = false;
                }
            }
        });
        return isValid;
    }

    nextButton.on('click', function() {
        if (validateStep(currentStep)) {
            currentStep++;
            renderStep(currentStep);
        } else {
            alert('Please fill out all required fields to proceed.');
        }
    });

    prevButton.on('click', function() {
        currentStep--;
        renderStep(currentStep);
    });

    // Initial render
    renderStep(currentStep);
});
                                        <?php echo htmlspecialchars($type['type_name']); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="form-group mb-3">
                            <label for="payment_cycle">Payment Cycle (Year)</label>
                            <input type="number" name="payment_cycle" id="payment_cycle" class="form-control" value="<?php echo date('Y'); ?>" required>
                        </div>
                        <button type="submit" name="add_obligation" class="btn btn-success">Add Obligation</button>
                    </form>
                </div>
            </div>
            <?php endif; ?>

        </main>
    </div>
</div>
<?php require_once 'includes/footer.php'; ?>