<?php
require_once '../includes/functions.php';
require_once '../includes/db_config.php';
session_start();

// Check if user is logged in and is admin
if (!is_logged_in() || !is_admin()) {
    http_response_code(403);
    exit('Access denied');
}

// Check database connection
if (!$link) {
    http_response_code(500);
    exit("ERROR: Could not connect. " . mysqli_connect_error());
}

// Filter settings
$record_type = isset($_GET['record_type']) && in_array($_GET['record_type'], ['all', 'expenses']) ? $_GET['record_type'] : 'all';
$time_period = isset($_GET['time_period']) && in_array($_GET['time_period'], ['date', 'week', 'month', 'year', 'all']) ? $_GET['time_period'] : 'all';
$filter_date = isset($_GET['filter_date']) ? $_GET['filter_date'] : date('Y-m-d');

// Build time filter
$time_filter = '';
$bind_params = [];
$bind_types = '';
if ($time_period !== 'all') {
    switch ($time_period) {
        case 'date':
            $time_filter = " AND DATE(e.created_at) = ?";
            $filter_date_param = $filter_date;
            $bind_params[] = &$filter_date_param;
            $bind_types .= 's';
            break;
        case 'week':
            $time_filter = " AND e.created_at >= DATE_SUB(CURDATE(), INTERVAL 1 WEEK)";
            break;
        case 'month':
            $time_filter = " AND e.created_at >= DATE_SUB(CURDATE(), INTERVAL 1 MONTH)";
            break;
        case 'year':
            $time_filter = " AND e.created_at >= DATE_SUB(CURDATE(), INTERVAL 1 YEAR)";
            break;
    }
}

// Fetch expenses data
$sql = "SELECT 
            e.id,
            et.type_name AS expense_type,
            e.amount,
            IFNULL(e.description, 'N/A') AS description,
            e.created_at
        FROM expenses e
        LEFT JOIN expense_types et ON e.expense_type_id = et.id
        WHERE 1=1" . ($record_type == 'all' || $record_type == 'expenses' ? $time_filter : ' AND FALSE') . "
        ORDER BY e.created_at DESC";
$stmt = mysqli_prepare($link, $sql);
if ($time_period == 'date' && !empty($bind_params)) {
    mysqli_stmt_bind_param($stmt, $bind_types, ...$bind_params);
}
mysqli_stmt_execute($stmt);
$result = mysqli_stmt_get_result($stmt);
$expenses = [];
while ($row = mysqli_fetch_assoc($result)) {
    $expenses[] = $row;
}
mysqli_stmt_close($stmt);

// Set headers for CSV download
header('Content-Type: text/csv; charset=utf-8');
header('Content-Disposition: attachment; filename="expenses_export_' . date('Ymd_His') . '.csv"');

// Create CSV file
$output = fopen('php://output', 'w');
fputcsv($output, ['Type', 'Amount', 'Description', 'Date']);

foreach ($expenses as $record) {
    fputcsv($output, [
        $record['expense_type'] ?? 'N/A',
        number_format($record['amount'] ?? 0, 2),
        $record['description'],
        date('Y-m-d', strtotime($record['created_at'] ?? 'now'))
    ]);
}

fclose($output);
mysqli_close($link);
exit;
?>