<?php
require_once 'includes/functions.php';
require_once 'includes/db_config.php';
session_start();

// Check if user is logged in
if (!is_logged_in()) {
    redirect("login.php");
}

$message = '';
$success = '';

// Handle form submission
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    $form_id = intval($_POST['form_id']);
    $form_data = json_encode($_POST['form_data'] ?? []);
    $user_id = $_SESSION['user_id'];

    // Insert or update form data
    $sql = "INSERT INTO form_data (form_id, user_id, data, created_at) VALUES (?, ?, ?, NOW()) 
            ON DUPLICATE KEY UPDATE data = ?, created_at = NOW()";
    if ($stmt = mysqli_prepare($link, $sql)) {
        mysqli_stmt_bind_param($stmt, "iiss", $form_id, $user_id, $form_data, $form_data);
        if (mysqli_stmt_execute($stmt)) {
            $success = "Form data saved successfully!";
        } else {
            $message = "Error saving form data: " . mysqli_error($link);
        }
        mysqli_stmt_close($stmt);
    } else {
        $message = "Database error: Could not prepare statement.";
    }
}

// Fetch available forms
$sql = "SELECT id, form_name, form_data FROM forms ORDER BY created_at DESC";
$result = mysqli_query($link, $sql);
$forms = [];
if ($result) {
    while ($row = mysqli_fetch_assoc($result)) {
        $forms[] = $row;
    }
}

// Fetch user-submitted form data
$user_id = $_SESSION['user_id'];
$submitted_forms = [];
$sql = "SELECT form_id, data FROM form_data WHERE user_id = ?";
if ($stmt = mysqli_prepare($link, $sql)) {
    mysqli_stmt_bind_param($stmt, "i", $user_id);
    mysqli_stmt_execute($stmt);
    $result = mysqli_stmt_get_result($stmt);
    while ($row = mysqli_fetch_assoc($result)) {
        $submitted_forms[$row['form_id']] = json_decode($row['data'], true);
    }
    mysqli_stmt_close($stmt);
}
?>

<?php require_once 'includes/header.php'; ?>
<div class="container-fluid">
    <div class="row">
        <?php require_once 'includes/sidebar.php'; ?>
        <main class="col-md-9 col-lg-10 main-content">
            <h2>Available Forms</h2>

            <?php if ($success): ?>
                <div class="message success-message"><?php echo htmlspecialchars($success); ?></div>
            <?php endif; ?>
            <?php if ($message): ?>
                <div class="message error-message"><?php echo htmlspecialchars($message); ?></div>
            <?php endif; ?>

            <div class="forms-list">
                <?php if (empty($forms)): ?>
                    <p>No forms available at the moment.</p>
                <?php else: ?>
                    <?php foreach ($forms as $form): ?>
                        <?php
                        $form_structure = json_decode($form['form_data'], true);
                        $form_id = $form['id'];
                        $existing_data = $submitted_forms[$form_id] ?? [];
                        ?>
                        <div class="card mb-3">
                            <div class="card-header"><?php echo htmlspecialchars($form['form_name']); ?></div>
                            <div class="card-body">
                                <form action="forms.php" method="post">
                                    <input type="hidden" name="form_id" value="<?php echo htmlspecialchars($form_id); ?>">
                                    <?php if (isset($form_structure['fields']) && is_array($form_structure['fields'])): ?>
                                        <?php foreach ($form_structure['fields'] as $field): ?>
                                            <div class="form-group">
                                                <label for="<?php echo htmlspecialchars($field['name']); ?>">
                                                    <?php echo htmlspecialchars($field['label']); ?>
                                                    <?php if ($field['required']): ?><span class="text-danger">*</span><?php endif; ?>
                                                </label>
                                                <?php
                                                $field_value = $existing_data[$field['name']] ?? '';
                                                switch ($field['type']) {
                                                    case 'text':
                                                    case 'email':
                                                    case 'password':
                                                    case 'username':
                                                    case 'date':
                                                        echo "<input type='{$field['type']}' name='form_data[{$field['name']}]' id='{$field['name']}' class='form-control' value='" . htmlspecialchars($field_value) . "' " . ($field['required'] ? 'required' : '') . ">";
                                                        break;
                                                    case 'textarea':
                                                        echo "<textarea name='form_data[{$field['name']}]' id='{$field['name']}' class='form-control' " . ($field['required'] ? 'required' : '') . ">" . htmlspecialchars($field_value) . "</textarea>";
                                                        break;
                                                    case 'checkbox':
                                                        echo "<input type='checkbox' name='form_data[{$field['name']}]' id='{$field['name']}' value='1' " . ($field_value ? 'checked' : '') . " " . ($field['required'] ? 'required' : '') . ">";
                                                        break;
                                                    case 'radio':
                                                        if (isset($field['options']) && is_array($field['options'])) {
                                                            foreach ($field['options'] as $option) {
                                                                echo "<div class='form-check'>";
                                                                echo "<input type='radio' name='form_data[{$field['name']}]' id='{$field['name']}_{$option}' value='{$option}' class='form-check-input' " . ($field_value === $option ? 'checked' : '') . " " . ($field['required'] ? 'required' : '') . ">";
                                                                echo "<label class='form-check-label' for='{$field['name']}_{$option}'>{$option}</label>";
                                                                echo "</div>";
                                                            }
                                                        }
                                                        break;
                                                    case 'select':
                                                        if (isset($field['options']) && is_array($field['options'])) {
                                                            echo "<select name='form_data[{$field['name']}]' id='{$field['name']}' class='form-control' " . ($field['required'] ? 'required' : '') . ">";
                                                            echo "<option value=''>Select an option</option>";
                                                            foreach ($field['options'] as $option) {
                                                                echo "<option value='{$option}' " . ($field_value === $option ? 'selected' : '') . ">{$option}</option>";
                                                            }
                                                            echo "</select>";
                                                        }
                                                        break;
                                                    case 'file':
                                                        echo "<input type='file' name='form_data[{$field['name']}]' id='{$field['name']}' class='form-control-file' " . ($field['required'] ? 'required' : '') . ">";
                                                        break;
                                                }
                                                ?>
                                            </div>
                                        <?php endforeach; ?>
                                    <?php endif; ?>
                                    <button type="submit" class="btn btn-primary">Submit Form</button>
                                </form>
                            </div>
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
        </main>
    </div>
</div>
<?php require_once 'includes/footer.php'; ?>