<?php
// token_settings.php
ob_start();
session_start();
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);
require_once 'includes/functions.php';
require_once 'includes/db_config.php';

// ---------- SESSION ----------
$timeout = 1800;
if (isset($_SESSION['last_activity']) && (time() - $_SESSION['last_activity']) > $timeout) {
    session_unset(); session_destroy(); redirect("login.php?timeout=1");
}
$_SESSION['last_activity'] = time();

// ---------- SUPER ADMIN ONLY ----------
if (!is_logged_in() || $_SESSION['user_role'] !== 'super_admin') {
    redirect("login.php");
}

// ---------- CSRF ----------
if (!isset($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

$message = $message_type = '';
$currency_symbol = get_setting('currency_symbol', $link) ?? '₦';

// ---------------------------------------------------------------------
// 1. TOKEN PACKAGE HANDLING
// ---------------------------------------------------------------------
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['save_package'])
    && isset($_POST['csrf_token']) && hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {

    $package_id   = isset($_POST['package_id']) ? intval($_POST['package_id']) : 0;
    $package_name = sanitize_input($_POST['package_name'] ?? '');
    $quantity     = intval($_POST['quantity'] ?? 0);
    $price        = floatval($_POST['price'] ?? 0);
    $status       = sanitize_input($_POST['status'] ?? 'active');

    if (empty($package_name) || $quantity <= 0 || $price < 0 || !in_array($status, ['active', 'inactive'])) {
        $message = "Please fill all fields with valid values.";
        $message_type = 'danger';
    } else {
        if ($package_id > 0) {
            $sql = "UPDATE token_packages SET package_name = ?, quantity = ?, price = ?, status = ? WHERE id = ? AND created_by = ?";
            $stmt = mysqli_prepare($link, $sql);
            mysqli_stmt_bind_param($stmt, "sidsii", $package_name, $quantity, $price, $status, $package_id, $_SESSION['user_id']);
        } else {
            $sql = "INSERT INTO token_packages (package_name, quantity, price, status, created_by, created_at)
                    VALUES (?, ?, ?, ?, ?, NOW())";
            $stmt = mysqli_prepare($link, $sql);
            mysqli_stmt_bind_param($stmt, "sidsi", $package_name, $quantity, $price, $status, $_SESSION['user_id']);
        }

        if ($stmt && mysqli_stmt_execute($stmt)) {
            $message = $package_id ? "Package updated." : "Package added.";
            $message_type = 'success';
        } else {
            $message = "DB error: " . mysqli_error($link);
            $message_type = 'danger';
        }
        if ($stmt) mysqli_stmt_close($stmt);
    }
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

// ---------------------------------------------------------------------
// 2. TOKEN COSTS
// ---------------------------------------------------------------------
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['save_token_costs'])
    && isset($_POST['csrf_token']) && hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {

    $tokens_per_payment = intval($_POST['tokens_per_payment'] ?? 1);
    $tokens_per_expense = intval($_POST['tokens_per_expense'] ?? 1);
    $tokens_per_benefit = intval($_POST['tokens_per_benefit'] ?? 1);
    $deduct_per_member  = intval($_POST['deduct_per_member'] ?? 0);

    if ($tokens_per_payment < 0 || $tokens_per_expense < 0 || $tokens_per_benefit < 0) {
        $message = "Token costs must be non-negative.";
        $message_type = 'danger';
    } else {
        $ok = true;
        foreach ([
            'tokens_per_payment'=> $tokens_per_payment,
            'tokens_per_expense'=> $tokens_per_expense,
            'tokens_per_benefit'=> $tokens_per_benefit,
            'deduct_per_member'=> $deduct_per_member
        ] as $k=>$v) {
            if (!update_setting($link, $k, $v)) $ok = false;
        }
        $message = $ok ? "Token costs saved." : "Error saving settings.";
        $message_type = $ok ? 'success' : 'danger';
    }
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

// ---------------------------------------------------------------------
// 3. SUPER-ADMIN FEATURES
// ---------------------------------------------------------------------

// ---- 3.1 Reset all admin / accountant token balances ----
if (isset($_POST['reset_tokens']) && isset($_POST['csrf_token'])
    && hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {

    mysqli_begin_transaction($link);
    $ok = true;

    // Log
    $log_sql = "INSERT INTO token_usage_logs (user_id, action, tokens_used, description, created_at)
                VALUES (?, ?, 0, ?, NOW())";
    $log_stmt = mysqli_prepare($link, $log_sql);
    $action = 'reset';
    $desc = "SUPER ADMIN RESET ALL TOKENS";
    mysqli_stmt_bind_param($log_stmt, "iss", $_SESSION['user_id'], $action, $desc);
    if (!mysqli_stmt_execute($log_stmt)) $ok = false;
    mysqli_stmt_close($log_stmt);

    // Zero all
    $zero_sql = "UPDATE token_balances tb
                 JOIN users u ON tb.user_id = u.id
                 SET tb.balance = 0
                 WHERE u.role IN ('admin','accountant') AND u.deleted_at IS NULL";
    if (!mysqli_query($link, $zero_sql)) $ok = false;

    if ($ok) {
        mysqli_commit($link);
        $message = "All admin & accountant tokens reset to 0.";
        $message_type = 'success';
    } else {
        mysqli_rollback($link);
        $message = "Failed to reset tokens.";
        $message_type = 'danger';
    }
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

// ---- 3.2 CREDIT / WITHDRAW (FIXED WITH ON DUPLICATE) ----
if (isset($_POST['adjust_tokens']) && isset($_POST['csrf_token'])
    && hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {

    $target_user_id = intval($_POST['target_user_id'] ?? 0);
    $adjust_amount  = intval($_POST['adjust_amount'] ?? 0);
    $reason         = sanitize_input($_POST['adjust_reason'] ?? '');

    if ($target_user_id <= 0 || $adjust_amount == 0) {
        $message = "Select a user and non-zero amount.";
        $message_type = 'danger';
    } else {
        // Verify target
        $chk = mysqli_prepare($link,
            "SELECT id FROM users WHERE id=? AND role IN ('admin','accountant') AND deleted_at IS NULL");
        mysqli_stmt_bind_param($chk, "i", $target_user_id);
        mysqli_stmt_execute($chk);
        mysqli_stmt_store_result($chk);
        if (mysqli_stmt_num_rows($chk) == 0) {
            $message = "Invalid target user.";
            $message_type = 'danger';
            mysqli_stmt_close($chk);
        } else {
            mysqli_stmt_close($chk);
            mysqli_begin_transaction($link);
            $ok = true;

            // INSERT OR UPDATE BALANCE
            $upsert = mysqli_prepare($link,
                "INSERT INTO token_balances (user_id, balance, created_at, updated_at)
                 VALUES (?, ?, NOW(), NOW())
                 ON DUPLICATE KEY UPDATE
                 balance = GREATEST(0, balance + VALUES(balance)),
                 updated_at = NOW()"
            );
            mysqli_stmt_bind_param($upsert, "ii", $target_user_id, $adjust_amount);
            if (!mysqli_stmt_execute($upsert)) {
                $ok = false;
                $message = "Failed to update balance.";
                $message_type = 'danger';
            }
            mysqli_stmt_close($upsert);

            // Log
            $action = $adjust_amount > 0 ? 'credit' : 'withdraw';
            $abs_amount = abs($adjust_amount);
            $desc = "SUPER ADMIN $action: $reason";

            $log = mysqli_prepare($link,
                "INSERT INTO token_usage_logs (user_id, action, tokens_used, description, created_at)
                 VALUES (?, ?, ?, ?, NOW())");
            mysqli_stmt_bind_param($log, "isis", $_SESSION['user_id'], $action, $abs_amount, $desc);
            if (!mysqli_stmt_execute($log)) $ok = false;
            mysqli_stmt_close($log);

            if ($ok) {
                mysqli_commit($link);
                $message = ($adjust_amount > 0 ? "Credited" : "Withdrew")
                         . " " . abs($adjust_amount) . " token(s).";
                $message_type = 'success';
            } else {
                mysqli_rollback($link);
                $message = "Failed to adjust tokens.";
                $message_type = 'danger';
            }
        }
    }
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

// ---- 3.3 Clear records ----
if (isset($_POST['clear_records']) && isset($_POST['csrf_token'])
    && hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {

    $type = sanitize_input($_POST['record_type'] ?? '');

    $allowed = ['payments','expenses','benefits','all'];
    if (!in_array($type, $allowed)) {
        $message = "Invalid record type.";
        $message_type = 'danger';
    } else {
        mysqli_begin_transaction($link);
        $ok = true;
        $tables = [];

        if ($type === 'all' || $type === 'payments')   $tables[] = 'payments';
        if ($type === 'all' || $type === 'expenses')   $tables[] = 'expenses';
        if ($type === 'all' || $type === 'benefits')   $tables[] = 'member_benefits';

        foreach ($tables as $tbl) {
            $sql = "TRUNCATE TABLE `$tbl`";
            if (!mysqli_query($link, $sql)) $ok = false;
        }

        // Log
        $action = 'clear_records';
        $desc = "SUPER ADMIN CLEARED RECORDS: " . strtoupper($type);

        $log = mysqli_prepare($link,
            "INSERT INTO token_usage_logs (user_id, action, tokens_used, description, created_at)
             VALUES (?, ?, 0, ?, NOW())");
        mysqli_stmt_bind_param($log, "iss", $_SESSION['user_id'], $action, $desc);
        if (!mysqli_stmt_execute($log)) $ok = false;
        mysqli_stmt_close($log);

        if ($ok) {
            mysqli_commit($link);
            $message = "Records cleared ($type).";
            $message_type = 'success';
        } else {
            mysqli_rollback($link);
            $message = "Failed to clear records.";
            $message_type = 'danger';
        }
    }
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

// ---------------------------------------------------------------------
// FETCH DATA
// ---------------------------------------------------------------------
$token_packages = [];
$sql = "SELECT id, package_name, quantity, price, status FROM token_packages WHERE created_by = ? ORDER BY created_at DESC";
$stmt = mysqli_prepare($link, $sql);
mysqli_stmt_bind_param($stmt, "i", $_SESSION['user_id']);
mysqli_stmt_execute($stmt);
$res = mysqli_stmt_get_result($stmt);
while ($row = mysqli_fetch_assoc($res)) $token_packages[] = $row;
mysqli_stmt_close($stmt);

$tokens_per_payment = get_setting('tokens_per_payment', $link) ?? 1;
$tokens_per_expense = get_setting('tokens_per_expense', $link) ?? 1;
$tokens_per_benefit = get_setting('tokens_per_benefit', $link) ?? 1;
$deduct_per_member  = get_setting('deduct_per_member', $link) ?? 0;

$admins = [];
$res = mysqli_query($link,
    "SELECT id, full_name, role FROM users
     WHERE role IN ('admin','accountant') AND deleted_at IS NULL
     ORDER BY full_name");
while ($row = mysqli_fetch_assoc($res)) $admins[] = $row;
mysqli_free_result($res);

ob_end_clean();
?>
<?php require_once 'includes/header.php'; ?>
<script>document.title = 'Token Settings - <?=htmlspecialchars(get_setting('system_name',$link)??'Club Portal')?>';</script>
<style>.main-content{padding:20px;}</style>

<div class="container-fluid">
    <div class="row">
        <?php require_once 'includes/sidebar.php'; ?>
        <main class="col-md-9 col-lg-10 main-content">
            <h2>Token Settings</h2>
            <?php if ($message): ?>
                <div class="alert alert-<?=$message_type?>"><?=$message?></div>
            <?php endif; ?>

            <!-- TOKEN PACKAGES -->
            <div class="card mb-4">
                <div class="card-header">Add / Edit Token Package</div>
                <div class="card-body">
                    <form action="" method="post">
                        <input type="hidden" name="save_package" value="1">
                        <input type="hidden" name="csrf_token" value="<?=htmlspecialchars($_SESSION['csrf_token'])?>">
                        <input type="hidden" name="package_id" id="package_id" value="">
                        <div class="row g-3">
                            <div class="col-md-3"><label class="form-label">Package Name</label>
                                <input type="text" name="package_name" id="package_name" class="form-control" required></div>
                            <div class="col-md-2"><label class="form-label">Quantity</label>
                                <input type="number" name="quantity" id="quantity" class="form-control" min="1" required></div>
                            <div class="col-md-2"><label class="form-label">Price (<?=$currency_symbol?>)</label>
                                <input type="number" name="price" id="price" class="form-control" min="0" step="0.01" required></div>
                            <div class="col-md-2"><label class="form-label">Status</label>
                                <select name="status" id="status" class="form-select" required>
                                    <option value="active">Active</option>
                                    <option value="inactive">Inactive</option>
                                </select></div>
                            <div class="col-md-3 d-flex align-items-end">
                                <button type="submit" class="btn btn-primary w-100">Save</button>
                            </div>
                        </div>
                    </form>
                </div>
            </div>

            <!-- TOKEN COSTS -->
            <div class="card mb-4">
                <div class="card-header">Token Costs per Record</div>
                <div class="card-body">
                    <form action="" method="post">
                        <input type="hidden" name="save_token_costs" value="1">
                        <input type="hidden" name="csrf_token" value="<?=htmlspecialchars($_SESSION['csrf_token'])?>">
                        <div class="row g-3">
                            <div class="col-md-3"><label class="form-label">Per Payment</label>
                                <input type="number" name="tokens_per_payment" class="form-control" min="0" value="<?=htmlspecialchars($tokens_per_payment)?>" required></div>
                            <div class="col-md-3"><label class="form-label">Per Expense</label>
                                <input type="number" name="tokens_per_expense" class="form-control" min="0" value="<?=htmlspecialchars($tokens_per_expense)?>" required></div>
                            <div class="col-md-3"><label class="form-label">Per Benefit</label>
                                <input type="number" name="tokens_per_benefit" class="form-control" min="0" value="<?=htmlspecialchars($tokens_per_benefit)?>" required></div>
                            <div class="col-md-3"><label class="form-label">Deduct per Member (0/1)</label>
                                <input type="number" name="deduct_per_member" class="form-control" min="0" max="1" value="<?=htmlspecialchars($deduct_per_member)?>" required></div>
                            <div class="col-12 text-end">
                                <button type="submit" class="btn btn-primary">Save Costs</button>
                            </div>
                        </div>
                    </form>
                </div>
            </div>

            <!-- SUPER ADMIN TOOLS -->
            <div class="card mb-4">
                <div class="card-header text-danger">SUPER ADMIN TOOLS</div>
                <div class="card-body">

                    <!-- Reset Tokens -->
                    <form action="" method="post" class="mb-4" onsubmit="return confirm('WARNING: This will ZERO all admin/accountant tokens. Continue?');">
                        <input type="hidden" name="reset_tokens" value="1">
                        <input type="hidden" name="csrf_token" value="<?=htmlspecialchars($_SESSION['csrf_token'])?>">
                        <button type="submit" class="btn btn-danger">Reset ALL Admin / Accountant Tokens</button>
                    </form>

                    <!-- Credit / Withdraw -->
                    <form action="" method="post" class="mb-4">
                        <input type="hidden" name="adjust_tokens" value="1">
                        <input type="hidden" name="csrf_token" value="<?=htmlspecialchars($_SESSION['csrf_token'])?>">
                        <div class="row g-3 align-items-end">
                            <div class="col-md-4">
                                <label class="form-label">Select Admin / Accountant</label>
                                <select name="target_user_id" class="form-select" required>
                                    <option value="">-- choose --</option>
                                    <?php foreach ($admins as $a): ?>
                                        <option value="<?=htmlspecialchars($a['id'])?>">
                                            <?=htmlspecialchars($a['full_name'] . ' (' . ucfirst($a['role']) . ')')?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div class="col-md-2">
                                <label class="form-label">Amount</label>
                                <input type="number" name="adjust_amount" class="form-control" required placeholder="+ = credit, - = withdraw">
                            </div>
                            <div class="col-md-4">
                                <label class="form-label">Reason</label>
                                <input type="text" name="adjust_reason" class="form-control" required>
                            </div>
                            <div class="col-md-2">
                                <button type="submit" class="btn btn-warning w-100">Apply</button>
                            </div>
                        </div>
                    </form>

                    <!-- Clear Records -->
                    <form action="" method="post" onsubmit="return confirm('WARNING: This will DELETE selected records permanently. Continue?');">
                        <input type="hidden" name="clear_records" value="1">
                        <input type="hidden" name="csrf_token" value="<?=htmlspecialchars($_SESSION['csrf_token'])?>">
                        <div class="row g-3 align-items-center">
                            <div class="col-md-4">
                                <label class="form-label">Record Type</label>
                                <select name="record_type" class="form-select" required>
                                    <option value="">-- select --</option>
                                    <option value="payments">Payments</option>
                                    <option value="expenses">Expenses</option>
                                    <option value="benefits">Benefits</option>
                                    <option value="all">ALL Records</option>
                                </select>
                            </div>
                            <div class="col-md-2">
                                <button type="submit" class="btn btn-danger w-100">CLEAR</button>
                            </div>
                        </div>
                    </form>
                </div>
            </div>

            <!-- EXISTING PACKAGES -->
            <div class="card">
                <div class="card-header">Existing Token Packages</div>
                <div class="card-body">
                    <?php if (empty($token_packages)): ?>
                        <p class="text-muted">No packages.</p>
                    <?php else: ?>
                        <div class="table-responsive">
                            <table class="table table-striped">
                                <thead>
                                    <tr><th>Name</th><th>Qty</th><th>Price (<?=$currency_symbol?>)</th><th>Status</th><th>Actions</th></tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($token_packages as $p): ?>
                                        <tr>
                                            <td><?=htmlspecialchars($p['package_name'])?></td>
                                            <td><?=htmlspecialchars($p['quantity'])?></td>
                                            <td><?=number_format($p['price'],2)?></td>
                                            <td><?=ucfirst($p['status'])?></td>
                                            <td>
                                                <button class="btn btn-sm btn-primary edit-pkg"
                                                        data-id="<?=$p['id']?>"
                                                        data-name="<?=htmlspecialchars($p['package_name'])?>"
                                                        data-qty="<?=$p['quantity']?>"
                                                        data-price="<?=$p['price']?>"
                                                        data-status="<?=$p['status']?>">
                                                    Edit
                                                </button>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </main>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', () => {
    document.querySelectorAll('.edit-pkg').forEach(btn => {
        btn.addEventListener('click', () => {
            document.getElementById('package_id').value   = btn.dataset.id;
            document.getElementById('package_name').value = btn.dataset.name;
            document.getElementById('quantity').value     = btn.dataset.qty;
            document.getElementById('price').value        = btn.dataset.price;
            document.getElementById('status').value       = btn.dataset.status;
        });
    });
});
</script>

<?php require_once 'includes/footer.php'; ?>