<?php
// record_expenses.php
ob_start();
session_start();
ini_set('display_errors',1);
ini_set('display_startup_errors',1);
error_reporting(E_ALL);
mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);
if (function_exists('opcache_reset')) { opcache_reset(); }

require_once 'includes/functions.php';
require_once 'includes/db_config.php';

/* ---------- SESSION / PERMISSIONS ---------- */
$timeout = 1800;
if (isset($_SESSION['last_activity']) && (time() - $_SESSION['last_activity']) > $timeout) {
    session_unset(); session_destroy();
    redirect("login.php?timeout=1");
}
$_SESSION['last_activity'] = time();

if (!isset($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}
if (!is_logged_in() || !user_has_permission($link, 'view_expenses') ||
    !in_array($_SESSION['user_role'], ['admin','accountant'])) {
    redirect("login.php"); exit;
}

/* ---------- SETTINGS ---------- */
$currency_symbol = get_setting('currency_symbol',$link) ?? '₦';
$token_cost      = (int)get_setting('tokens_per_expense',$link) ?: 1;

/* ---------- ORGANIZATION BALANCE FUNCTION ---------- */
function get_organization_token_balance($link) {
    $sql = "SELECT COALESCE(SUM(tb.balance), 0) AS total
            FROM token_balances tb
            JOIN users u ON tb.user_id = u.id
            WHERE u.role IN ('admin', 'accountant')
              AND u.deleted_at IS NULL
              AND tb.balance > 0";
    $result = mysqli_query($link, $sql);
    if ($result && $row = mysqli_fetch_assoc($result)) {
        mysqli_free_result($result);
        return (int)$row['total'];
    }
    return 0;
}

/* ---------- INITIAL BALANCES ---------- */
$current_user_balance      = get_user_token_balance($link, $_SESSION['user_id']);
$organization_total_tokens = get_organization_token_balance($link);

/* ---------- FORM HANDLING ---------- */
$message = $message_type = '';
if ($_SERVER["REQUEST_METHOD"]=="POST"
    && isset($_POST['log_expense'])
    && isset($_POST['csrf_token'])
    && hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {

    $amount          = floatval($_POST['amount'] ?? 0);
    $expense_type_id = intval($_POST['expense_type_id'] ?? 0);
    $description     = sanitize_input($_POST['description'] ?? '');

    $required_tokens = $token_cost;

    if ($amount<=0 || $expense_type_id<=0) {
        $message = "Please provide a valid amount and expense type.";
        $message_type = 'danger';
    } elseif ($current_user_balance < $required_tokens) {
        $message = "Insufficient tokens. Need $required_tokens token".($required_tokens>1?'s':'')
                 .", you have $current_user_balance. <a href='token_store.php'>Buy / request more</a>.";
        $message_type = 'danger';
    } else {
        $stmt = mysqli_prepare($link,"SELECT 1 FROM expense_types WHERE id=? AND deleted_at IS NULL");
        mysqli_stmt_bind_param($stmt,"i",$expense_type_id);
        mysqli_stmt_execute($stmt);
        mysqli_stmt_store_result($stmt);
        if (mysqli_stmt_num_rows($stmt)===0) {
            $message = "Invalid expense type."; $message_type='danger';
            mysqli_stmt_close($stmt);
        } else {
            mysqli_stmt_close($stmt);
            mysqli_begin_transaction($link);
            $ok = true;

            $tokens_used = deduct_tokens($link,$_SESSION['user_id'],'expense',$required_tokens);
            if ($tokens_used===false) { $ok=false; $message="Token deduction failed."; $message_type='danger'; }

            if ($ok) {
                $sql = "INSERT INTO expenses (expense_type_id,amount,description,created_at)
                        VALUES (?,?,?,NOW())";
                $stmt = mysqli_prepare($link,$sql);
                mysqli_stmt_bind_param($stmt,"ids",$expense_type_id,$amount,$description);
                if (!mysqli_stmt_execute($stmt)) {
                    $ok=false; $message="Insert error: ".mysqli_stmt_error($stmt); $message_type='danger';
                }
                mysqli_stmt_close($stmt);
            }

            if ($ok) {
                mysqli_commit($link);
                $message = "Expense logged! Used $tokens_used token".($tokens_used>1?'s':'').".";
                $message_type = 'success';

                /* ---- REFRESH BALANCES ---- */
                $current_user_balance      = get_user_token_balance($link, $_SESSION['user_id']);
                $organization_total_tokens = get_organization_token_balance($link);
            } else {
                mysqli_rollback($link);
                if (!$message) $message = "Transaction rolled back.";
                $message_type = 'danger';
            }
        }
    }
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
} elseif ($_SERVER["REQUEST_METHOD"]=="POST") {
    $message = "Invalid CSRF token.";
    $message_type = 'danger';
}

/* ---------- EXPENSE TYPES ---------- */
$expense_types = [];
$res = mysqli_query($link,
    "SELECT id,type_name FROM expense_types WHERE deleted_at IS NULL ORDER BY type_name");
while ($row = mysqli_fetch_assoc($res)) $expense_types[] = $row;
mysqli_free_result($res);
?>
<?php require_once 'includes/header.php'; ?>
<script>document.title='Record Expenses - <?=htmlspecialchars(get_setting('system_name',$link)??'Club Portal')?>';</script>
<style>.main-content{padding:20px;}</style>

<div class="container-fluid">
    <div class="row">
        <?php require_once 'includes/sidebar.php'; ?>
        <main class="col-md-9 col-lg-10 main-content">
            <h2>Record Expenses</h2>
            <?php if ($message): ?>
                <div class="alert alert-<?=$message_type?>" role="alert"><?=$message?></div>
            <?php endif; ?>

            <div class="card mb-4">
                <div class="card-header">Log New Expense</div>
                <div class="card-body">
                    <p><strong>Your Token Balance:</strong> <?=$current_user_balance?> tokens</p>
                    <p><strong>Organization Token Balance:</strong> <?=$organization_total_tokens?> tokens</p>

                    <form action="record_expenses.php" method="post">
                        <input type="hidden" name="log_expense" value="1">
                        <input type="hidden" name="csrf_token" value="<?=htmlspecialchars($_SESSION['csrf_token'])?>">

                        <div class="mb-3">
                            <label class="form-label">Expense Type</label>
                            <select name="expense_type_id" class="form-select" required>
                                <option value="">Select Type</option>
                                <?php foreach ($expense_types as $t): ?>
                                    <option value="<?=htmlspecialchars($t['id'])?>"><?=htmlspecialchars($t['type_name'])?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Amount (<?=$currency_symbol?>)</label>
                            <input type="number" name="amount" class="form-control" step="0.01" min="0.01" required>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Description (optional)</label>
                            <textarea name="description" class="form-control" rows="3"></textarea>
                        </div>

                        <button type="submit" class="btn btn-primary">
                            Log Expense (<?=$token_cost?> token<?=$token_cost>1?'s':''?>)
                        </button>
                    </form>
                </div>
            </div>
        </main>
    </div>
</div>

<?php require_once 'includes/footer.php'; ?>