<?php
require_once 'includes/functions.php';
require_once 'includes/db_config.php';
session_start();

if (!is_logged_in() || !is_admin()) {
    redirect("login.php");
}

$message = '';
$message_type = '';

if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['add_benefit'])) {
    $member_id = intval($_POST['member_id']);
    $benefit_type_id = intval($_POST['benefit_type_id']);
    $description = sanitize_input($_POST['description'] ?? '');
    $date_given = sanitize_input($_POST['date_given']);

    $stmt = $link->prepare("INSERT INTO member_benefits (member_id, benefit_type_id, description, date_given) VALUES (?, ?, ?, ?)");
    $stmt->bind_param("iiss", $member_id, $benefit_type_id, $description, $date_given);
    
    if ($stmt->execute()) {
        $message = "Benefit added successfully.";
        $message_type = "success";
    } else {
        $message = "Error adding benefit: " . $stmt->error;
        $message_type = "danger";
    }
    $stmt->close();
}

$members = [];
$result_members = $link->query("SELECT id, surname, other_names FROM users ORDER BY surname ASC");
if ($result_members) {
    while ($row = $result_members->fetch_assoc()) {
        $members[] = $row;
    }
}

$benefit_types = [];
$result_types = $link->query("SELECT id, type_name FROM benefit_types ORDER BY type_name ASC");
if ($result_types) {
    while ($row = $result_types->fetch_assoc()) {
        $benefit_types[] = $row;
    }
}
?>

<?php require_once 'includes/header.php'; ?>
<div class="container-fluid">
    <div class="row">
        <?php require_once 'includes/sidebar.php'; ?>
        <main class="col-md-9 col-lg-10 main-content">
            <h2>Member Benefits Management</h2>

            <?php if ($message): ?>
                <div class="alert alert-<?php echo htmlspecialchars($message_type); ?>" role="alert">
                    <?php echo htmlspecialchars($message); ?>
                </div>
            <?php endif; ?>

            <div class="card mb-4">
                <div class="card-header">Record New Benefit</div>
                <div class="card-body">
                    <form action="member_benefits.php" method="POST">
                        <div class="form-group mb-3">
                            <label for="member_id">Member</label>
                            <select id="member_id" name="member_id" class="form-control" required>
                                <option value="">Select a Member</option>
                                <?php foreach ($members as $member): ?>
                                    <option value="<?php echo htmlspecialchars($member['id']); ?>"><?php echo htmlspecialchars($member['surname'] . ', ' . $member['other_names']); ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="form-group mb-3">
                            <label for="benefit_type_id">Benefit Type</label>
                            <select id="benefit_type_id" name="benefit_type_id" class="form-control" required>
                                <option value="">Select Benefit Type</option>
                                <?php foreach ($benefit_types as $type): ?>
                                    <option value="<?php echo htmlspecialchars($type['id']); ?>"><?php echo htmlspecialchars($type['type_name']); ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="form-group mb-3">
                            <label for="description">Description (Optional)</label>
                            <textarea id="description" name="description" class="form-control" rows="3"></textarea>
                        </div>
                        <div class="form-group mb-3">
                            <label for="date_given">Date Given</label>
                            <input type="date" id="date_given" name="date_given" class="form-control" value="<?php echo date('Y-m-d'); ?>" required>
                        </div>
                        <button type="submit" name="add_benefit" class="btn btn-primary">Record Benefit</button>
                    </form>
                </div>
            </div>

            <div class="card">
                <div class="card-header">Recorded Benefits</div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table table-striped table-hover">
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>Member Name</th>
                                    <th>Benefit Type</th>
                                    <th>Description</th>
                                    <th>Date Given</th>
                                </tr>
                            </thead>
                            <tbody id="benefitsTableBody">
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </main>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', () => {
    fetchBenefits();
});

async function fetchBenefits() {
    try {
        const response = await fetch('api/get_member_benefits.php');
        const result = await response.json();

        const tableBody = document.getElementById('benefitsTableBody');
        tableBody.innerHTML = '';

        if (result.success && result.benefits.length > 0) {
            result.benefits.forEach(benefit => {
                const row = document.createElement('tr');
                row.innerHTML = `
                    <td>${benefit.benefit_id}</td>
                    <td>${benefit.surname}, ${benefit.other_names}</td>
                    <td>${benefit.benefit_type}</td>
                    <td>${benefit.description || 'N/A'}</td>
                    <td>${benefit.date_given}</td>
                `;
                tableBody.appendChild(row);
            });
        } else {
            const row = document.createElement('tr');
            row.innerHTML = `<td colspan="5" class="text-center">${result.message}</td>`;
            tableBody.appendChild(row);
        }
    } catch (error) {
        console.error('Error fetching benefits:', error);
        const tableBody = document.getElementById('benefitsTableBody');
        tableBody.innerHTML = `<tr><td colspan="5" class="text-center">Failed to load benefits. Please try again.</td></tr>`;
    }
}
</script>

<?php require_once 'includes/footer.php'; ?>