<?php
// manage_token_purchases.php
ob_start();
if (session_status() !== PHP_SESSION_ACTIVE) {
    session_start();
}
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

if (function_exists('opcache_reset')) {
    opcache_reset();
}

require_once 'includes/functions.php';
require_once 'includes/db_config.php';

// Session timeout
$timeout = 1800;
if (isset($_SESSION['last_activity']) && (time() - $_SESSION['last_activity']) > $timeout) {
    session_unset();
    session_destroy();
    redirect("login.php?timeout=1");
}
$_SESSION['last_activity'] = time();

// Super Admin Only
if (!is_logged_in() || $_SESSION['user_role'] !== 'super_admin') {
    redirect("login.php");
}

// CSRF Token
if (!isset($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

$message = '';
$message_type = '';
$purchases = [];

// DB Connection
if (mysqli_connect_error()) {
    $message = "Database connection failed.";
    $message_type = 'danger';
} else {
    $check = mysqli_query($link, "SHOW TABLES LIKE 'token_purchases'");
    if (mysqli_num_rows($check) == 0) {
        $message = "Token purchases table missing.";
        $message_type = 'danger';
    } else {

        // === HANDLE APPROVE / REJECT ===
        if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['action'], $_POST['csrf_token']) && hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {
            $purchase_id = (int)($_POST['purchase_id'] ?? 0);
            $action      = sanitize_input($_POST['action']);
            $user_id     = (int)($_POST['user_id'] ?? 0);
            $quantity    = (int)($_POST['quantity'] ?? 0);

            if ($action === 'approve') {
                mysqli_begin_transaction($link);
                $ok = true;

                // 1. Approve purchase
                $stmt = mysqli_prepare($link, "UPDATE token_purchases SET status = 'approved' WHERE id = ?");
                if ($stmt) {
                    mysqli_stmt_bind_param($stmt, "i", $purchase_id);
                    $ok = $ok && mysqli_stmt_execute($stmt);
                    mysqli_stmt_close($stmt);
                } else {
                    $ok = false;
                }

                // 2. Add tokens (NO created_at/updated_at)
                if ($ok) {
                    $stmt = mysqli_prepare($link, "INSERT INTO token_balances (user_id, balance) VALUES (?, ?) ON DUPLICATE KEY UPDATE balance = balance + ?");
                    if ($stmt) {
                        mysqli_stmt_bind_param($stmt, "iii", $user_id, $quantity, $quantity);
                        $ok = $ok && mysqli_stmt_execute($stmt);
                        mysqli_stmt_close($stmt);
                    } else {
                        $ok = false;
                    }
                }

                // 3. Auto-log income
                if ($ok) {
                    $amount = 0;
                    $stmt = mysqli_prepare($link, "SELECT amount FROM token_purchases WHERE id = ?");
                    if ($stmt) {
                        mysqli_stmt_bind_param($stmt, "i", $purchase_id);
                        mysqli_stmt_execute($stmt);
                        $res = mysqli_stmt_get_result($stmt);
                        if ($row = mysqli_fetch_assoc($res)) {
                            $amount = (float)$row['amount'];
                        }
                        mysqli_stmt_close($stmt);
                    }

                    if ($amount > 0) {
                        $payment_type = 'Token Purchase';

                        // INSERT IGNORE into payment_types (NO created_at)
                        $stmt = mysqli_prepare($link, "INSERT IGNORE INTO payment_types (type_name, payment_category) VALUES (?, 'income')");
                        if ($stmt) {
                            mysqli_stmt_bind_param($stmt, "s", $payment_type);
                            mysqli_stmt_execute($stmt);
                            mysqli_stmt_close($stmt);
                        }

                        // INSERT into payments (NO obligation_id → NULL)
                        $description = "Auto-logged token purchase of $quantity tokens";
                        $member_id = 0;
                        $payment_status = 'Paid';
                        $status = 'Paid';
                        $type = 'income';

                        $sql = "INSERT INTO payments 
                                (member_id, payment_type_id, user_id, payment_type, amount, payment_status, payment_date, description, status, type, obligation_id)
                                VALUES (?, (SELECT id FROM payment_types WHERE type_name = ? LIMIT 1), ?, ?, ?, ?, NOW(), ?, ?, ?, NULL)";
                        
                        $stmt = mysqli_prepare($link, $sql);
                        if ($stmt) {
                            mysqli_stmt_bind_param($stmt, "iisdsisss", $member_id, $payment_type, $user_id, $payment_type, $amount, $payment_status, $description, $status, $type);
                            $ok = $ok && mysqli_stmt_execute($stmt);
                            mysqli_stmt_close($stmt);
                        } else {
                            $ok = false;
                        }
                    }
                }

                if ($ok) {
                    mysqli_commit($link);
                    $message = "Approved. Tokens added. Revenue logged.";
                    $message_type = 'success';

                    // Email
                    $stmt = mysqli_prepare($link, "SELECT email FROM users WHERE id = ?");
                    if ($stmt) {
                        mysqli_stmt_bind_param($stmt, "i", $user_id);
                        mysqli_stmt_execute($stmt);
                        $res = mysqli_stmt_get_result($stmt);
                        if ($row = mysqli_fetch_assoc($res)) {
                            send_mail($row['email'], "Token Purchase Approved", "Your $quantity tokens have been added.");
                        }
                        mysqli_stmt_close($stmt);
                    }
                } else {
                    mysqli_rollback($link);
                    $message = "Error approving purchase.";
                    $message_type = 'danger';
                }

            } elseif ($action === 'reject') {
                $stmt = mysqli_prepare($link, "UPDATE token_purchases SET status = 'rejected' WHERE id = ?");
                if ($stmt && mysqli_stmt_bind_param($stmt, "i", $purchase_id) && mysqli_stmt_execute($stmt)) {
                    $message = "Purchase rejected.";
                    $message_type = 'success';
                    mysqli_stmt_close($stmt);

                    $stmt = mysqli_prepare($link, "SELECT email FROM users WHERE id = ?");
                    if ($stmt) {
                        mysqli_stmt_bind_param($stmt, "i", $user_id);
                        mysqli_stmt_execute($stmt);
                        $res = mysqli_stmt_get_result($stmt);
                        if ($row = mysqli_fetch_assoc($res)) {
                            send_mail($row['email'], "Token Purchase Rejected", "Your request was rejected.");
                        }
                        mysqli_stmt_close($stmt);
                    }
                } else {
                    $message = "Error rejecting.";
                    $message_type = 'danger';
                }
            }

            $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
        }

        // === LOAD PENDING ===
        $sql = "SELECT tp.id, tp.user_id, u.email, tp.quantity, tp.amount, tp.created_at, tp.package_name
                FROM token_purchases tp
                JOIN users u ON tp.user_id = u.id
                WHERE tp.status = 'pending'";
        $res = mysqli_query($link, $sql);
        if ($res) {
            while ($row = mysqli_fetch_assoc($res)) {
                $purchases[] = $row;
            }
            mysqli_free_result($res);
        }
    }
}
ob_end_clean();
?>
<?php require_once 'includes/header.php'; ?>
<script>document.title = 'Manage Token Purchases - <?=htmlspecialchars(get_setting('system_name',$link) ?? 'Club Portal')?>';</script>
<style>.main-content{padding:20px;}</style>

<div class="container-fluid">
    <div class="row">
        <?php require_once 'includes/sidebar.php'; ?>
        <main class="col-md-9 col-lg-10 main-content">
            <h2>Manage Token Purchases</h2>
            <?php if ($message): ?>
                <div class="alert alert-<?=$message_type?>"><?=$message?></div>
            <?php endif; ?>

            <div class="card">
                <div class="card-header">Pending Requests</div>
                <div class="card-body">
                    <?php if (empty($purchases)): ?>
                        <p class="text-muted">No pending requests.</p>
                    <?php else: ?>
                        <table class="table table-striped">
                            <thead>
                                <tr><th>Email</th><th>Package</th><th>Qty</th><th>Amount</th><th>Date</th><th>Actions</th></tr>
                            </thead>
                            <tbody>
                                <?php foreach ($purchases as $p): ?>
                                <tr>
                                    <td><?=htmlspecialchars($p['email'])?></td>
                                    <td><?=htmlspecialchars($p['package_name'])?></td>
                                    <td><?=htmlspecialchars($p['quantity'])?></td>
                                    <td><?=format_currency($p['amount'], get_setting('currency_symbol',$link) ?? '₦')?></td>
                                    <td><?=htmlspecialchars($p['created_at'])?></td>
                                    <td>
                                        <form method="post" style="display:inline;">
                                            <input type="hidden" name="action" value="approve">
                                            <input type="hidden" name="purchase_id" value="<?=$p['id']?>">
                                            <input type="hidden" name="user_id" value="<?=$p['user_id']?>">
                                            <input type="hidden" name="quantity" value="<?=$p['quantity']?>">
                                            <input type="hidden" name="csrf_token" value="<?=htmlspecialchars($_SESSION['csrf_token'])?>">
                                            <button class="btn btn-success btn-sm">Approve</button>
                                        </form>
                                        <form method="post" style="display:inline;">
                                            <input type="hidden" name="action" value="reject">
                                            <input type="hidden" name="purchase_id" value="<?=$p['id']?>">
                                            <input type="hidden" name="user_id" value="<?=$p['user_id']?>">
                                            <input type="hidden" name="csrf_token" value="<?=htmlspecialchars($_SESSION['csrf_token'])?>">
                                            <button class="btn btn-danger btn-sm">Reject</button>
                                        </form>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    <?php endif; ?>
                </div>
            </div>
        </main>
    </div>
</div>

<?php require_once 'includes/footer.php'; ?>