<?php
// includes/reports_data.php
// -----------------------------------------------------------------------
// Loads all data required by the financial reports page
// Compatible with corrected financial_queries.php and financial_utils.php
// NOW INCLUDES: Total Excess Donations
// -----------------------------------------------------------------------

// Prevent direct access
if (!defined('ABSPATH') && !defined('IN_REPORTS')) {
    exit('Direct access not allowed.');
}

// Ensure database connection is available
if (!isset($link) || !$link) {
    error_log("Database connection not available in reports_data.php");
    die("ERROR: Database connection not available.");
}

// === INPUT VARIABLES (from GET) ===
$record_type   = $_GET['record_type']   ?? 'all';
$time_period   = $_GET['time_period']   ?? 'date_range';
$start_date    = $_GET['start_date']    ?? '';
$end_date      = $_GET['end_date']      ?? '';
$type_filter   = $_GET['type_filter']   ?? '';
$search        = $_GET['search']        ?? '';

// Sanitize inputs
$valid_record_types = ['all', 'income', 'expenses', 'benefits', 'debts'];
$record_type = in_array($record_type, $valid_record_types, true) ? $record_type : 'all';

$valid_time_periods = ['week', 'month', 'year', 'date_range', 'all'];
$time_period = in_array($time_period, $valid_time_periods, true) ? $time_period : 'date_range';

// Normalize dates
if ($time_period === 'date_range') {
    $start_date = $start_date ? date('Y-m-d', strtotime($start_date)) : date('Y-m-d', strtotime('-30 days'));
    $end_date   = $end_date   ? date('Y-m-d', strtotime($end_date))   : date('Y-m-d');
}

// === INCLUDE DEPENDENCIES ===
require_once __DIR__ . '/financial_utils.php';
require_once __DIR__ . '/functions.php';
require_once __DIR__ . '/financial_queries.php';

// === TYPE FILTER CONDITION ===
$type_filter_condition = '';
$type_filter_param = null;
if ($type_filter !== '') {
    $type_parts = explode('_', $type_filter, 2);
    $type_category = $type_parts[0] ?? '';
    $type_name = $type_parts[1] ?? '';
    if ($type_category && $type_name) {
        if ($type_category === 'income' || $type_category === 'payment') {
            $type_filter_condition = " AND pt.type_name = ?";
        } elseif ($type_category === 'expense') {
            $type_filter_condition = " AND et.type_name = ?";
        } elseif ($type_category === 'benefit') {
            $type_filter_condition = " AND bt.type_name = ?";
        }
        $type_filter_param = $type_name;
    } else {
        error_log("Invalid type_filter format: $type_filter");
        $type_filter = '';
    }
}

// === TIME FILTERS ===
$time_filter = " AND DATE(created_at) BETWEEN ? AND ?";
$time_filter_debts = " AND (po.due_date IS NULL OR DATE(po.due_date) BETWEEN ? AND ?)";

// === PAGINATION ===
$tables = ['income', 'expenses', 'debts', 'benefits'];
$pagination = [];
foreach ($tables as $table) {
    $per_page = isset($_GET["per_page_$table"]) && in_array((int)$_GET["per_page_$table"], [5, 10, 25, 50, 100])
        ? (int)$_GET["per_page_$table"] : 5;
    $page = isset($_GET["page_$table"]) && is_numeric($_GET["page_$table"]) && $_GET["page_$table"] > 0
        ? (int)$_GET["page_$table"] : 1;
    $offset = ($page - 1) * $per_page;

    $pagination[$table] = [
        'per_page' => $per_page,
        'page'     => $page,
        'offset'   => $offset
    ];
}

// === TOTAL COUNTS FOR PAGINATION ===
$total_counts = [];
$total_pages = [];
foreach ($tables as $table) {
    $sql_count = '';
    $count_bind_types = '';
    $count_bind_params = [];
    $search_condition = '';

    if ($table === 'income') {
        $sql_count = "SELECT COUNT(*) as total FROM payments p JOIN users u ON p.user_id = u.id WHERE payment_status = 'Paid'";
        if ($record_type === 'all' || $record_type === 'income') {
            $sql_count .= str_replace('created_at', 'p.created_at', $time_filter);
            $count_bind_types .= 'ss';
            $count_bind_params[] = $start_date;
            $count_bind_params[] = $end_date;
            if ($type_filter && (strpos($type_filter, 'income_') === 0 || strpos($type_filter, 'payment_') === 0)) {
                $sql_count .= $type_filter_condition;
                $count_bind_types .= 's';
                $count_bind_params[] = $type_filter_param;
            }
        } else {
            $sql_count .= ' AND FALSE';
        }
        $search_condition = " AND (u.full_name LIKE ? OR p.payment_type LIKE ? OR CAST(p.id AS CHAR) LIKE ?)";
    } elseif ($table === 'expenses') {
        $sql_count = "SELECT COUNT(*) as total FROM expenses e LEFT JOIN expense_types et ON e.expense_type_id = et.id WHERE 1=1";
        if ($record_type === 'all' || $record_type === 'expenses') {
            $sql_count .= str_replace('created_at', 'e.created_at', $time_filter);
            $count_bind_types .= 'ss';
            $count_bind_params[] = $start_date;
            $count_bind_params[] = $end_date;
            if ($type_filter && strpos($type_filter, 'expense_') === 0) {
                $sql_count .= $type_filter_condition;
                $count_bind_types .= 's';
                $count_bind_params[] = $type_filter_param;
            }
        } else {
            $sql_count .= ' AND FALSE';
        }
        $search_condition = " AND (et.type_name LIKE ? OR e.description LIKE ? OR CAST(e.id AS CHAR) LIKE ?)";
    } elseif ($table === 'benefits') {
        $sql_count = "SELECT COUNT(*) as total FROM member_benefits b JOIN users u ON b.member_id = u.id JOIN benefit_types bt ON b.benefit_type_id = bt.id WHERE 1=1";
        if ($record_type === 'all' || $record_type === 'benefits') {
            $sql_count .= str_replace('created_at', 'b.created_at', $time_filter);
            $count_bind_types .= 'ss';
            $count_bind_params[] = $start_date;
            $count_bind_params[] = $end_date;
            if ($type_filter && strpos($type_filter, 'benefit_') === 0) {
                $sql_count .= $type_filter_condition;
                $count_bind_types .= 's';
                $count_bind_params[] = $type_filter_param;
            }
        } else {
            $sql_count .= ' AND FALSE';
        }
        $search_condition = " AND (u.full_name LIKE ? OR bt.type_name LIKE ? OR CAST(b.id AS CHAR) LIKE ?)";
    } elseif ($table === 'debts') {
        $sql_count = "SELECT po.id, u.id, po.required_amount, po.due_date
                      FROM payment_obligations po 
                      JOIN users u ON po.user_id = u.id 
                      JOIN payment_types pt ON po.payment_type = pt.type_name 
                      LEFT JOIN payments p ON p.obligation_id = po.id AND p.payment_status = 'Paid'
                      WHERE po.deleted_at IS NULL 
                          AND u.role = 'member' 
                          AND (u.deleted_at IS NULL OR u.deleted_at > NOW())
                          AND COALESCE(po.required_amount, 0) > 0";
        if ($record_type === 'all' || $record_type === 'debts') {
            $sql_count .= $time_filter_debts;
            $count_bind_types .= 'ss';
            $count_bind_params[] = $start_date;
            $count_bind_params[] = $end_date;
            if ($type_filter && (strpos($type_filter, 'income_') === 0 || strpos($type_filter, 'payment_') === 0)) {
                $sql_count .= $type_filter_condition;
                $count_bind_types .= 's';
                $count_bind_params[] = $type_filter_param;
            }
        } else {
            $sql_count .= ' AND FALSE';
        }
        $search_condition = " AND (u.full_name LIKE ? OR pt.type_name LIKE ? OR CAST(po.id AS CHAR) LIKE ?)";
        if (!empty($search)) {
            $sql_count .= $search_condition;
            $count_bind_types .= 'sss';
            $search_param = "%$search%";
            $count_bind_params[] = $search_param;
            $count_bind_params[] = $search_param;
            $count_bind_params[] = $search_param;
        }
        $sql_count .= " GROUP BY po.id, u.id, po.required_amount, po.due_date 
                        HAVING (COALESCE(po.required_amount, 0) - COALESCE(SUM(p.amount), 0)) > 0";
    }

    if ($table !== 'debts' && !empty($search)) {
        $sql_count .= $search_condition;
        $count_bind_types .= 'sss';
        $search_param = "%$search%";
        $count_bind_params[] = $search_param;
        $count_bind_params[] = $search_param;
        $count_bind_params[] = $search_param;
    }

    $stmt = mysqli_prepare($link, $sql_count);
    if ($stmt === false) {
        error_log("Failed to prepare statement for $table count: " . mysqli_error($link));
        $total_counts[$table] = 0;
        continue;
    }
    if ($count_bind_types && !empty($count_bind_params)) {
        if (!mysqli_stmt_bind_param($stmt, $count_bind_types, ...$count_bind_params)) {
            error_log("Failed to bind parameters for $table count: " . mysqli_stmt_error($stmt));
            $total_counts[$table] = 0;
            mysqli_stmt_close($stmt);
            continue;
        }
    }
    if (!mysqli_stmt_execute($stmt)) {
        error_log("Failed to execute $table count query: " . mysqli_stmt_error($stmt));
        $total_counts[$table] = 0;
        mysqli_stmt_close($stmt);
        continue;
    }
    $result = mysqli_stmt_get_result($stmt);
    $total_counts[$table] = ($table === 'debts') ? mysqli_num_rows($result) : (mysqli_fetch_assoc($result)['total'] ?? 0);
    mysqli_stmt_close($stmt);
    $total_pages[$table] = ceil($total_counts[$table] / $pagination[$table]['per_page']);
}

// === TOTAL DEBT FROM REPORT ===
$total_debt_from_report = 0;
if ($record_type === 'all' || $record_type === 'debts') {
    $sql_total_debt = "SELECT COALESCE(SUM(debt), 0) AS total_debt
                       FROM (
                           SELECT (COALESCE(po.required_amount, 0) - COALESCE(SUM(p.amount), 0)) AS debt
                           FROM payment_obligations po 
                           JOIN users u ON po.user_id = u.id 
                           JOIN payment_types pt ON po.payment_type = pt.type_name 
                           LEFT JOIN payments p ON p.obligation_id = po.id AND p.payment_status = 'Paid'
                           WHERE po.deleted_at IS NULL 
                               AND u.role = 'member' 
                               AND (u.deleted_at IS NULL OR u.deleted_at > NOW())
                               AND COALESCE(po.required_amount, 0) > 0";
    $total_debt_bind_types = '';
    $total_debt_bind_params = [];
    $sql_total_debt .= $time_filter_debts;
    $total_debt_bind_types .= 'ss';
    $total_debt_bind_params[] = $start_date;
    $total_debt_bind_params[] = $end_date;
    if ($type_filter && (strpos($type_filter, 'income_') === 0 || strpos($type_filter, 'payment_') === 0)) {
        $sql_total_debt .= $type_filter_condition;
        $total_debt_bind_types .= 's';
        $total_debt_bind_params[] = $type_filter_param;
    }
    if (!empty($search)) {
        $sql_total_debt .= " AND (u.full_name LIKE ? OR pt.type_name LIKE ? OR CAST(po.id AS CHAR) LIKE ?)";
        $total_debt_bind_types .= 'sss';
        $search_param = "%$search%";
        $total_debt_bind_params[] = $search_param;
        $total_debt_bind_params[] = $search_param;
        $total_debt_bind_params[] = $search_param;
    }
    $sql_total_debt .= " GROUP BY po.id, u.id, po.required_amount, po.due_date, po.payment_cycle
                         HAVING debt > 0
                       ) AS subquery";

    $stmt = mysqli_prepare($link, $sql_total_debt);
    if ($stmt && $total_debt_bind_types && !empty($total_debt_bind_params)) {
        mysqli_stmt_bind_param($stmt, $total_debt_bind_types, ...$total_debt_bind_params);
    }
    if ($stmt && mysqli_stmt_execute($stmt)) {
        $result = mysqli_stmt_get_result($stmt);
        $row = mysqli_fetch_assoc($result);
        $total_debt_from_report = (float)($row['total_debt'] ?? 0);
        mysqli_free_result($result);
    } else {
        error_log("Total debt query failed: " . ($stmt ? mysqli_stmt_error($stmt) : mysqli_error($link)));
    }
    if ($stmt) mysqli_stmt_close($stmt);
}

// === TYPE SUMMARIES & METRICS ===
$payment_types_summary = [];
$expense_types_summary = [];
$benefit_types_summary = [];
$financial_metrics = [];

if ($record_type === 'all' || $record_type === 'income') {
    $payment_types_summary = get_payment_types_summary(
        $link, $record_type, $time_period, $start_date, $end_date, $type_filter, 'due_date'
    );
}
if ($record_type === 'all' || $record_type === 'expenses') {
    $expense_types_summary = get_expense_types_summary(
        $link, $record_type, $time_period, $start_date, $end_date, $type_filter
    );
}
if ($record_type === 'all' || $record_type === 'benefits') {
    $benefit_types_summary = get_benefit_types_summary(
        $link, $record_type, $time_period, $start_date, $end_date, $type_filter
    );
}

// === FALLBACK FOR financial_metrics ===
if (function_exists('get_financial_metrics')) {
    $financial_metrics = get_financial_metrics(
        $link, $record_type, $time_period, $start_date, $end_date, $type_filter, 'due_date'
    );
} else {
    $financial_metrics = [
        'total_required' => 0.0,
        'total_income'   => 0.0,
        'total_debt'     => $total_debt_from_report,
        'excess_donations' => 0.0,
        'available_balance' => 0.0
    ];
}

// === ENSURE ARRAY STRUCTURE ===
$payment_types_summary = is_array($payment_types_summary) ? $payment_types_summary : [];
$expense_types_summary = is_array($expense_types_summary) ? $expense_types_summary : [];
$benefit_types_summary = is_array($benefit_types_summary) ? $benefit_types_summary : [];
$financial_metrics = is_array($financial_metrics) ? $financial_metrics : [];

// === TOTALS FROM functions.php + EXCESS DONATIONS ===
$totals = function_exists('get_financial_totals')
    ? get_financial_totals($link, $record_type, $start_date, $end_date)
    : ['total_payments' => 0, 'total_expenses' => 0, 'total_benefits' => 0];

$total_income   = $totals['total_payments'] ?? 0.0;
$total_expenses = $totals['total_expenses'] ?? 0.0;
$total_benefits = $totals['total_benefits'] ?? 0.0;
$total_debts    = $total_debt_from_report;

// Calculate Excess Donations
$total_excess_donations = 0.0;
if (!empty($payment_types_summary)) {
    foreach ($payment_types_summary as $row) {
        $total_excess_donations += (float)($row['excess_amount'] ?? 0.0);
    }
}

// Override with summary data
$total_required = array_sum(array_column($payment_types_summary, 'required_amount'));
$total_income   = array_sum(array_column($payment_types_summary, 'total_paid')) ?: $total_income;

$financial_metrics['total_required'] = $total_required;
$financial_metrics['total_income']   = $total_income;
$financial_metrics['total_debt']     = $total_debts;
$financial_metrics['excess_donations'] = $total_excess_donations; // NEW
$financial_metrics['available_balance'] = $total_income - $total_expenses - $total_benefits;

// === FETCH RECORDS ===
$income = $expenses = $benefits = $debts = [];

if ($record_type === 'all' || $record_type === 'income') {
    $income = get_payments($link, $record_type, $time_period, $start_date, $end_date, $type_filter, $search, $pagination['income']);
}

if ($record_type === 'all' || $record_type === 'expenses') {
    $expenses = get_expenses($link, $record_type, $time_period, $start_date, $end_date, $type_filter, $search, $pagination['expenses']);
}

if ($record_type === 'all' || $record_type === 'benefits') {
    $benefits = get_benefits($link, $record_type, $time_period, $start_date, $end_date, $type_filter, $search, $pagination['benefits']);
}

if ($record_type === 'all' || $record_type === 'debts') {
    $debts = get_debts($link, $record_type, $time_period, $start_date, $end_date, $type_filter, $search, 'due_date', $pagination['debts']);
}
?>