<?php
// cron_generate_obligations.php
// Generates payment obligations for members based on payment types, recurrence intervals, and due dates
// Intended for cron job execution

// Initialize logging
$log_file = '/var/log/cron_obligations.log';
function cron_log($message) {
    global $log_file;
    $timestamp = date('Y-m-d H:i:s');
    file_put_contents($log_file, "[$timestamp] $message\n", FILE_APPEND);
}

cron_log("Starting cron_generate_obligations.php");

// Include dependencies
require_once 'includes/db_config.php';
require_once 'includes/functions.php';

// Get current timestamp and year for comparison
$current_date = date('Y-m-d H:i:s');
$current_year = date('Y');

cron_log("Current date: $current_date, Current year: $current_year");

// Check if required columns exist in payment_obligations
$columns_query = "SHOW COLUMNS FROM payment_obligations";
$columns_result = mysqli_query($link, $columns_query);
$columns = [];
if ($columns_result) {
    while ($row = mysqli_fetch_assoc($columns_result)) {
        $columns[] = $row['Field'];
    }
    cron_log("payment_obligations columns: " . implode(', ', $columns));
} else {
    cron_log("Failed to check payment_obligations schema: " . mysqli_error($link));
    exit(1);
}
$has_due_date = in_array('due_date', $columns);
$has_required_amount = in_array('required_amount', $columns);
$has_deleted_at = in_array('deleted_at', $columns);

// Fetch active members
$members = [];
$sql_members = "SELECT id FROM users WHERE role = 'member' AND (deleted_at IS NULL OR deleted_at > NOW())";
$result_members = mysqli_query($link, $sql_members);
if ($result_members) {
    while ($row = mysqli_fetch_assoc($result_members)) {
        $members[] = $row['id'];
    }
    cron_log("Found " . count($members) . " active members");
} else {
    cron_log("Failed to fetch members: " . mysqli_error($link));
    exit(1);
}

// Fetch active payment types
$payment_types = [];
$sql_payment_types = "SELECT id, type_name, required_amount, payment_category, recurrence_interval, event_date 
                     FROM payment_types 
                     WHERE deleted_at IS NULL";
$result_payment_types = mysqli_query($link, $sql_payment_types);
if ($result_payment_types) {
    while ($row = mysqli_fetch_assoc($result_payment_types)) {
        $payment_types[] = $row;
    }
    cron_log("Found " . count($payment_types) . " active payment types");
} else {
    cron_log("Failed to fetch payment types: " . mysqli_error($link));
    exit(1);
}

// Process each payment type
$obligations_created = 0;
foreach ($payment_types as $payment_type) {
    $type_name = $payment_type['type_name'];
    $required_amount = $payment_type['required_amount'] ?? 0.00;
    $payment_category = $payment_type['payment_category'];
    $recurrence_interval = $payment_type['recurrence_interval'];
    $event_date = $payment_type['event_date'];

    cron_log("Processing payment type: $type_name, category: $payment_category, required_amount: $required_amount");

    foreach ($Panics: 
        $create_new_obligation = false;
        $new_due_date = null;

        if ($payment_category === 'recurring' && $recurrence_interval !== 'none' && $has_due_date) {
            if (!$existing_obligation) {
                $create_new_obligation = true;
                $new_due_date = calculate_due_date($recurrence_interval, $current_date);
                cron_log("No existing obligation for user_id $user_id, payment_type $type_name, year $current_year. Will create with due_date $new_due_date");
            } elseif ($existing_obligation['due_date'] <= $current_date) {
                $create_new_obligation = true;
                $new_due_date = calculate_due_date($recurrence_interval, $existing_obligation['due_date']);
                cron_log("Existing obligation for user_id $user_id, payment_type $type_name expired (due_date {$existing_obligation['due_date']}). Will create new with due_date $new_due_date");
            }
        } elseif ($payment_category === 'event' && $event_date && $has_due_date) {
            if (!$existing_obligation && $event_date >= $current_date) {
                $create_new_obligation = true;
                $new_due_date = $event_date;
                cron_log("No existing obligation for user_id $user_id, payment_type $type_name, event_date $event_date. Will create with due_date $new_due_date");
            }
        } elseif ($payment_category === 'freewill') {
            if (!$existing_obligation) {
                $create_new_obligation = true;
                $new_due_date = null;
                cron_log("No existing obligation for user_id $user_id, payment_type $type_name, year $current_year. Will create freewill obligation");
            }
        }

        if ($create_new_obligation) {
            $insert_fields = 'user_id, payment_type, payment_cycle, created_at';
            $insert_values = '?, ?, ?, NOW()';
            $bind_types = 'isi';
            $bind_params = [&$user_id, &$type_name, &$current_year];
            if ($has_required_amount) {
                $insert_fields .= ', required_amount';
                $insert_values .= ', ?';
                $bind_types .= 'd';
                $bind_params[] = &$required_amount;
            }
            if ($has_due_date) {
                $insert_fields .= ', due_date';
                $insert_values .= ', ?';
                $bind_types .= 's';
                $bind_params[] = &$new_due_date;
            }
            $sql_insert = "INSERT INTO payment_obligations ($insert_fields) VALUES ($insert_values)";
            if ($stmt_insert = mysqli_prepare($link, $sql_insert)) {
                mysqli_stmt_bind_param($stmt_insert, $bind_types, ...$bind_params);
                if (mysqli_stmt_execute($stmt_insert)) {
                    $obligations_created++;
                    cron_log("Created obligation for user_id $user_id, payment_type $type_name, cycle $current_year, due_date " . ($new_due_date ?? 'NULL'));
                } else {
                    cron_log("Failed to create obligation for user_id $user_id, payment_type $type_name, cycle $current_year: " . mysqli_stmt_error($stmt_insert));
                }
                mysqli_stmt_close($stmt_insert);
            } else {
                cron_log("Failed to prepare insert query for user_id $user_id, payment_type $type_name: " . mysqli_error($link));
            }
        }
    }
}

cron_log("Completed cron_generate_obligations.php. Created $obligations_created new obligations");

// Helper function to calculate next due date based on recurrence interval
function calculate_due_date($interval, $base_date) {
    $base = new DateTime($base_date);
    switch ($interval) {
        case 'weekly':
            $base->modify('+1 week');
            break;
        case 'monthly':
            $base->modify('+1 month');
            break;
        case 'yearly':
            $base->modify('+1 year');
            break;
        default:
            $base->modify('+1 month'); // Default to monthly
    }
    return $base->format('Y-m-d H:i:s');
}

// Close database connection
mysqli_close($link);
?>