<?php
require_once 'includes/functions.php';
require_once 'includes/db_config.php';

// Check if user is logged in and is an admin
if (!is_logged_in() || !is_admin()) {
    redirect("login.php");
}

// Auto-create audit_logs table if it doesn't exist
$create_table_sql = "
CREATE TABLE IF NOT EXISTS audit_logs (
    id INT AUTO_INCREMENT PRIMARY KEY,
    action VARCHAR(50) NOT NULL,
    table_name VARCHAR(100),
    record_id INT,
    user_id INT NOT NULL,
    timestamp DATETIME DEFAULT CURRENT_TIMESTAMP,
    details TEXT,
    ip_address VARCHAR(45),
    INDEX idx_user (user_id),
    INDEX idx_timestamp (timestamp),
    INDEX idx_action (action)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
";

mysqli_query($link, $create_table_sql); // Ignore result; table may already exist

// Fetch logs safely
$logs = [];
$error_msg = null;
$sql = "SELECT * FROM audit_logs ORDER BY timestamp DESC LIMIT 100";
$result = mysqli_query($link, $sql);

if ($result) {
    while ($row = mysqli_fetch_assoc($result)) {
        $logs[] = $row;
    }
    mysqli_free_result($result);
} else {
    if (mysqli_errno($link) == 1146) {
        $error_msg = "Audit logs table was missing and has been created. Please refresh the page.";
    } else {
        $error_msg = "Database error: " . mysqli_error($link);
        error_log("Audit log query failed: " . $error_msg);
    }
}
?>
<?php require_once 'includes/header.php'; ?>
<div class="container-fluid">
    <div class="row">
        <?php require_once 'includes/sidebar.php'; ?>
        <main class="col-md-9 col-lg-10 main-content">
            <h2>Audit Log</h2>
            <hr>
            <div class="card mb-4">
                <div class="card-header">Recent Actions</div>
                <div class="card-body table-responsive">
                    <table class="table table-striped">
                        <thead>
                            <tr>
                                <th>ID</th>
                                <th>Action</th>
                                <th>Table</th>
                                <th>Record ID</th>
                                <th>User ID</th>
                                <th>Timestamp</th>
                                <th>Details</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if ($error_msg): ?>
                                <tr>
                                    <td colspan="7" class="text-center text-warning">
                                        <strong>Note:</strong> <?php echo htmlspecialchars($error_msg); ?>
                                    </td>
                                </tr>
                            <?php elseif (!empty($logs)): ?>
                                <?php foreach ($logs as $log): ?>
                                    <tr>
                                        <td><?php echo htmlspecialchars($log['id'] ?? ''); ?></td>
                                        <td><?php echo htmlspecialchars($log['action'] ?? ''); ?></td>
                                        <td><?php echo htmlspecialchars($log['table_name'] ?? ''); ?></td>
                                        <td><?php echo htmlspecialchars($log['record_id'] ?? ''); ?></td>
                                        <td><?php echo htmlspecialchars($log['user_id'] ?? ''); ?></td>
                                        <td><?php echo htmlspecialchars($log['timestamp'] ?? ''); ?></td>
                                        <td><?php echo htmlspecialchars($log['details'] ?? ''); ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php else: ?>
                                <tr><td colspan="7" class="text-center">No audit logs found.</td></tr>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </main>
    </div>
</div>
<?php require_once 'includes/footer.php'; ?>