<?php
require_once 'includes/functions.php';
require_once 'includes/db_config.php';

// Initialize session and check access
initialize_session();
if (!is_logged_in() || !user_has_permission($link, 'view_settings')) {
    error_log("Access denied to role_permissions.php. Role: " . ($_SESSION['user_role'] ?? 'none') . ", Permissions: " . json_encode(get_role_permissions($link, $_SESSION['user_role'] ?? '')));
    redirect("admin_dashboard.php");
}

// Generate CSRF token
if (!isset($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

$success_message = '';
$error_message = '';
$system_name = get_setting('system_name', $link) ?? 'Club Portal';

// Handle permission updates
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['csrf_token']) && hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {
    if (isset($_POST['update_permissions'])) {
        $role = sanitize_input($_POST['role']);
        $selected_permissions = $_POST['permissions'] ?? [];
        
        // Validate role
        if (!in_array($role, ['admin', 'accountant', 'super_admin'])) {
            $error_message = "Invalid role selected.";
            error_log("Invalid role selected: $role");
        } else {
            // Start transaction
            mysqli_begin_transaction($link);
            try {
                // Clear existing permissions for the role
                $sql = "DELETE FROM role_permissions WHERE role_name = ?";
                if ($stmt = mysqli_prepare($link, $sql)) {
                    mysqli_stmt_bind_param($stmt, "s", $role);
                    mysqli_stmt_execute($stmt);
                    $deleted_rows = mysqli_affected_rows($link);
                    error_log("Deleted $deleted_rows permissions for role: $role");
                    mysqli_stmt_close($stmt);
                } else {
                    throw new Exception("Failed to prepare delete statement: " . mysqli_error($link));
                }

                // Insert new permissions
                foreach ($selected_permissions as $permission) {
                    $sql = "INSERT INTO role_permissions (role_name, permission_key) VALUES (?, ?)";
                    if ($stmt = mysqli_prepare($link, $sql)) {
                        mysqli_stmt_bind_param($stmt, "ss", $role, $permission);
                        mysqli_stmt_execute($stmt);
                        mysqli_stmt_close($stmt);
                    } else {
                        throw new Exception("Failed to prepare insert statement: " . mysqli_error($link));
                    }
                }
                mysqli_commit($link);
                $success_message = "Permissions updated successfully for role: " . htmlspecialchars(ucfirst(str_replace('_', ' ', $role)));
                error_log("Permissions updated for role: $role, Permissions: " . json_encode($selected_permissions));
            } catch (Exception $e) {
                mysqli_rollback($link);
                $error_message = "Failed to update permissions: " . $e->getMessage();
                error_log("Error updating permissions for role: $role, Error: " . $e->getMessage());
            }
            // Regenerate CSRF token
            $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
        }
    }
} elseif ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $error_message = "Invalid CSRF token.";
    error_log("CSRF token validation failed in role_permissions.php");
}

// Fetch all permissions from permissions table
$permissions = [];
$sql = "SELECT permission_key, description FROM permissions ORDER BY description";
if ($result = mysqli_query($link, $sql)) {
    while ($row = mysqli_fetch_assoc($result)) {
        $permissions[] = $row;
    }
    mysqli_free_result($result);
} else {
    error_log("Failed to fetch permissions: " . mysqli_error($link));
}

// Fetch current permissions for the selected role
$current_role = isset($_POST['role']) ? sanitize_input($_POST['role']) : 'admin';
$current_permissions = get_role_permissions($link, $current_role);
?>

<?php require_once 'includes/header.php'; ?>
<script>
document.title = 'Role & Permission Settings - <?php echo htmlspecialchars($system_name); ?>';
</script>

<div class="container-fluid">
    <div class="row">
        <?php require_once 'includes/sidebar.php'; ?>
        <main class="col-md-9 col-lg-10 main-content">
            <h2>Role & Permission Settings</h2>

            <?php if ($success_message): ?>
                <div class="alert alert-success"><?php echo htmlspecialchars($success_message); ?></div>
            <?php endif; ?>
            <?php if ($error_message): ?>
                <div class="alert alert-danger"><?php echo htmlspecialchars($error_message); ?></div>
            <?php endif; ?>

            <div class="card mb-4">
                <div class="card-header">
                    Manage Role Permissions
                    <a href="add_admin.php" class="btn btn-success btn-sm float-end"><i class="fas fa-plus-circle me-1"></i> Add Admin User</a>
                </div>
                <div class="card-body">
                    <form method="post" action="role_permissions.php">
                        <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($_SESSION['csrf_token']); ?>">
                        <div class="form-group mb-3">
                            <label for="role" class="form-label">Select Role</label>
                            <select name="role" id="role" class="form-control" onchange="this.form.submit()">
                                <option value="admin" <?php echo $current_role === 'admin' ? 'selected' : ''; ?>>Admin</option>
                                <option value="accountant" <?php echo $current_role === 'accountant' ? 'selected' : ''; ?>>Accountant</option>
                                <option value="super_admin" <?php echo $current_role === 'super_admin' ? 'selected' : ''; ?>>Super Admin</option>
                            </select>
                        </div>
                        <hr>
                        <h4>Permissions for '<?php echo htmlspecialchars(ucfirst(str_replace('_', ' ', $current_role))); ?>'</h4>
                        <div class="form-group mb-3">
                            <div class="row">
                                <?php foreach ($permissions as $perm): ?>
                                    <div class="col-md-4">
                                        <div class="form-check">
                                            <input type="checkbox" name="permissions[]" value="<?php echo htmlspecialchars($perm['permission_key']); ?>"
                                                class="form-check-input" id="perm-<?php echo htmlspecialchars($perm['permission_key']); ?>"
                                                <?php echo in_array($perm['permission_key'], $current_permissions) ? 'checked' : ''; ?>>
                                            <label class="form-check-label" for="perm-<?php echo htmlspecialchars($perm['permission_key']); ?>">
                                                <?php echo htmlspecialchars($perm['description']); ?>
                                            </label>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                        </div>
                        <button type="submit" name="update_permissions" class="btn btn-primary">Update Permissions</button>
                    </form>
                </div>
            </div>
        </main>
    </div>
</div>
<?php require_once 'includes/footer.php'; ?>