<?php
require_once 'includes/functions.php';
require_once 'includes/db_config.php';

$message = '';
$token = '';
$email = '';
$valid_token = false;

// Check if a token is provided in the URL
if (isset($_GET['token'])) {
    $token = sanitize_input($_GET['token']);

    // Find the token in the database and check if it has expired
    $sql = "SELECT email, expires_at FROM password_resets WHERE token = ? AND expires_at >= NOW()";
    if ($stmt = mysqli_prepare($link, $sql)) {
        mysqli_stmt_bind_param($stmt, "s", $token);
        mysqli_stmt_execute($stmt);
        mysqli_stmt_bind_result($stmt, $email, $expires_at);
        if (mysqli_stmt_fetch($stmt)) {
            $valid_token = true;
        } else {
            $message = "Invalid or expired password reset token.";
        }
        mysqli_stmt_close($stmt);
    }
} else {
    $message = "No password reset token provided.";
}

// Handle new password submission
if ($_SERVER["REQUEST_METHOD"] == "POST" && $valid_token) {
    $new_password = $_POST['new_password'];
    $confirm_password = $_POST['confirm_password'];
    $form_token = sanitize_input($_POST['token']);

    if ($form_token !== $token) {
        $message = "Invalid form submission. Please use the original link.";
        $valid_token = false;
    } elseif ($new_password !== $confirm_password) {
        $message = "Passwords do not match.";
    } elseif (strlen($new_password) < 6) {
        $message = "Password must be at least 6 characters long.";
    } else {
        // Hash the new password and update the user's record
        $hashed_password = password_hash($new_password, PASSWORD_DEFAULT);
        $sql = "UPDATE users SET password = ? WHERE email = ?";
        $stmt = mysqli_prepare($link, $sql);
        mysqli_stmt_bind_param($stmt, "ss", $hashed_password, $email);
        if (mysqli_stmt_execute($stmt)) {
            // Delete the used token
            $sql_delete = "DELETE FROM password_resets WHERE email = ?";
            $stmt_delete = mysqli_prepare($link, $sql_delete);
            mysqli_stmt_bind_param($stmt_delete, "s", $email);
            mysqli_stmt_execute($stmt_delete);
            mysqli_stmt_close($stmt_delete);

            $message = "Your password has been reset successfully! You can now <a href='login.php'>log in</a>.";
            $valid_token = false; // Prevent the form from showing again
        } else {
            $message = "Error resetting password.";
        }
        mysqli_stmt_close($stmt);
    }
}
?>

<?php require_once 'includes/header.php'; ?>
    <h2>Reset Password</h2>
    <?php if ($message): ?>
        <div class="message <?php echo strpos($message, 'Error') !== false || strpos($message, 'Invalid') !== false ? 'error-message' : 'success-message'; ?>">
            <?php echo $message; ?>
        </div>
    <?php endif; ?>
    <?php if ($valid_token): ?>
        <div class="form-container">
            <form action="reset_password.php?token=<?php echo htmlspecialchars($token); ?>" method="post">
                <input type="hidden" name="token" value="<?php echo htmlspecialchars($token); ?>">
                <div class="form-group">
                    <label for="new_password">New Password</label>
                    <input type="password" name="new_password" id="new_password" class="form-control" required>
                </div>
                <div class="form-group">
                    <label for="confirm_password">Confirm New Password</label>
                    <input type="password" name="confirm_password" id="confirm_password" class="form-control" required>
                </div>
                <button type="submit" class="btn btn-primary">Reset Password</button>
            </form>
        </div>
    <?php endif; ?>
<?php require_once 'includes/footer.php'; ?>