<?php
ob_start(); // Start output buffering to prevent header issues
require_once 'includes/functions.php';
require_once 'includes/db_config.php';

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

$error_message = '';
$success_message = '';
$form_data = null;

// Generate math CAPTCHA only if not already set
if (!isset($_SESSION['captcha_answer']) || !isset($_SESSION['captcha_question'])) {
    $num1 = rand(1, 10);
    $num2 = rand(1, 10);
    $_SESSION['captcha_answer'] = $num1 + $num2;
    $_SESSION['captcha_question'] = "What is $num1 + $num2?";
}
$captcha_question = $_SESSION['captcha_question'];

// Fetch the most recent form from the database
$sql = "SELECT id, form_data FROM forms ORDER BY id DESC LIMIT 1";
$result = mysqli_query($link, $sql);
if ($result && mysqli_num_rows($result) > 0) {
    $row = mysqli_fetch_assoc($result);
    $registration_form_id = $row['id'];
    $form_data = json_decode($row['form_data'], true);
    // Log form_data for debugging (remove in production)
    error_log('form_data: ' . print_r($form_data, true));
} else {
    $error_message = "No registration form available. Please contact the administrator.";
}

// Handle form submission
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['form_submission'])) {
    // Validate CAPTCHA
    $captcha_input = sanitize_input($_POST['captcha_input']);
    // Convert both to integer for comparison
    $captcha_input = (int) $captcha_input;
    $captcha_answer = (int) $_SESSION['captcha_answer'];
    if (!isset($_SESSION['captcha_answer']) || $captcha_input !== $captcha_answer) {
        $error_message = "Incorrect CAPTCHA answer.";
    } else {
        // Unset CAPTCHA session variables
        unset($_SESSION['captcha_answer']);
        unset($_SESSION['captcha_question']);

        $user_data = [];
        $email = null;
        $full_name = '';
        $password = '';

        // Process form fields
        if ($form_data && isset($form_data['fields']) && is_array($form_data['fields'])) {
            foreach ($form_data['fields'] as $field) {
                // Ensure field properties are strings
                $field_type = isset($field['type']) && is_string($field['type']) ? $field['type'] : 'text';
                $field_label = isset($field['label']) && is_string($field['label']) ? $field['label'] : (isset($field['name']) && is_string($field['name']) ? $field['name'] : 'Unnamed Field');
                $field_name = isset($field['name']) && is_string($field['name']) ? $field['name'] : str_replace(' ', '_', strtolower($field_label));
                $is_required = isset($field['required']) && $field['required'];
                $input_value = isset($_POST[$field_name]) ? $_POST[$field_name] : '';

                if ($is_required && empty($input_value)) {
                    $error_message = "The field '$field_label' is required.";
                    break;
                }

                // Specific validations
                if ($field_type === 'email') {
                    if (!filter_var($input_value, FILTER_VALIDATE_EMAIL)) {
                        $error_message = "Invalid email format for '$field_label'.";
                        break;
                    }
                    $email = $input_value;
                } elseif ($field_type === 'password') {
                    if (strlen($input_value) < 6) {
                        $error_message = "Password must be at least 6 characters long.";
                        break;
                    }
                    $password = $input_value;
                } elseif ($field_type === 'text' && $field_label === 'Full Name') {
                    if (empty($full_name) && $is_required) {
                        $full_name = is_array($input_value) ? reset($input_value) : $input_value;
                    }
                }

                $user_data[$field_name] = is_array($input_value) ? implode(',', $input_value) : $input_value;
            }
        }

        // Additional validation for required fields
        if (empty($error_message)) {
            if (empty($full_name) || empty($email) || empty($password)) {
                $error_message = "Full Name, Email, and Password are required.";
            }
        }

        // Proceed with registration if no errors
        if (empty($error_message)) {
            // Check if email already exists
            $sql = "SELECT id FROM users WHERE email = ?";
            $stmt = mysqli_prepare($link, $sql);
            if ($stmt === false) {
                $error_message = "Error preparing email check: " . mysqli_error($link);
                error_log("Email check prepare failed: " . mysqli_error($link));
            } else {
                mysqli_stmt_bind_param($stmt, "s", $email);
                if (!mysqli_stmt_execute($stmt)) {
                    $error_message = "Error executing email check: " . mysqli_error($link);
                    error_log("Email check execute failed: " . mysqli_error($link));
                    mysqli_stmt_close($stmt);
                } else {
                    mysqli_stmt_store_result($stmt);
                    if (mysqli_stmt_num_rows($stmt) > 0) {
                        $error_message = "This email is already registered.";
                    }
                    mysqli_stmt_close($stmt); // Close statement only after successful execution
                }
            }

            // Proceed only if no errors and email is unique
            if (empty($error_message)) {
                $hashed_password = password_hash($password, PASSWORD_DEFAULT);

                // Generate unique member ID
                $initials = '';
                $words = explode(' ', $full_name);
                foreach ($words as $word) {
                    if (!empty($word)) {
                        $initials .= strtoupper($word[0]);
                    }
                }
                $unique_member_id = '';
                $attempts = 0;
                $max_attempts = 10; // Prevent infinite loop
                do {
                    $serialized_digits = str_pad(rand(0, 999), 3, '0', STR_PAD_LEFT);
                    $unique_member_id = $initials . $serialized_digits;
                    $check_sql = "SELECT id FROM users WHERE member_id = ?";
                    $check_stmt = mysqli_prepare($link, $check_sql);
                    if ($check_stmt === false) {
                        $error_message = "Database error: Failed to prepare member ID check.";
                        error_log("Member ID check prepare failed: " . mysqli_error($link));
                        break;
                    }
                    mysqli_stmt_bind_param($check_stmt, "s", $unique_member_id);
                    if (!mysqli_stmt_execute($check_stmt)) {
                        $error_message = "Error checking member ID: " . mysqli_error($link);
                        error_log("Member ID check execute failed: " . mysqli_error($link));
                        mysqli_stmt_close($check_stmt);
                        break;
                    }
                    mysqli_stmt_store_result($check_stmt);
                    $exists = mysqli_stmt_num_rows($check_stmt) > 0;
                    mysqli_stmt_close($check_stmt); // Close statement in loop
                    $attempts++;
                } while ($exists && $attempts < $max_attempts);

                if ($attempts >= $max_attempts) {
                    $error_message = "Unable to generate a unique member ID. Please try again.";
                } elseif (empty($error_message)) {
                    // Insert new user
                    $sql = "INSERT INTO users (member_id, full_name, email, password, role, membership_status, created_at) VALUES (?, ?, ?, ?, 'member', 'pending', NOW())";
                    $stmt = mysqli_prepare($link, $sql);
                    if ($stmt === false) {
                        $error_message = "Error preparing user insert: " . mysqli_error($link);
                        error_log("User insert prepare failed: " . mysqli_error($link));
                    } else {
                        mysqli_stmt_bind_param($stmt, "ssss", $unique_member_id, $full_name, $email, $hashed_password);
                        if (mysqli_stmt_execute($stmt)) {
                            $user_id = mysqli_insert_id($link);
                            $success_message = "Registration successful! Your account is pending approval. Your Member ID is: $unique_member_id";

                            // Save custom form data
                            $custom_data = json_encode($user_data);
                            $data_sql = "INSERT INTO form_data (form_id, user_id, data, created_at) VALUES (?, ?, ?, NOW())";
                            $data_stmt = mysqli_prepare($link, $data_sql);
                            if ($data_stmt === false) {
                                $error_message = "Error preparing form data insert: " . mysqli_error($link);
                                error_log("Form data insert prepare failed: " . mysqli_error($link));
                            } else {
                                mysqli_stmt_bind_param($data_stmt, "iis", $registration_form_id, $user_id, $custom_data);
                                if (!mysqli_stmt_execute($data_stmt)) {
                                    $error_message = "Error saving form data: " . mysqli_error($link);
                                    error_log("Form data insert execute failed: " . mysqli_error($link));
                                }
                                mysqli_stmt_close($data_stmt);
                            }
                        } else {
                            $error_message = "Error inserting user: " . mysqli_error($link);
                            error_log("User insert execute failed: " . mysqli_error($link));
                        }
                        mysqli_stmt_close($stmt);
                    }
                }
            }
        }
    }
    // Generate new CAPTCHA for next attempt
    if (empty($success_message)) {
        $num1 = rand(1, 10);
        $num2 = rand(1, 10);
        $_SESSION['captcha_answer'] = $num1 + $num2;
        $_SESSION['captcha_question'] = "What is $num1 + $num2?";
        $captcha_question = $_SESSION['captcha_question'];
    }
}
?>

<?php require_once 'includes/header.php'; ?>
<div class="container-fluid">
    <div class="row justify-content-center">
        <div class="col-md-8 col-lg-6">
            <div class="card">
                <div class="card-header text-center">
                    <h2>Join Our Club</h2>
                </div>
                <div class="card-body">
                    <?php if (!empty($error_message)): ?>
                        <div class="message error-message"><?php echo htmlspecialchars($error_message); ?></div>
                    <?php endif; ?>
                    <?php if (!empty($success_message)): ?>
                        <div class="message success-message"><?php echo htmlspecialchars($success_message); ?></div>
                    <?php else: ?>
                        <div class="registration-form-container">
                            <form id="registration-form" action="<?php echo htmlspecialchars($_SERVER["PHP_SELF"]); ?>" method="post" enctype="multipart/form-data">
                                <?php if ($form_data && isset($form_data['fields']) && is_array($form_data['fields'])): ?>
                                    <?php foreach ($form_data['fields'] as $field): ?>
                                        <?php
                                        // Ensure field properties are strings
                                        $field_type = isset($field['type']) && is_string($field['type']) ? $field['type'] : 'text';
                                        $field_label = isset($field['label']) && is_string($field['label']) ? $field['label'] : (isset($field['name']) && is_string($field['name']) ? $field['name'] : 'Unnamed Field');
                                        $field_name = isset($field['name']) && is_string($field['name']) ? $field['name'] : str_replace(' ', '_', strtolower($field_label));
                                        $is_required = isset($field['required']) && $field['required'];
                                        ?>
                                        <div class="form-group mb-3">
                                            <label for="<?php echo htmlspecialchars($field_name); ?>">
                                                <?php echo htmlspecialchars($field_label); ?>
                                                <?php echo $is_required ? '<span class="text-danger">*</span>' : ''; ?>
                                            </label>
                                            <?php switch ($field_type):
                                                case 'text':
                                                case 'email':
                                                case 'username':
                                                case 'phone':
                                                case 'date':
                                                    ?>
                                                    <input type="<?php echo $field_type === 'phone' ? 'tel' : $field_type; ?>" 
                                                           class="form-control" 
                                                           id="<?php echo htmlspecialchars($field_name); ?>" 
                                                           name="<?php echo htmlspecialchars($field_name); ?>" 
                                                           <?php echo $is_required ? 'required' : ''; ?>>
                                                    <?php break; case 'password': ?>
                                                    <input type="password" 
                                                           class="form-control" 
                                                           id="<?php echo htmlspecialchars($field_name); ?>" 
                                                           name="<?php echo htmlspecialchars($field_name); ?>" 
                                                           minlength="6" 
                                                           <?php echo $is_required ? 'required' : ''; ?>>
                                                    <?php break; case 'textarea': ?>
                                                    <textarea class="form-control" 
                                                              id="<?php echo htmlspecialchars($field_name); ?>" 
                                                              name="<?php echo htmlspecialchars($field_name); ?>" 
                                                              rows="4" 
                                                              <?php echo $is_required ? 'required' : ''; ?>></textarea>
                                                    <?php break; case 'select': ?>
                                                    <select class="form-select" 
                                                            id="<?php echo htmlspecialchars($field_name); ?>" 
                                                            name="<?php echo htmlspecialchars($field_name); ?>" 
                                                            <?php echo $is_required ? 'required' : ''; ?>>
                                                        <option value="">Select an option...</option>
                                                        <?php if (isset($field['options']) && is_array($field['options'])): ?>
                                                            <?php foreach ($field['options'] as $option): ?>
                                                                <?php
                                                                $option_value = is_array($option) && isset($option['value']) ? $option['value'] : $option;
                                                                $option_label = is_array($option) && isset($option['label']) ? $option['label'] : $option;
                                                                ?>
                                                                <option value="<?php echo htmlspecialchars(is_string($option_value) ? $option_value : ''); ?>">
                                                                    <?php echo htmlspecialchars(is_string($option_label) ? $option_label : ''); ?>
                                                                </option>
                                                            <?php endforeach; ?>
                                                        <?php endif; ?>
                                                    </select>
                                                    <?php break; case 'checkbox': ?>
                                                    <input type="checkbox" 
                                                           class="form-check-input" 
                                                           id="<?php echo htmlspecialchars($field_name); ?>" 
                                                           name="<?php echo htmlspecialchars($field_name); ?>" 
                                                           value="1" 
                                                           <?php echo $is_required ? 'required' : ''; ?>>
                                                    <?php break; case 'multi-checkbox': ?>
                                                    <?php if (isset($field['options']) && is_array($field['options'])): ?>
                                                        <?php foreach ($field['options'] as $option): ?>
                                                            <?php
                                                            $option_value = is_array($option) && isset($option['value']) ? $option['value'] : $option;
                                                            $option_label = is_array($option) && isset($option['label']) ? $option['label'] : $option;
                                                            ?>
                                                            <div class="form-check">
                                                                <input type="checkbox" 
                                                                       class="form-check-input" 
                                                                       name="<?php echo htmlspecialchars($field_name); ?>[]" 
                                                                       value="<?php echo htmlspecialchars(is_string($option_value) ? $option_value : ''); ?>" 
                                                                       <?php echo $is_required ? 'required' : ''; ?>>
                                                                <label class="form-check-label">
                                                                    <?php echo htmlspecialchars(is_string($option_label) ? $option_label : ''); ?>
                                                                </label>
                                                            </div>
                                                        <?php endforeach; ?>
                                                    <?php endif; ?>
                                                    <?php break; case 'radio': ?>
                                                    <?php if (isset($field['options']) && is_array($field['options'])): ?>
                                                        <?php foreach ($field['options'] as $option): ?>
                                                            <?php
                                                            $option_value = is_array($option) && isset($option['value']) ? $option['value'] : $option;
                                                            $option_label = is_array($option) && isset($option['label']) ? $option['label'] : $option;
                                                            ?>
                                                            <div class="form-check">
                                                                <input type="radio" 
                                                                       class="form-check-input" 
                                                                       name="<?php echo htmlspecialchars($field_name); ?>" 
                                                                       value="<?php echo htmlspecialchars(is_string($option_value) ? $option_value : ''); ?>" 
                                                                       <?php echo $is_required ? 'required' : ''; ?>>
                                                                <label class="form-check-label">
                                                                    <?php echo htmlspecialchars(is_string($option_label) ? $option_label : ''); ?>
                                                                </label>
                                                            </div>
                                                        <?php endforeach; ?>
                                                    <?php endif; ?>
                                                    <?php break; case 'file': ?>
                                                    <input type="file" 
                                                           class="form-control" 
                                                           id="<?php echo htmlspecialchars($field_name); ?>" 
                                                           name="<?php echo htmlspecialchars($field_name); ?>" 
                                                           <?php echo $is_required ? 'required' : ''; ?>>
                                                    <?php break; default: ?>
                                                    <input type="text" 
                                                           class="form-control" 
                                                           id="<?php echo htmlspecialchars($field_name); ?>" 
                                                           name="<?php echo htmlspecialchars($field_name); ?>" 
                                                           <?php echo $is_required ? 'required' : ''; ?>>
                                            <?php endswitch; ?>
                                        </div>
                                    <?php endforeach; ?>
                                <?php else: ?>
                                    <div class="alert alert-warning">
                                        No form fields available. Please contact the administrator.
                                    </div>
                                <?php endif; ?>
                                <div class="form-group mb-3">
                                    <label for="captcha_input"><?php echo htmlspecialchars($captcha_question); ?></label>
                                    <input type="text" class="form-control" id="captcha_input" name="captcha_input" required>
                                </div>
                                <input type="hidden" name="form_submission" value="1">
                                <div class="form-actions mt-3 text-center">
                                    <button type="submit" class="btn btn-success">Complete Registration</button>
                                </div>
                            </form>
                        </div>
                    <?php endif; ?>
                </div>
                <div class="card-footer text-center">
                    <p>Already have an account? <a href="login.php">Login here</a></p>
                </div>
            </div>
        </div>
    </div>
</div>
<?php ob_end_flush(); // End output buffering ?>
<?php require_once 'includes/footer.php'; ?>