<?php
require_once 'includes/functions.php';
require_once 'includes/db_config.php';

if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

// Check login
if (!is_logged_in()) {
    redirect("login.php");
}

$currency_symbol = get_setting('currency_symbol', $link) ?? '₦';
$user_id = intval($_SESSION['user_id']);
$user_name = '';
$membership_status = '';
$is_admin = is_admin();

// Fetch user info
$sql = "SELECT full_name, membership_status FROM users WHERE id = ?";
if ($stmt = mysqli_prepare($link, $sql)) {
    mysqli_stmt_bind_param($stmt, "i", $user_id);
    mysqli_stmt_execute($stmt);
    mysqli_stmt_bind_result($stmt, $user_name, $membership_status);
    mysqli_stmt_fetch($stmt);
    mysqli_stmt_close($stmt);
}

// Pagination setup
$page_debts = isset($_GET['page_debts']) ? max(1, intval($_GET['page_debts'])) : 1;
$per_page_debts = isset($_GET['per_page_debts']) ? intval($_GET['per_page_debts']) : 10;
$offset_debts = ($page_debts - 1) * $per_page_debts;

// Fetch debts using obligation_id
$sql_debts = "SELECT 
                 po.payment_type,
                 po.required_amount,
                 COALESCE(SUM(p.amount), 0) AS total_paid,
                 (COALESCE(po.required_amount, 0) - COALESCE(SUM(p.amount), 0)) AS debt,
                 po.payment_cycle,
                 po.due_date
              FROM payment_obligations po
              JOIN users u ON po.user_id = u.id
              LEFT JOIN payments p ON p.obligation_id = po.id AND p.payment_status = 'Paid'
              WHERE po.user_id = ? AND po.deleted_at IS NULL 
                  AND u.role = 'member' 
                  AND (u.deleted_at IS NULL OR u.deleted_at > NOW())
              GROUP BY po.id, po.required_amount, po.payment_cycle, po.due_date
              HAVING debt > 0
              ORDER BY debt DESC
              LIMIT ? OFFSET ?";
$stmt = mysqli_prepare($link, $sql_debts);
$debts = [];
if ($stmt) {
    mysqli_stmt_bind_param($stmt, 'iii', $user_id, $per_page_debts, $offset_debts);
    if (!mysqli_stmt_execute($stmt)) {
        error_log("Debts query execution failed: " . mysqli_stmt_error($stmt));
    } else {
        $result_debts = mysqli_stmt_get_result($stmt);
        while ($row = mysqli_fetch_assoc($result_debts)) {
            $debts[] = $row;
        }
    }
    mysqli_stmt_close($stmt);
} else {
    error_log("Failed to prepare debts query: " . mysqli_error($link));
}

// Debts count for pagination
$sql_count_debts = "SELECT COUNT(*) AS total
                    FROM (
                        SELECT 
                            po.id,
                            po.required_amount,
                            COALESCE(SUM(p.amount), 0) AS total_paid
                        FROM payment_obligations po
                        JOIN users u ON po.user_id = u.id
                        LEFT JOIN payments p ON p.obligation_id = po.id AND p.payment_status = 'Paid'
                        WHERE po.user_id = ? AND po.deleted_at IS NULL 
                            AND u.role = 'member' 
                            AND (u.deleted_at IS NULL OR u.deleted_at > NOW())
                        GROUP BY po.id, po.required_amount
                        HAVING (COALESCE(po.required_amount, 0) - COALESCE(SUM(p.amount), 0)) > 0
                    ) AS debt_count";
$stmt = mysqli_prepare($link, $sql_count_debts);
$total_records_debts = 0;
$total_pages_debts = 1;
if ($stmt) {
    mysqli_stmt_bind_param($stmt, 'i', $user_id);
    if (!mysqli_stmt_execute($stmt)) {
        error_log("Debts count query execution failed: " . mysqli_stmt_error($stmt));
    } else {
        $result = mysqli_stmt_get_result($stmt);
        $total_records_debts = $result ? mysqli_fetch_assoc($result)['total'] : 0;
        $total_pages_debts = ceil($total_records_debts / $per_page_debts);
    }
    mysqli_stmt_close($stmt);
} else {
    error_log("Failed to prepare debts count query: " . mysqli_error($link));
}
?>

<?php require_once 'includes/header.php'; ?>
<div class="container-fluid">
    <div class="row">
        <?php require_once 'includes/sidebar.php'; ?>
        <main class="col-md-9 col-lg-10 main-content">
            <h2>Welcome, <?php echo htmlspecialchars($user_name); ?>!</h2>

            <!-- Membership Status -->
            <div class="card mb-4">
                <div class="card-header">Membership Status</div>
                <div class="card-body">
                    <p>Your current membership status is: <strong><?php echo ucfirst($membership_status); ?></strong>.</p>
                </div>
            </div>

            <!-- Debts Table -->
            <div class="card mb-4">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <span>Outstanding Debts</span>
                    <?php if ($is_admin): ?>
                        <div>
                            <button class="btn btn-sm btn-outline-secondary" onclick="printTable('debtsTable')"><i class="fas fa-print"></i> Print</button>
                            <a href="api/export_debts.php" class="btn btn-sm btn-outline-success"><i class="fas fa-file-csv"></i> Export to CSV</a>
                        </div>
                    <?php endif; ?>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table table-striped table-hover" id="debtsTable">
                            <thead>
                                <tr>
                                    <th>Payment Type</th>
                                    <th>Required Amount</th>
                                    <th>Total Paid</th>
                                    <th>Debt</th>
                                    <th>Payment Cycle</th>
                                    <th>Due Date</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (empty($debts)): ?>
                                    <tr><td colspan="6" class="text-center">No debts found.</td></tr>
                                <?php else: ?>
                                    <?php foreach ($debts as $record): ?>
                                        <tr>
                                            <td><?php echo htmlspecialchars($record['payment_type'] ?? 'N/A'); ?></td>
                                            <td><?php echo htmlspecialchars($currency_symbol . ' ' . number_format($record['required_amount'] ?? 0.00, 2)); ?></td>
                                            <td><?php echo htmlspecialchars($currency_symbol . ' ' . number_format($record['total_paid'], 2)); ?></td>
                                            <td><?php echo htmlspecialchars($currency_symbol . ' ' . number_format($record['debt'], 2)); ?></td>
                                            <td><?php echo htmlspecialchars($record['payment_cycle'] ?? 'N/A'); ?></td>
                                            <td><?php echo htmlspecialchars($record['due_date'] ? date('Y-m-d', strtotime($record['due_date'])) : 'N/A'); ?></td>
                                        </tr>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>

                    <!-- Pagination -->
                    <?php if ($total_pages_debts > 1): ?>
                        <nav aria-label="Debts page navigation">
                            <ul class="pagination justify-content-center mt-3">
                                <li class="page-item <?php echo $page_debts <= 1 ? 'disabled' : ''; ?>">
                                    <a class="page-link" href="?page_debts=<?php echo $page_debts-1; ?>&per_page_debts=<?php echo $per_page_debts; ?>">Previous</a>
                                </li>
                                <?php for ($i=1; $i<=$total_pages_debts; $i++): ?>
                                    <li class="page-item <?php echo $i==$page_debts?'active':''; ?>">
                                        <a class="page-link" href="?page_debts=<?php echo $i; ?>&per_page_debts=<?php echo $per_page_debts; ?>"><?php echo $i; ?></a>
                                    </li>
                                <?php endfor; ?>
                                <li class="page-item <?php echo $page_debts >= $total_pages_debts ? 'disabled':''; ?>">
                                    <a class="page-link" href="?page_debts=<?php echo $page_debts+1; ?>&per_page_debts=<?php echo $per_page_debts; ?>">Next</a>
                                </li>
                            </ul>
                        </nav>
                    <?php endif; ?>
                </div>
            </div>

        </main>
    </div>
</div>

<script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
<script>
function printTable(tableId){
    let table = document.getElementById(tableId).outerHTML;
    let w = window.open('', '', 'width=800,height=600');
    w.document.write('<html><head><title>Print</title></head><body>');
    w.document.write(table);
    w.document.write('</body></html>');
    w.document.close();
    w.print();
}
</script>
<?php require_once 'includes/footer.php'; ?>