<?php
// Ensure database connection is available
if (!isset($link) || !$link) {
    error_log("Database connection not available in reports_data.php");
    die("ERROR: Database connection not available.");
}

// Include utility functions
require_once 'includes/financial_utils.php';
require_once 'includes/functions.php';
require_once 'includes/financial_queries.php';

// Process type filter
$type_filter_condition = '';
$type_filter_param = null;
if ($type_filter !== '') {
    $type_parts = explode('_', $type_filter, 2);
    $type_category = $type_parts[0] ?? '';
    $type_name = $type_parts[1] ?? '';
    if ($type_category && $type_name) {
        if ($type_category === 'income' || $type_category === 'payment') {
            $type_filter_condition = " AND pt.type_name = ?";
        } elseif ($type_category === 'expense') {
            $type_filter_condition = " AND et.type_name = ?";
        } elseif ($type_category === 'benefit') {
            $type_filter_condition = " AND bt.type_name = ?";
        }
        $type_filter_param = $type_name;
    } else {
        error_log("Invalid type_filter format: $type_filter");
        $type_filter = '';
    }
}

// Define time_period
$time_period = 'date_range';

// Time filter for date range, allowing NULL due dates
$time_filter = " AND DATE(created_at) BETWEEN ? AND ?";
$time_filter_debts = " AND (po.due_date IS NULL OR DATE(po.due_date) BETWEEN ? AND ?)";

// Pagination settings
$tables = ['income', 'expenses', 'debts', 'benefits'];
$pagination = [];
foreach ($tables as $table) {
    $pagination[$table] = [
        'per_page' => isset($_GET["per_page_$table"]) && in_array((int)$_GET["per_page_$table"], [5, 10, 25, 50, 100]) ? (int)$_GET["per_page_$table"] : 5,
        'page' => isset($_GET["page_$table"]) && is_numeric($_GET["page_$table"]) && $_GET["page_$table"] > 0 ? (int)$_GET["page_$table"] : 1
    ];
    $pagination[$table]['offset'] = ($pagination[$table]['page'] - 1) * $pagination[$table]['per_page'];
}

// Total counts for pagination
$total_counts = [];
$total_pages = [];
foreach ($tables as $table) {
    $sql_count = '';
    $count_bind_types = '';
    $count_bind_params = [];
    $search_condition = '';

    if ($table === 'income') {
        $sql_count = "SELECT COUNT(*) as total FROM payments p JOIN users u ON p.user_id = u.id WHERE payment_status = 'Paid'";
        if ($record_type === 'all' || $record_type === 'income') {
            $sql_count .= str_replace('created_at', 'p.created_at', $time_filter);
            $count_bind_types .= 'ss';
            $count_bind_params[] = $start_date;
            $count_bind_params[] = $end_date;
            if ($type_filter && (strpos($type_filter, 'income_') === 0 || strpos($type_filter, 'payment_') === 0)) {
                $sql_count .= $type_filter_condition;
                $count_bind_types .= 's';
                $count_bind_params[] = $type_filter_param;
            }
        } else {
            $sql_count .= ' AND FALSE';
        }
        $search_condition = " AND (u.full_name LIKE ? OR p.payment_type LIKE ? OR CAST(p.id AS CHAR) LIKE ?)";
    } elseif ($table === 'expenses') {
        $sql_count = "SELECT COUNT(*) as total FROM expenses e LEFT JOIN expense_types et ON e.expense_type_id = et.id WHERE 1=1";
        if ($record_type === 'all' || $record_type === 'expenses') {
            $sql_count .= str_replace('created_at', 'e.created_at', $time_filter);
            $count_bind_types .= 'ss';
            $count_bind_params[] = $start_date;
            $count_bind_params[] = $end_date;
            if ($type_filter && strpos($type_filter, 'expense_') === 0) {
                $sql_count .= $type_filter_condition;
                $count_bind_types .= 's';
                $count_bind_params[] = $type_filter_param;
            }
        } else {
            $sql_count .= ' AND FALSE';
        }
        $search_condition = " AND (et.type_name LIKE ? OR e.description LIKE ? OR CAST(e.id AS CHAR) LIKE ?)";
    } elseif ($table === 'benefits') {
        $sql_count = "SELECT COUNT(*) as total FROM member_benefits b JOIN users u ON b.member_id = u.id JOIN benefit_types bt ON b.benefit_type_id = bt.id WHERE 1=1";
        if ($record_type === 'all' || $record_type === 'benefits') {
            $sql_count .= str_replace('created_at', 'b.created_at', $time_filter);
            $count_bind_types .= 'ss';
            $count_bind_params[] = $start_date;
            $count_bind_params[] = $end_date;
            if ($type_filter && strpos($type_filter, 'benefit_') === 0) {
                $sql_count .= $type_filter_condition;
                $count_bind_types .= 's';
                $count_bind_params[] = $type_filter_param;
            }
        } else {
            $sql_count .= ' AND FALSE';
        }
        $search_condition = " AND (u.full_name LIKE ? OR bt.type_name LIKE ? OR CAST(b.id AS CHAR) LIKE ?)";
    } elseif ($table === 'debts') {
        $sql_count = "SELECT po.id, u.id, po.required_amount, po.due_date
                      FROM payment_obligations po 
                      JOIN users u ON po.user_id = u.id 
                      JOIN payment_types pt ON po.payment_type = pt.type_name 
                      LEFT JOIN payments p ON p.obligation_id = po.id AND p.payment_status = 'Paid'
                      WHERE po.deleted_at IS NULL 
                          AND u.role = 'member' 
                          AND (u.deleted_at IS NULL OR u.deleted_at > NOW())
                          AND COALESCE(po.required_amount, 0) > 0";
        $count_bind_types = '';
        $count_bind_params = [];
        if ($record_type === 'all' || $record_type === 'debts') {
            $sql_count .= $time_filter_debts;
            $count_bind_types .= 'ss';
            $count_bind_params[] = $start_date;
            $count_bind_params[] = $end_date;
            if ($type_filter && (strpos($type_filter, 'income_') === 0 || strpos($type_filter, 'payment_') === 0)) {
                $sql_count .= $type_filter_condition;
                $count_bind_types .= 's';
                $count_bind_params[] = $type_filter_param;
            }
        } else {
            $sql_count .= ' AND FALSE';
        }
        $search_condition = " AND (u.full_name LIKE ? OR pt.type_name LIKE ? OR CAST(po.id AS CHAR) LIKE ?)";
        if (!empty($search)) {
            $sql_count .= $search_condition;
            $count_bind_types .= 'sss';
            $search_param = "%$search%";
            $count_bind_params[] = $search_param;
            $count_bind_params[] = $search_param;
            $count_bind_params[] = $search_param;
        }
        $sql_count .= " GROUP BY po.id, u.id, po.required_amount, po.due_date 
                        HAVING (COALESCE(po.required_amount, 0) - COALESCE(SUM(p.amount), 0)) > 0";
    }

    if ($table !== 'debts') {
        if (!empty($search)) {
            $sql_count .= $search_condition;
            $count_bind_types .= 'sss';
            $search_param = "%$search%";
            $count_bind_params[] = $search_param;
            $count_bind_params[] = $search_param;
            $count_bind_params[] = $search_param;
        }
    }

    $stmt = mysqli_prepare($link, $sql_count);
    if ($stmt === false) {
        error_log("Failed to prepare statement for $table count: " . mysqli_error($link));
        $total_counts[$table] = 0;
        continue;
    }
    if ($count_bind_types && !empty($count_bind_params)) {
        if (!mysqli_stmt_bind_param($stmt, $count_bind_types, ...$count_bind_params)) {
            error_log("Failed to bind parameters for $table count: types=$count_bind_types, params=" . json_encode($count_bind_params));
            $total_counts[$table] = 0;
            mysqli_stmt_close($stmt);
            continue;
        }
    }
    if (!mysqli_stmt_execute($stmt)) {
        error_log("Failed to execute $table count query: " . mysqli_stmt_error($stmt));
        $total_counts[$table] = 0;
        mysqli_stmt_close($stmt);
        continue;
    }
    $result = mysqli_stmt_get_result($stmt);
    $total_counts[$table] = ($table === 'debts') ? mysqli_num_rows($result) : (mysqli_fetch_assoc($result)['total'] ?? 0);
    mysqli_stmt_close($stmt);
    $total_pages[$table] = ceil($total_counts[$table] / $pagination[$table]['per_page']);
}

// Calculate total debt from debt report to ensure consistency
$total_debt_from_report = 0;
if ($record_type === 'all' || $record_type === 'debts') {
    $sql_total_debt = "SELECT COALESCE(SUM(debt), 0) AS total_debt
                       FROM (
                           SELECT (COALESCE(po.required_amount, 0) - COALESCE(SUM(p.amount), 0)) AS debt
                           FROM payment_obligations po 
                           JOIN users u ON po.user_id = u.id 
                           JOIN payment_types pt ON po.payment_type = pt.type_name 
                           LEFT JOIN payments p ON p.obligation_id = po.id AND p.payment_status = 'Paid'
                           WHERE po.deleted_at IS NULL 
                               AND u.role = 'member' 
                               AND (u.deleted_at IS NULL OR u.deleted_at > NOW())
                               AND COALESCE(po.required_amount, 0) > 0";
    $total_debt_bind_types = '';
    $total_debt_bind_params = [];
    $sql_total_debt .= $time_filter_debts;
    $total_debt_bind_types .= 'ss';
    $total_debt_bind_params[] = $start_date;
    $total_debt_bind_params[] = $end_date;
    if ($type_filter && (strpos($type_filter, 'income_') === 0 || strpos($type_filter, 'payment_') === 0)) {
        $sql_total_debt .= $type_filter_condition;
        $total_debt_bind_types .= 's';
        $total_debt_bind_params[] = $type_filter_param;
    }
    if (!empty($search)) {
        $sql_total_debt .= " AND (u.full_name LIKE ? OR pt.type_name LIKE ? OR CAST(po.id AS CHAR) LIKE ?)";
        $total_debt_bind_types .= 'sss';
        $search_param = "%$search%";
        $total_debt_bind_params[] = $search_param;
        $total_debt_bind_params[] = $search_param;
        $total_debt_bind_params[] = $search_param;
    }
    $sql_total_debt .= " GROUP BY po.id, u.id, po.required_amount, po.due_date, po.payment_cycle
                         HAVING debt > 0
                       ) AS subquery";
    $stmt = mysqli_prepare($link, $sql_total_debt);
    if ($stmt === false) {
        error_log("Failed to prepare total debt query: " . mysqli_error($link));
        $total_debt_from_report = 0;
    } else {
        if ($total_debt_bind_types && !empty($total_debt_bind_params)) {
            if (!mysqli_stmt_bind_param($stmt, $total_debt_bind_types, ...$total_debt_bind_params)) {
                error_log("Failed to bind parameters for total debt query: types=$total_debt_bind_types, params=" . json_encode($total_debt_bind_params));
            } elseif (mysqli_stmt_execute($stmt)) {
                $result = mysqli_stmt_get_result($stmt);
                $row = mysqli_fetch_assoc($result);
                $total_debt_from_report = (float)($row['total_debt'] ?? 0);
                mysqli_free_result($result);
            } else {
                error_log("Failed to execute total debt query: " . mysqli_stmt_error($stmt));
            }
        } else {
            if (mysqli_stmt_execute($stmt)) {
                $result = mysqli_stmt_get_result($stmt);
                $row = mysqli_fetch_assoc($result);
                $total_debt_from_report = (float)($row['total_debt'] ?? 0);
                mysqli_free_result($result);
            } else {
                error_log("Failed to execute total debt query: " . mysqli_stmt_error($stmt));
            }
        }
        mysqli_stmt_close($stmt);
    }
}

// Fetch Type Summaries and Financial Metrics
$payment_types_summary = get_payment_types_summary($link, $record_type, $time_period, $start_date, $type_filter, 'due_date', $time_filter_debts);
$expense_types_summary = get_expense_types_summary($link, $record_type, $time_period, $start_date, $type_filter);
$benefit_types_summary = get_benefit_types_summary($link, $record_type, $time_period, $start_date, $type_filter);
$financial_metrics = get_financial_metrics($link, $record_type, $time_period, $start_date, $type_filter, 'due_date', $time_filter_debts);

// Debug: Log raw payment_types_summary
error_log("Raw payment_types_summary: " . var_export($payment_types_summary, true));

// Handle payment_types_summary if it's a scalar or unexpected type
if (is_scalar($payment_types_summary) || !is_array($payment_types_summary)) {
    // If it's a single number (e.g., total_debt), convert to a summary array
    $payment_types_summary = [[
        'type_name' => 'Total Payments',
        'required_amount' => $financial_metrics['total_required'] ?? 0.0,
        'total_paid' => $financial_metrics['total_income'] ?? 0.0,
        'total_debt' => $total_debt_from_report
    ]];
} elseif (is_string($payment_types_summary)) {
    $payment_types_summary = json_decode($payment_types_summary, true) ?? [];
}

// Ensure payment_types_summary is an array of records
if (!is_array($payment_types_summary) || !isset($payment_types_summary[0]['type_name'])) {
    $payment_types_summary = [[
        'type_name' => 'Total Payments',
        'required_amount' => $financial_metrics['total_required'] ?? 0.0,
        'total_paid' => $financial_metrics['total_income'] ?? 0.0,
        'total_debt' => $total_debt_from_report
    ]];
}

// Handle financial_metrics if it's a string or unexpected type
if (is_string($financial_metrics)) {
    $financial_metrics = json_decode($financial_metrics, true) ?? [];
} elseif (!is_array($financial_metrics)) {
    $financial_metrics = [];
}

// Ensure financial_metrics has required keys
$financial_metrics = array_merge([
    'total_required' => 0.0,
    'total_income' => 0.0,
    'total_debt' => 0.0,
    'excess_donations' => 0.0,
    'available_balance' => 0.0
], $financial_metrics);

// Fetch totals using the existing function from functions.php
$totals = get_financial_totals($link, $record_type, $start_date, $end_date);
$total_income = $totals['total_payments'] ?? 0.0;
$total_expenses = $totals['total_expenses'] ?? 0.0;
$total_benefits = $totals['total_benefits'] ?? 0.0;
$total_debts = $total_debt_from_report;

// Override totals with values from summaries and debt report if needed
$total_required = array_sum(array_column($payment_types_summary, 'required_amount')) ?: $financial_metrics['total_required'];
if (!empty($payment_types_summary)) {
    $total_income = array_sum(array_column($payment_types_summary, 'total_paid')) ?: $total_income;
}
$financial_metrics['total_required'] = $total_required;
$financial_metrics['total_income'] = $total_income;
$financial_metrics['total_debt'] = $total_debts;

// Calculate available balance as total_income - total_expenses - total_benefits
$available_balance = $total_income - $total_expenses - $total_benefits;
$financial_metrics['available_balance'] = $available_balance;

// Debug: Log final totals
error_log("Final totals: total_required=$total_required, total_income=$total_income, total_expenses=$total_expenses, total_benefits=$total_benefits, total_debts=$total_debts, available_balance=$available_balance");

// Fetch records
$income = [];
if ($record_type === 'all' || $record_type === 'income') {
    $sql_income = "SELECT
                     p.id,
                     u.member_id,
                     IFNULL(u.full_name, 'Unknown') AS full_name,
                     p.payment_type,
                     p.amount,
                     p.payment_status,
                     p.created_at,
                     p.payment_date,
                     po.payment_cycle,
                     pt.event_date
                   FROM payments p
                   JOIN users u ON p.user_id = u.id
                   LEFT JOIN payment_obligations po ON p.obligation_id = po.id
                   LEFT JOIN payment_types pt ON p.payment_type = pt.type_name
                   WHERE payment_status = 'Paid'";
    $income_bind_types = '';
    $income_bind_params = [];
    if ($record_type === 'all' || $record_type === 'income') {
        $sql_income .= str_replace('created_at', 'p.created_at', $time_filter);
        $income_bind_types .= 'ss';
        $income_bind_params[] = $start_date;
        $income_bind_params[] = $end_date;
        if ($type_filter && (strpos($type_filter, 'income_') === 0 || strpos($type_filter, 'payment_') === 0)) {
            $sql_income .= $type_filter_condition;
            $income_bind_types .= 's';
            $income_bind_params[] = $type_filter_param;
        }
        if (!empty($search)) {
            $sql_income .= " AND (u.full_name LIKE ? OR p.payment_type LIKE ? OR CAST(p.id AS CHAR) LIKE ?)";
            $income_bind_types .= 'sss';
            $search_param = "%$search%";
            $income_bind_params[] = $search_param;
            $income_bind_params[] = $search_param;
            $income_bind_params[] = $search_param;
        }
    }
    $sql_income .= " ORDER BY p.created_at DESC LIMIT ? OFFSET ?";
    $income_bind_types .= 'ii';
    $income_bind_params[] = $pagination['income']['per_page'];
    $income_bind_params[] = $pagination['income']['offset'];
    $stmt = mysqli_prepare($link, $sql_income);
    if ($stmt === false) {
        error_log("Failed to prepare income query: " . mysqli_error($link));
        $income = [];
    } else {
        if ($income_bind_types) {
            if (!mysqli_stmt_bind_param($stmt, $income_bind_types, ...$income_bind_params)) {
                error_log("Failed to bind parameters for income query: types=$income_bind_types, params=" . json_encode($income_bind_params));
                $income = [];
            } elseif (mysqli_stmt_execute($stmt)) {
                $result = mysqli_stmt_get_result($stmt);
                while ($row = mysqli_fetch_assoc($result)) {
                    $income[] = $row;
                }
                mysqli_free_result($result);
            } else {
                error_log("Failed to execute income query: " . mysqli_stmt_error($stmt));
                $income = [];
            }
        } else {
            if (mysqli_stmt_execute($stmt)) {
                $result = mysqli_stmt_get_result($stmt);
                while ($row = mysqli_fetch_assoc($result)) {
                    $income[] = $row;
                }
                mysqli_free_result($result);
            } else {
                error_log("Failed to execute income query: " . mysqli_stmt_error($stmt));
                $income = [];
            }
        }
        mysqli_stmt_close($stmt);
    }
}

$expenses = [];
if ($record_type === 'all' || $record_type === 'expenses') {
    $sql_expenses = "SELECT
                      e.id,
                      et.type_name AS expense_type,
                      e.amount,
                      IFNULL(e.description, 'N/A') AS description,
                      e.created_at
                    FROM expenses e
                    LEFT JOIN expense_types et ON e.expense_type_id = et.id
                    WHERE 1=1";
    $expenses_bind_types = '';
    $expenses_bind_params = [];
    if ($record_type === 'all' || $record_type === 'expenses') {
        $sql_expenses .= str_replace('created_at', 'e.created_at', $time_filter);
        $expenses_bind_types .= 'ss';
        $expenses_bind_params[] = $start_date;
        $expenses_bind_params[] = $end_date;
        if ($type_filter && strpos($type_filter, 'expense_') === 0) {
            $sql_expenses .= $type_filter_condition;
            $expenses_bind_types .= 's';
            $expenses_bind_params[] = $type_filter_param;
        }
        if (!empty($search)) {
            $sql_expenses .= " AND (et.type_name LIKE ? OR e.description LIKE ? OR CAST(e.id AS CHAR) LIKE ?)";
            $expenses_bind_types .= 'sss';
            $search_param = "%$search%";
            $expenses_bind_params[] = $search_param;
            $expenses_bind_params[] = $search_param;
            $expenses_bind_params[] = $search_param;
        }
    }
    $sql_expenses .= " ORDER BY e.created_at DESC LIMIT ? OFFSET ?";
    $expenses_bind_types .= 'ii';
    $expenses_bind_params[] = $pagination['expenses']['per_page'];
    $expenses_bind_params[] = $pagination['expenses']['offset'];
    $stmt = mysqli_prepare($link, $sql_expenses);
    if ($stmt === false) {
        error_log("Failed to prepare expenses query: " . mysqli_error($link));
        $expenses = [];
    } else {
        if ($expenses_bind_types) {
            if (!mysqli_stmt_bind_param($stmt, $expenses_bind_types, ...$expenses_bind_params)) {
                error_log("Failed to bind parameters for expenses query: types=$expenses_bind_types, params=" . json_encode($expenses_bind_params));
                $expenses = [];
            } elseif (mysqli_stmt_execute($stmt)) {
                $result = mysqli_stmt_get_result($stmt);
                while ($row = mysqli_fetch_assoc($result)) {
                    $expenses[] = $row;
                }
                mysqli_free_result($result);
            } else {
                error_log("Failed to execute expenses query: " . mysqli_stmt_error($stmt));
                $expenses = [];
            }
        } else {
            if (mysqli_stmt_execute($stmt)) {
                $result = mysqli_stmt_get_result($stmt);
                while ($row = mysqli_fetch_assoc($result)) {
                    $expenses[] = $row;
                }
                mysqli_free_result($result);
            } else {
                error_log("Failed to execute expenses query: " . mysqli_stmt_error($stmt));
                $expenses = [];
            }
        }
        mysqli_stmt_close($stmt);
    }
}

$benefits = [];
if ($record_type === 'all' || $record_type === 'benefits') {
    $sql_benefits = "SELECT
                      b.id,
                      u.member_id,
                      IFNULL(u.full_name, 'Unknown') AS full_name,
                      bt.type_name AS benefit_type,
                      b.amount,
                      b.created_at
                    FROM member_benefits b
                    JOIN users u ON b.member_id = u.id
                    JOIN benefit_types bt ON b.benefit_type_id = bt.id
                    WHERE 1=1";
    $benefits_bind_types = '';
    $benefits_bind_params = [];
    if ($record_type === 'all' || $record_type === 'benefits') {
        $sql_benefits .= str_replace('created_at', 'b.created_at', $time_filter);
        $benefits_bind_types .= 'ss';
        $benefits_bind_params[] = $start_date;
        $benefits_bind_params[] = $end_date;
        if ($type_filter && strpos($type_filter, 'benefit_') === 0) {
            $sql_benefits .= $type_filter_condition;
            $benefits_bind_types .= 's';
            $benefits_bind_params[] = $type_filter_param;
        }
        if (!empty($search)) {
            $sql_benefits .= " AND (u.full_name LIKE ? OR bt.type_name LIKE ? OR CAST(b.id AS CHAR) LIKE ?)";
            $benefits_bind_types .= 'sss';
            $search_param = "%$search%";
            $benefits_bind_params[] = $search_param;
            $benefits_bind_params[] = $search_param;
            $benefits_bind_params[] = $search_param;
        }
    }
    $sql_benefits .= " ORDER BY b.created_at DESC LIMIT ? OFFSET ?";
    $benefits_bind_types .= 'ii';
    $benefits_bind_params[] = $pagination['benefits']['per_page'];
    $benefits_bind_params[] = $pagination['benefits']['offset'];
    $stmt = mysqli_prepare($link, $sql_benefits);
    if ($stmt === false) {
        error_log("Failed to prepare benefits query: " . mysqli_error($link));
        $benefits = [];
    } else {
        if ($benefits_bind_types) {
            if (!mysqli_stmt_bind_param($stmt, $benefits_bind_types, ...$benefits_bind_params)) {
                error_log("Failed to bind parameters for benefits query: types=$benefits_bind_types, params=" . json_encode($benefits_bind_params));
                $benefits = [];
            } elseif (mysqli_stmt_execute($stmt)) {
                $result = mysqli_stmt_get_result($stmt);
                while ($row = mysqli_fetch_assoc($result)) {
                    $benefits[] = $row;
                }
                mysqli_free_result($result);
            } else {
                error_log("Failed to execute benefits query: " . mysqli_stmt_error($stmt));
                $benefits = [];
            }
        } else {
            if (mysqli_stmt_execute($stmt)) {
                $result = mysqli_stmt_get_result($stmt);
                while ($row = mysqli_fetch_assoc($result)) {
                    $benefits[] = $row;
                }
                mysqli_free_result($result);
            } else {
                error_log("Failed to execute benefits query: " . mysqli_stmt_error($stmt));
                $benefits = [];
            }
        }
        mysqli_stmt_close($stmt);
    }
}

$debts = [];
if ($record_type === 'all' || $record_type === 'debts') {
    $sql_debts = "SELECT 
                u.id AS user_id,
                u.member_id,
                IFNULL(u.full_name, 'Unknown') AS full_name,
                pt.type_name AS payment_type,
                COALESCE(po.required_amount, 0) AS required_amount,
                COALESCE(SUM(p.amount), 0) AS total_paid,
                (COALESCE(po.required_amount, 0) - COALESCE(SUM(p.amount), 0)) AS debt,
                po.payment_cycle,
                po.due_date
             FROM payment_obligations po
             JOIN users u ON po.user_id = u.id
             JOIN payment_types pt ON po.payment_type = pt.type_name
             LEFT JOIN payments p ON p.obligation_id = po.id AND p.payment_status = 'Paid'
             WHERE po.deleted_at IS NULL
                 AND u.role = 'member'
                 AND (u.deleted_at IS NULL OR u.deleted_at > NOW())
                 AND COALESCE(po.required_amount, 0) > 0";
    $debts_bind_types = '';
    $debts_bind_params = [];
    if ($record_type === 'all' || $record_type === 'debts') {
        $sql_debts .= $time_filter_debts;
        $debts_bind_types .= 'ss';
        $debts_bind_params[] = $start_date;
        $debts_bind_params[] = $end_date;
        if ($type_filter && (strpos($type_filter, 'income_') === 0 || strpos($type_filter, 'payment_') === 0)) {
            $sql_debts .= $type_filter_condition;
            $debts_bind_types .= 's';
            $debts_bind_params[] = $type_filter_param;
        }
        if (!empty($search)) {
            $sql_debts .= " AND (u.full_name LIKE ? OR pt.type_name LIKE ? OR CAST(po.id AS CHAR) LIKE ?)";
            $debts_bind_types .= 'sss';
            $search_param = "%$search%";
            $debts_bind_params[] = $search_param;
            $debts_bind_params[] = $search_param;
            $debts_bind_params[] = $search_param;
        }
    }
    $sql_debts .= " GROUP BY po.id, u.id, po.required_amount, po.payment_cycle, po.due_date
                    HAVING debt > 0
                    ORDER BY debt DESC, po.due_date ASC
                    LIMIT ? OFFSET ?";
    $debts_bind_types .= 'ii';
    $debts_bind_params[] = $pagination['debts']['per_page'];
    $debts_bind_params[] = $pagination['debts']['offset'];
    $stmt = mysqli_prepare($link, $sql_debts);
    if ($stmt === false) {
        error_log("Failed to prepare debts query: " . mysqli_error($link));
        $debts = [];
    } else {
        if ($debts_bind_types) {
            if (!mysqli_stmt_bind_param($stmt, $debts_bind_types, ...$debts_bind_params)) {
                error_log("Failed to bind parameters for debts query: types=$debts_bind_types, params=" . json_encode($debts_bind_params));
                $debts = [];
            } elseif (mysqli_stmt_execute($stmt)) {
                $result = mysqli_stmt_get_result($stmt);
                while ($row = mysqli_fetch_assoc($result)) {
                    $debts[] = $row;
                }
                mysqli_free_result($result);
            } else {
                error_log("Failed to execute debts query: " . mysqli_stmt_error($stmt));
                $debts = [];
            }
        } else {
            if (mysqli_stmt_execute($stmt)) {
                $result = mysqli_stmt_get_result($stmt);
                while ($row = mysqli_fetch_assoc($result)) {
                    $debts[] = $row;
                }
                mysqli_free_result($result);
            } else {
                error_log("Failed to execute debts query: " . mysqli_stmt_error($stmt));
                $debts = [];
            }
        }
        mysqli_stmt_close($stmt);
    }
}

?>