<?php
// bank_settings.php
ob_start();
if (session_status() !== PHP_SESSION_ACTIVE) {
    session_start();
}

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Clear OPCache
if (function_exists('opcache_reset')) {
    opcache_reset();
}

require_once 'includes/functions.php';
require_once 'includes/db_config.php';

// Debug session status
if (session_status() !== PHP_SESSION_ACTIVE) {
    error_log("Session not active in bank_settings.php");
}

// Session timeout
$timeout = 1800; // 30 minutes
if (isset($_SESSION['last_activity']) && (time() - $_SESSION['last_activity']) > $timeout) {
    session_unset();
    session_destroy();
    redirect("login.php?timeout=1");
}
$_SESSION['last_activity'] = time();

// Restrict to super_admin role
if (!is_logged_in() || $_SESSION['user_role'] !== 'super_admin') {
    redirect("login.php");
}

// Generate CSRF token
if (!isset($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

$message = '';
$message_type = '';
$account_name = '';
$bank_name = '';
$account_number = '';

// Check database connection
if (mysqli_connect_error()) {
    $message = "Database connection failed: " . mysqli_connect_error();
    $message_type = 'danger';
    error_log("Database connection failed in bank_settings.php: " . mysqli_connect_error());
} else {
    // Check if settings table exists
    $check_table = mysqli_query($link, "SHOW TABLES LIKE 'settings'");
    if (mysqli_num_rows($check_table) == 0) {
        $message = "Error: Settings table does not exist. Please contact the system administrator.";
        $message_type = 'danger';
        error_log("Settings table missing in bank_settings.php");
    } else {
        // Handle form submission
        if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['save_bank_details']) && isset($_POST['csrf_token']) && hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {
            $account_name = sanitize_input($_POST['account_name'] ?? '');
            $bank_name = sanitize_input($_POST['bank_name'] ?? '');
            $account_number = sanitize_input($_POST['account_number'] ?? '');

            if (empty($account_name) || empty($bank_name) || empty($account_number)) {
                $message = "Please fill in all fields.";
                $message_type = 'danger';
            } else {
                // Save super admin bank details
                $settings = [
                    'super_admin_account_name' => $account_name,
                    'super_admin_bank_name' => $bank_name,
                    'super_admin_account_number' => $account_number
                ];

                $save_success = true;
                foreach ($settings as $key => $value) {
                    if (!update_setting($link, $key, $value)) {
                        $message = "Error saving $key: " . mysqli_error($link);
                        $message_type = 'danger';
                        error_log("Error saving $key in bank_settings.php: " . mysqli_error($link));
                        $save_success = false;
                        break;
                    }
                }

                if ($save_success) {
                    $message = "Bank details saved successfully.";
                    $message_type = 'success';
                    // Redirect to clear form and load fresh data
                    header("Location: bank_settings.php?saved=1");
                    exit;
                }
            }
            // Regenerate CSRF token
            $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
        } elseif ($_SERVER["REQUEST_METHOD"] == "POST") {
            $message = "Invalid CSRF token.";
            $message_type = 'danger';
            error_log("CSRF token validation failed in bank_settings.php");
            $account_name = sanitize_input($_POST['account_name'] ?? '');
            $bank_name = sanitize_input($_POST['bank_name'] ?? '');
            $account_number = sanitize_input($_POST['account_number'] ?? '');
        }

        // Fetch super admin bank details
        $account_name = get_setting('super_admin_account_name', $link) ?? '';
        $bank_name = get_setting('super_admin_bank_name', $link) ?? '';
        $account_number = get_setting('super_admin_account_number', $link) ?? '';
    }
}

ob_end_clean();
?>

<?php require_once 'includes/header.php'; ?>

<script>
document.title = 'Bank Settings - <?php echo htmlspecialchars(get_setting('system_name', $link) ?? 'Club Portal'); ?>';
</script>

<style>
.main-content {
    padding: 20px;
}
</style>

<div class="container-fluid">
    <div class="row">
        <?php require_once 'includes/sidebar.php'; ?>
        <main class="col-md-9 col-lg-10 main-content">
            <h2>Bank Settings</h2>

            <?php if ($message): ?>
                <div class="alert alert-<?php echo htmlspecialchars($message_type); ?>" role="alert">
                    <?php echo htmlspecialchars($message); ?>
                </div>
            <?php endif; ?>

            <div class="card">
                <div class="card-header">Manage Super Admin Bank Details</div>
                <div class="card-body">
                    <form action="bank_settings.php" method="post">
                        <input type="hidden" name="save_bank_details" value="1">
                        <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($_SESSION['csrf_token']); ?>">
                        <div class="mb-3">
                            <label for="account_name" class="form-label">Account Name</label>
                            <input type="text" name="account_name" id="account_name" class="form-control" value="<?php echo htmlspecialchars($account_name); ?>" required>
                        </div>
                        <div class="mb-3">
                            <label for="bank_name" class="form-label">Bank Name</label>
                            <input type="text" name="bank_name" id="bank_name" class="form-control" value="<?php echo htmlspecialchars($bank_name); ?>" required>
                        </div>
                        <div class="mb-3">
                            <label for="account_number" class="form-label">Account Number</label>
                            <input type="text" name="account_number" id="account_number" class="form-control" value="<?php echo htmlspecialchars($account_number); ?>" required>
                        </div>
                        <button type="submit" class="btn btn-primary">Save Bank Details</button>
                    </form>
                </div>
            </div>
        </main>
    </div>
</div>

<?php require_once 'includes/footer.php'; ?>