<?php
require_once '../includes/functions.php';
require_once '../includes/db_config.php';
session_start();

// Check if user is logged in and is admin
if (!is_logged_in() || !is_admin()) {
    http_response_code(403);
    exit('Access denied');
}

// Check database connection
if (!$link) {
    http_response_code(500);
    exit("ERROR: Could not connect. " . mysqli_connect_error());
}

// Filter settings
$record_type = isset($_GET['record_type']) && in_array($_GET['record_type'], ['all', 'payments']) ? $_GET['record_type'] : 'all';
$time_period = isset($_GET['time_period']) && in_array($_GET['time_period'], ['date', 'week', 'month', 'year', 'all']) ? $_GET['time_period'] : 'all';
$filter_date = isset($_GET['filter_date']) ? $_GET['filter_date'] : date('Y-m-d');

// Build time filter
$time_filter = '';
$bind_params = [];
$bind_types = '';
if ($time_period !== 'all') {
    switch ($time_period) {
        case 'date':
            $time_filter = " AND DATE(p.created_at) = ?";
            $filter_date_param = $filter_date;
            $bind_params[] = &$filter_date_param;
            $bind_types .= 's';
            break;
        case 'week':
            $time_filter = " AND p.created_at >= DATE_SUB(CURDATE(), INTERVAL 1 WEEK)";
            break;
        case 'month':
            $time_filter = " AND p.created_at >= DATE_SUB(CURDATE(), INTERVAL 1 MONTH)";
            break;
        case 'year':
            $time_filter = " AND p.created_at >= DATE_SUB(CURDATE(), INTERVAL 1 YEAR)";
            break;
    }
}

// Fetch payments data
$sql = "SELECT 
            p.id,
            u.member_id,
            IFNULL(u.full_name, 'Unknown') AS full_name,
            p.payment_type,
            p.amount,
            p.payment_status,
            p.created_at,
            p.payment_cycle,
            p.event_date
        FROM payments p
        JOIN users u ON p.user_id = u.id
        WHERE payment_status = 'Paid'" . ($record_type == 'all' || $record_type == 'payments' ? $time_filter : ' AND FALSE') . "
        ORDER BY p.created_at DESC";
$stmt = mysqli_prepare($link, $sql);
if ($time_period == 'date' && !empty($bind_params)) {
    mysqli_stmt_bind_param($stmt, $bind_types, ...$bind_params);
}
mysqli_stmt_execute($stmt);
$result = mysqli_stmt_get_result($stmt);
$payments = [];
while ($row = mysqli_fetch_assoc($result)) {
    $payments[] = $row;
}
mysqli_stmt_close($stmt);

// Set headers for CSV download
header('Content-Type: text/csv; charset=utf-8');
header('Content-Disposition: attachment; filename="payments_export_' . date('Ymd_His') . '.csv"');

// Create CSV file
$output = fopen('php://output', 'w');
fputcsv($output, ['Member ID', 'Member Name', 'Payment Type', 'Amount', 'Status', 'Date', 'Payment Cycle', 'Event Date']);

foreach ($payments as $record) {
    fputcsv($output, [
        $record['member_id'] ?? 'N/A',
        $record['full_name'],
        $record['payment_type'] ?? 'N/A',
        number_format($record['amount'] ?? 0, 2),
        $record['payment_status'] ?? 'N/A',
        date('Y-m-d', strtotime($record['created_at'] ?? 'now')),
        $record['payment_cycle'] ?? 'N/A',
        $record['event_date'] ? date('Y-m-d', strtotime($record['event_date'])) : 'N/A'
    ]);
}

fclose($output);
mysqli_close($link);
exit;
?>